/* ************************************************************************
 * Copyright (C) 2016-2024 Advanced Micro Devices, Inc. All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * ************************************************************************ */

//! HIP = Heterogeneous-compute Interface for Portability
//!
//! Define a extremely thin runtime layer that allows source code to be compiled unmodified
//! through either AMD HCC or NVCC.   Key features tend to be in the spirit
//! and terminology of CUDA, but with a portable path to other accelerators as well.
//!
//!  This is the master include file for hipblas, wrapping around rocblas and cublas "version 2"

#ifndef HIPBLAS_H
#define HIPBLAS_H

#include "hipblas-export.h"
#include "hipblas-version.h"
#include <hip/hip_complex.h>
#include <hip/hip_runtime_api.h>
#include <hip/library_types.h>
#include <stdint.h>

#ifdef __HIP_PLATFORM_NVCC__
#include <cublas_v2.h>
#endif

/* Workaround clang bug:

   This macro expands to static if clang is used; otherwise it expands empty.
   It is intended to be used in variable template specializations, where clang
   requires static in order for the specializations to have internal linkage,
   while technically, storage class specifiers besides thread_local are not
   allowed in template specializations, and static in the primary template
   definition should imply internal linkage for all specializations.

   If clang shows an error for improperly using a storage class specifier in
   a specialization, then HIPBLAS_CLANG_STATIC should be redefined as empty,
   and perhaps removed entirely, if the above bug has been fixed.
*/
#if __clang__
#define HIPBLAS_CLANG_STATIC static
#else
#define HIPBLAS_CLANG_STATIC
#endif

#ifndef HIPBLAS_NO_DEPRECATED_WARNINGS
#ifndef HIPBLAS_DEPRECATED_MSG
#ifndef _MSC_VER
#define HIPBLAS_DEPRECATED_MSG(MSG) //__attribute__((deprecated(#MSG)))
#else
#define HIPBLAS_DEPRECATED_MSG(MSG) //__declspec(deprecated(#MSG))
#endif
#endif
#else
#ifndef HIPBLAS_DEPRECATED_MSG
#define HIPBLAS_DEPRECATED_MSG(MSG)
#endif
#endif

/*
 *  Naming conventions and Notations:
 *
 *  1. Please follow the naming convention
 *      Big case for matrix, e.g. matrix A, B, C   GEMM (C = A*B)
 *      Lower case for vector, e.g. vector x, y    GEMV (y = A*x)
 *
 *  2. Specify the supported precisions for both rocBLAS and cuBLAS backend for any new functions:
 *     For eg:
 *          - Supported precisions in rocBLAS : h,bf,s,d,c,z
 *          - Supported precisions in cuBLAS  : s,d,c,z
 *     Use the following notations,
 *     h  = half
 *     bf = 16 bit brian floating point
 *     s  = single
 *     d  = double
 *     c  = single complex
 *     z  = double complex
 */

/*! \brief hipblasHanlde_t is a void pointer, to store the library context (either rocBLAS or cuBLAS)*/
typedef void* hipblasHandle_t;

/*! \brief To specify the datatype to be unsigned short */

#if __cplusplus < 201103L || !defined(HIPBLAS_USE_HIP_HALF)

typedef uint16_t hipblasHalf;

#else

#include <hip/hip_fp16.h>
typedef __half hipblasHalf;

#endif

/*! \brief  To specify the datatype to be signed char */
typedef int8_t hipblasInt8;

/*! \brief  Stride between matrices or vectors in strided_batched functions.*/
typedef int64_t hipblasStride;

/*! \brief  Struct to represent a 16 bit Brain floating-point number.*/
#if defined(HIPBLAS_USE_HIP_BFLOAT16)

// note that hip_bfloat16 isn't currently supported with cuda backend
#include <hip/hip_bfloat16.h>
typedef hip_bfloat16 hipblasBfloat16;

#elif __cplusplus < 201103L || !defined(HIPBLAS_BFLOAT16_CLASS)

// If this is a C or C++ compiler below C++11, or not requesting HIPBLAS_BFLOAT16_CLASS,
// we only include a minimal definition of hipblasBfloat16
typedef struct hipblasBfloat16
{
    uint16_t data;
} hipblasBfloat16;

#else

class hipblasBfloat16
{
public:
    uint16_t data;

    // zero extend lower 16 bits of bfloat16 to convert to IEEE float
    static float bfloat16_to_float(hipblasBfloat16 val)
    {
        union
        {
            uint32_t int32;
            float    fp32;
        } u = {uint32_t(val.data) << 16};
        return u.fp32;
    }

    static hipblasBfloat16 float_to_bfloat16(float f)
    {
        hipblasBfloat16 rv;
        union
        {
            float    fp32;
            uint32_t int32;
        } u = {f};
        if(~u.int32 & 0x7f800000)
        {
            u.int32 += 0x7fff + ((u.int32 >> 16) & 1); // Round to nearest, round to even
        }
        else if(u.int32 & 0xffff)
        {
            u.int32 |= 0x10000; // Preserve signaling NaN
        }
        rv.data = uint16_t(u.int32 >> 16);
        return rv;
    }

    hipblasBfloat16() = default;

    // round upper 16 bits of IEEE float to convert to bfloat16
    explicit hipblasBfloat16(float f)
        : data(float_to_bfloat16(f))
    {
    }

    // zero extend lower 16 bits of bfloat16 to convert to IEEE float
    operator float() const
    {
        union
        {
            uint32_t int32;
            float    fp32;
        } u = {uint32_t(data) << 16};
        return u.fp32;
    }

    explicit operator bool() const
    {
        return data & 0x7fff;
    }

    inline hipblasBfloat16& operator+=(const hipblasBfloat16& a)
    {
        *this = hipblasBfloat16(float(*this) + float(a));
        return *this;
    }
};

typedef struct
{
    uint16_t data;
} hipblasBfloat16_public;

static_assert(std::is_standard_layout<hipblasBfloat16>{},
              "hipblasBfloat16 is not a standard layout type, and thus is "
              "incompatible with C.");

static_assert(std::is_trivial<hipblasBfloat16>{},
              "hipblasBfloat16 is not a trivial type, and thus is "
              "incompatible with C.");

static_assert(sizeof(hipblasBfloat16) == sizeof(hipblasBfloat16_public)
                  && offsetof(hipblasBfloat16, data) == offsetof(hipblasBfloat16_public, data),
              "internal hipblasBfloat16 does not match public hipblasBfloat16_public");

#endif

#if defined(ROCM_MATHLIBS_API_USE_HIP_COMPLEX)
// Using hip complex types

/*! \brief hip type to represent a complex number with single precision real and imaginary parts. */
HIPBLAS_DEPRECATED_MSG("ROCM_MATHLIBS_API_USE_HIP_COMPLEX define to use HIP complex types is \
deprecated and will be removed in a future release. \
HIP complex datatypes will be used by default in the future.")
typedef hipFloatComplex hipblasComplex;

/*! \brief hip type to represent a complex number with double precision real and imaginary parts. */
typedef hipDoubleComplex hipblasDoubleComplex;

#else
// using internal complex class for API

/*! \brief Struct to represent a complex number with single precision real and imaginary parts.*/
HIPBLAS_DEPRECATED_MSG("hipblasComplex and hipblasDoubleComplex are deprecated for external use \
and will be replaced by hipComplex and hipDoubleComplex \
in the API in the future. Compile with -DHIPBLAS_V2 to get new API with \
the HIP complex types now.")
typedef struct hipblasComplex
{
#ifndef __cplusplus

    float x, y;

#else

private:
    float x, y;

public:
#if __cplusplus >= 201103L
    hipblasComplex() = default;
#else
    hipblasComplex() {}
#endif

    hipblasComplex(float r, float i = 0)
        : x(r)
        , y(i)
    {
    }

    float real() const
    {
        return x;
    }
    float imag() const
    {
        return y;
    }
    void real(float r)
    {
        x = r;
    }
    void imag(float i)
    {
        y = i;
    }

#endif
} hipblasComplex;

/*! \brief Struct to represent a complex number with double precision real and imaginary parts.*/
typedef struct hipblasDoubleComplex
{
#ifndef __cplusplus

    double x, y;

#else

private:
    double x, y;

public:

#if __cplusplus >= 201103L
    hipblasDoubleComplex() = default;
#else
    hipblasDoubleComplex() {}
#endif

    hipblasDoubleComplex(double r, double i = 0)
        : x(r)
        , y(i)
    {
    }
    double real() const
    {
        return x;
    }
    double imag() const
    {
        return y;
    }
    void real(double r)
    {
        x = r;
    }
    void imag(double i)
    {
        y = i;
    }

#endif
} hipblasDoubleComplex;

// this isn't needed right now
// typedef struct hipblasInt8Complex
// {
// #ifndef __cplusplus

//     hipblasInt8 x, y;

// #else

// private:
//     hipblasInt8 x, y;

// public:
// #if __cplusplus >= 201103L
//     hipblasInt8Complex() = default;
// #else
//     hipblasInt8Complex() {}
// #endif

//     hipblasInt8Complex(hipblasInt8 r, hipblasInt8 i = 0)
//         : x(r)
//         , y(i)
//     {
//     }

//     hipblasInt8 real() const
//     {
//         return x;
//     }
//     hipblasInt8 imag() const
//     {
//         return y;
//     }
//     void real(hipblasInt8 r)
//     {
//         x = r;
//     }
//     void imag(hipblasInt8 i)
//     {
//         y = i;
//     }

// #endif
// } hipblasInt8Complex;

#if __cplusplus >= 201103L
static_assert(std::is_trivial<hipblasComplex>{},
              "hipblasComplex is not a trivial type, and thus is incompatible with C.");
static_assert(std::is_trivial<hipblasDoubleComplex>{},
              "hipblasDoubleComplex is not a trivial type, and thus is incompatible with C.");
#endif

#endif // using internal complex class for API

#if __cplusplus >= 201103L
#include <type_traits>
static_assert(std::is_standard_layout<hipblasComplex>{},
              "hipblasComplex is not a standard layout type, and thus is incompatible with C.");
static_assert(
    std::is_standard_layout<hipblasDoubleComplex>{},
    "hipblasDoubleComplex is not a standard layout type, and thus is incompatible with C.");
static_assert(sizeof(hipblasComplex) == sizeof(float) * 2
                  && sizeof(hipblasDoubleComplex) == sizeof(double) * 2
                  && sizeof(hipblasDoubleComplex) == sizeof(hipblasComplex) * 2,
              "Sizes of hipblasComplex or hipblasDoubleComplex are inconsistent");
#endif

/*! \brief hipblas status codes definition */
typedef enum
{
    HIPBLAS_STATUS_SUCCESS           = 0, /**< Function succeeds */
    HIPBLAS_STATUS_NOT_INITIALIZED   = 1, /**< HIPBLAS library not initialized */
    HIPBLAS_STATUS_ALLOC_FAILED      = 2, /**< resource allocation failed */
    HIPBLAS_STATUS_INVALID_VALUE     = 3, /**< unsupported numerical value was passed to function */
    HIPBLAS_STATUS_MAPPING_ERROR     = 4, /**< access to GPU memory space failed */
    HIPBLAS_STATUS_EXECUTION_FAILED  = 5, /**< GPU program failed to execute */
    HIPBLAS_STATUS_INTERNAL_ERROR    = 6, /**< an internal HIPBLAS operation failed */
    HIPBLAS_STATUS_NOT_SUPPORTED     = 7, /**< function not implemented */
    HIPBLAS_STATUS_ARCH_MISMATCH     = 8, /**< architecture mismatch */
    HIPBLAS_STATUS_HANDLE_IS_NULLPTR = 9, /**< hipBLAS handle is null pointer */
    HIPBLAS_STATUS_INVALID_ENUM      = 10, /**<  unsupported enum value was passed to function */
    HIPBLAS_STATUS_UNKNOWN           = 11, /**<  back-end returned an unsupported status code */
} hipblasStatus_t;

/*! \brief Indicates if scalar pointers are on host or device. This is used for scalars alpha and beta and for scalar function return values. */
typedef enum
{
    HIPBLAS_POINTER_MODE_HOST, /**< Scalar values affected by this variable will be located on the host. */
    HIPBLAS_POINTER_MODE_DEVICE /**<  Scalar values affected by this variable will be located on the device. */
} hipblasPointerMode_t;

// set the values of enum constants to be the same as those used in cblas

#ifndef HIPBLAS_OPERATION_DECLARED
#define HIPBLAS_OPERATION_DECLARED
/*! \brief Used to specify whether the matrix is to be transposed or not. */
typedef enum
{
    HIPBLAS_OP_N = 111, /**<  Operate with the matrix. */
    HIPBLAS_OP_T = 112, /**<  Operate with the transpose of the matrix. */
    HIPBLAS_OP_C = 113 /**< Operate with the conjugate transpose of the matrix. */
} hipblasOperation_t;

#elif __cplusplus >= 201103L
static_assert(HIPBLAS_OP_N == 111, "Inconsistent declaration of HIPBLAS_OP_N");
static_assert(HIPBLAS_OP_T == 112, "Inconsistent declaration of HIPBLAS_OP_T");
static_assert(HIPBLAS_OP_C == 113, "Inconsistent declaration of HIPBLAS_OP_C");
#endif // HIPBLAS_OPERATION_DECLARED

#ifndef HIPBLAS_FILL_MODE_DECLARED
#define HIPBLAS_FILL_MODE_DECLARED
/*! \brief Used by the Hermitian, symmetric and triangular matrix routines to specify whether the upper or lower triangle is being referenced. */
typedef enum
{
    HIPBLAS_FILL_MODE_UPPER = 121, /**<  Upper triangle */
    HIPBLAS_FILL_MODE_LOWER = 122, /**<  Lower triangle */
    HIPBLAS_FILL_MODE_FULL  = 123
} hipblasFillMode_t;

#elif __cplusplus >= 201103L
static_assert(HIPBLAS_FILL_MODE_UPPER == 121,
              "Inconsistent declaration of HIPBLAS_FILL_MODE_UPPER");
static_assert(HIPBLAS_FILL_MODE_LOWER == 122,
              "Inconsistent declaration of HIPBLAS_FILL_MODE_LOWER");
static_assert(HIPBLAS_FILL_MODE_FULL == 123, "Inconsistent declaration of HIPBLAS_FILL_MODE_FULL");
#endif // HIPBLAS_FILL_MODE_DECLARED

#ifndef HIPBLAS_DIAG_TYPE_DECLARED
#define HIPBLAS_DIAG_TYPE_DECLARED
/*! \brief It is used by the triangular matrix routines to specify whether the matrix is unit triangular.*/
typedef enum
{
    HIPBLAS_DIAG_NON_UNIT = 131, /**<  Non-unit triangular. */
    HIPBLAS_DIAG_UNIT     = 132 /**<  Unit triangular. */
} hipblasDiagType_t;

#elif __cplusplus >= 201103L
static_assert(HIPBLAS_DIAG_NON_UNIT == 131, "Inconsistent declaration of HIPBLAS_DIAG_NON_UNIT");
static_assert(HIPBLAS_DIAG_UNIT == 132, "Inconsistent declaration of HIPBLAS_DIAG_UNIT");
#endif // HIPBLAS_DIAG_TYPE_DECLARED

#ifndef HIPBLAS_SIDE_MODE_DECLARED
#define HIPBLAS_SIDE_MODE_DECLARED
/*! \brief Indicates the side matrix A is located relative to matrix B during multiplication. */
typedef enum
{
    HIPBLAS_SIDE_LEFT
    = 141, /**< Multiply general matrix by symmetric, Hermitian or triangular matrix on the left. */
    HIPBLAS_SIDE_RIGHT
    = 142, /**< Multiply general matrix by symmetric, Hermitian or triangular matrix on the right. */
    HIPBLAS_SIDE_BOTH = 143
} hipblasSideMode_t;
#elif __cplusplus >= 201103L
static_assert(HIPBLAS_SIDE_LEFT == 141, "Inconsistent declaration of HIPBLAS_SIDE_LEFT");
static_assert(HIPBLAS_SIDE_RIGHT == 142, "Inconsistent declaration of HIPBLAS_SIDE_RIGHT");
static_assert(HIPBLAS_SIDE_BOTH == 143, "Inconsistent declaration of HIPBLAS_SIDE_BOTH");
#endif // HIPBLAS_SIDE_MODE_DECLARED

typedef enum
{
    HIPBLAS_DEFAULT_MATH,
    HIPBLAS_XF32_XDL_MATH, /* equivalent to rocblas_xf32_xdl_math_op, not supported in cuBLAS */
    HIPBLAS_PEDANTIC_MATH, /* equivalent to CUBLAS_PEDANTIC_MATH, not yet supported in rocBLAS */
    HIPBLAS_TF32_TENSOR_OP_MATH, /* use TF32 tensor cores with cuBLAS backend, not supported in rocBLAS */
    HIPBLAS_MATH_DISALLOW_REDUCED_PRECISION_REDUCTION, /* see cuBLAS documentation, not supported in rocBLAS */
    HIPBLAS_TENSOR_OP_MATH /* DEPRECATED, use Tensor Core operations with cuBLAS backend */
} hipblasMath_t;

#ifdef HIPBLAS_V2

// Replacing use of hipblasDatatype_t with hipDataType which will be used in a future release.
typedef hipDataType hipblasDatatype_t;

#define HIPBLAS_R_16F HIP_R_16F
#define HIPBLAS_R_32F HIP_R_32F
#define HIPBLAS_R_64F HIP_R_64F
#define HIPBLAS_C_16F HIP_C_16F
#define HIPBLAS_C_32F HIP_C_32F
#define HIPBLAS_C_64F HIP_C_64F
#define HIPBLAS_R_8I HIP_R_8I
#define HIPBLAS_R_8U HIP_R_8U
#define HIPBLAS_R_32I HIP_R_32I
#define HIPBLAS_R_32U HIP_R_32U
#define HIPBLAS_C_8I HIP_C_8I
#define HIPBLAS_C_8U HIP_C_8U
#define HIPBLAS_C_32I HIP_C_32I
#define HIPBLAS_C_32U HIP_C_32U
#define HIPBLAS_R_16B HIP_R_16BF
#define HIPBLAS_C_16B HIP_C_16BF
#define HIPBLAS_DATATYPE_INVALID hipDataType(31) // Temporary until hipblasDatatype_t is gone.

#else

// clang-format off
HIPBLAS_DEPRECATED_MSG("hipblasDatatype_t is deprecated and will be replaced by hipDataType in the future. Compile with -DHIPBLAS_V2 to get new API with hipDataType now.")
// clang-format on

/*! \brief Indicates the precision of data used. hipblasDatatype_t is deprecated as of hipBLAS 2.0.0 and will
           be removed in a future release as generally replaced by hipDataType. */
typedef enum
{
    HIPBLAS_R_16F            = 150, /**< 16 bit floating point, real */
    HIPBLAS_R_32F            = 151, /**< 32 bit floating point, real */
    HIPBLAS_R_64F            = 152, /**< 64 bit floating point, real */
    HIPBLAS_C_16F            = 153, /**< 16 bit floating point, complex */
    HIPBLAS_C_32F            = 154, /**< 32 bit floating point, complex */
    HIPBLAS_C_64F            = 155, /**< 64 bit floating point, complex */
    HIPBLAS_R_8I             = 160, /**<  8 bit signed integer, real */
    HIPBLAS_R_8U             = 161, /**<  8 bit unsigned integer, real */
    HIPBLAS_R_32I            = 162, /**< 32 bit signed integer, real */
    HIPBLAS_R_32U            = 163, /**< 32 bit unsigned integer, real */
    HIPBLAS_C_8I             = 164, /**<  8 bit signed integer, complex */
    HIPBLAS_C_8U             = 165, /**<  8 bit unsigned integer, complex */
    HIPBLAS_C_32I            = 166, /**< 32 bit signed integer, complex */
    HIPBLAS_C_32U            = 167, /**< 32 bit unsigned integer, complex */
    HIPBLAS_R_16B            = 168, /**< 16 bit bfloat, real */
    HIPBLAS_C_16B            = 169, /**< 16 bit bfloat, complex */
    HIPBLAS_DATATYPE_INVALID = 255, /**< Invalid datatype value, do not use */
} hipblasDatatype_t;

#endif

/*! \brief The compute type to be used. Currently only used with GemmEx with the HIPBLAS_V2 interface.
 *         Note that support for compute types is largely dependent on backend. */
typedef enum
{
    // Note that these types are taken from cuBLAS. With the rocBLAS backend, currently hipBLAS will
    // convert to rocBLAS types to get equivalent functionality where supported.
    HIPBLAS_COMPUTE_16F           = 0, /**< compute will be at least 16-bit precision */
    HIPBLAS_COMPUTE_16F_PEDANTIC  = 1, /**< compute will be exactly 16-bit precision */
    HIPBLAS_COMPUTE_32F           = 2, /**< compute will be at least 32-bit precision */
    HIPBLAS_COMPUTE_32F_PEDANTIC  = 3, /**< compute will be exactly 32-bit precision */
    HIPBLAS_COMPUTE_32F_FAST_16F  = 4, /**< 32-bit input can use 16-bit compute */
    HIPBLAS_COMPUTE_32F_FAST_16BF = 5, /**< 32-bit input can is bf16 compute */
    HIPBLAS_COMPUTE_32F_FAST_TF32
    = 6, /**< 32-bit input can use tensor cores w/ TF32 compute. Only supported with cuBLAS backend currently */
    HIPBLAS_COMPUTE_64F          = 7, /**< compute will be at least 64-bit precision */
    HIPBLAS_COMPUTE_64F_PEDANTIC = 8, /**< compute will be exactly 64-bit precision */
    HIPBLAS_COMPUTE_32I          = 9, /**< compute will be at least 32-bit integer precision */
    HIPBLAS_COMPUTE_32I_PEDANTIC = 10, /**< compute will be exactly 32-bit integer precision */
} hipblasComputeType_t;

/*! \brief Indicates if layer is active with bitmask. */
typedef enum
{
    HIPBLAS_GEMM_DEFAULT = 160 /**<  enumerator rocblas_gemm_algo_standard */
} hipblasGemmAlgo_t;

/*! \brief Indicates if atomics operations are allowed. Not allowing atomic operations may generally improve determinism and repeatability of results at a cost of performance.
 *         By default, the rocBLAS backend will allow atomic operations while the cuBLAS backend will disallow atomic operations. See backend documentation
 *         for more detail. */
typedef enum
{
    HIPBLAS_ATOMICS_NOT_ALLOWED = 0, /**<  Algorithms will refrain from atomics where applicable. */
    HIPBLAS_ATOMICS_ALLOWED = 1 /**< Algorithms will take advantage of atomics where applicable. */
} hipblasAtomicsMode_t;

/*! \brief Control flags passed into gemm ex with flags algorithms. Only relevant with rocBLAS backend. See rocBLAS documentation
 *         for more information.*/
typedef enum
{
    HIPBLAS_GEMM_FLAGS_NONE = 0x0, /**< Default empty flags */
    HIPBLAS_GEMM_FLAGS_USE_CU_EFFICIENCY
    = 0x2, /**< enumerator rocblas_gemm_flags_use_cu_efficiency */
    HIPBLAS_GEMM_FLAGS_FP16_ALT_IMPL = 0x4, /**< enumerator rocblas_gemm_flags_fp16_alt_impl */
    HIPBLAS_GEMM_FLAGS_CHECK_SOLUTION_INDEX
    = 0x8, /**< enumerator rocblas_gemm_flags_check_solution_index */
    HIPBLAS_GEMM_FLAGS_FP16_ALT_IMPL_RNZ
    = 0x10 /**< enumerator rocblas_gemm_flags_fp16_alt_impl_rnz */
} hipblasGemmFlags_t;

#ifdef __cplusplus
extern "C" {
#endif

/*! \brief Create hipblas handle. */
HIPBLAS_EXPORT hipblasStatus_t hipblasCreate(hipblasHandle_t* handle);

/*! \brief Destroys the library context created using hipblasCreate() */
HIPBLAS_EXPORT hipblasStatus_t hipblasDestroy(hipblasHandle_t handle);

/*! \brief Set stream for handle */
HIPBLAS_EXPORT hipblasStatus_t hipblasSetStream(hipblasHandle_t handle, hipStream_t streamId);

/*! \brief Get stream[0] for handle */
HIPBLAS_EXPORT hipblasStatus_t hipblasGetStream(hipblasHandle_t handle, hipStream_t* streamId);

/*! \brief Set hipblas pointer mode */
HIPBLAS_EXPORT hipblasStatus_t hipblasSetPointerMode(hipblasHandle_t      handle,
                                                     hipblasPointerMode_t mode);
/*! \brief Get hipblas pointer mode */
HIPBLAS_EXPORT hipblasStatus_t hipblasGetPointerMode(hipblasHandle_t       handle,
                                                     hipblasPointerMode_t* mode);

/*! \brief Set hipblas math mode */
HIPBLAS_EXPORT hipblasStatus_t hipblasSetMathMode(hipblasHandle_t handle, hipblasMath_t mode);

/*! \brief Get hipblas math mode */
HIPBLAS_EXPORT hipblasStatus_t hipblasGetMathMode(hipblasHandle_t handle, hipblasMath_t* mode);

/*! \brief copy vector from host to device
    @param[in]
    n           [int]
                number of elements in the vector
    @param[in]
    elemSize    [int]
                Size of both vectors in bytes
    @param[in]
    x           pointer to vector on the host
    @param[in]
    incx        [int]
                specifies the increment for the elements of the vector
    @param[out]
    y           pointer to vector on the device
    @param[in]
    incy        [int]
                specifies the increment for the elements of the vector
     ********************************************************************/
HIPBLAS_EXPORT hipblasStatus_t
    hipblasSetVector(int n, int elemSize, const void* x, int incx, void* y, int incy);

/*! \brief copy vector from device to host
    @param[in]
    n           [int]
                number of elements in the vector
    @param[in]
    elemSize    [int]
                Size of both vectors in bytes
    @param[in]
    x           pointer to vector on the device
    @param[in]
    incx        [int]
                specifies the increment for the elements of the vector
    @param[out]
    y           pointer to vector on the host
    @param[in]
    incy        [int]
                specifies the increment for the elements of the vector
     ********************************************************************/
HIPBLAS_EXPORT hipblasStatus_t
    hipblasGetVector(int n, int elemSize, const void* x, int incx, void* y, int incy);

/*! \brief copy matrix from host to device
    @param[in]
    rows        [int]
                number of rows in matrices
    @param[in]
    cols        [int]
                number of columns in matrices
    @param[in]
    elemSize   [int]
                number of bytes per element in the matrix
    @param[in]
    AP          pointer to matrix on the host
    @param[in]
    lda         [int]
                specifies the leading dimension of A, lda >= rows
    @param[out]
    BP           pointer to matrix on the GPU
    @param[in]
    ldb         [int]
                specifies the leading dimension of B, ldb >= rows
     ********************************************************************/
HIPBLAS_EXPORT hipblasStatus_t
    hipblasSetMatrix(int rows, int cols, int elemSize, const void* AP, int lda, void* BP, int ldb);

/*! \brief copy matrix from device to host
    @param[in]
    rows        [int]
                number of rows in matrices
    @param[in]
    cols        [int]
                number of columns in matrices
    @param[in]
    elemSize   [int]
                number of bytes per element in the matrix
    @param[in]
    AP          pointer to matrix on the GPU
    @param[in]
    lda         [int]
                specifies the leading dimension of A, lda >= rows
    @param[out]
    BP          pointer to matrix on the host
    @param[in]
    ldb         [int]
                specifies the leading dimension of B, ldb >= rows
     ********************************************************************/
HIPBLAS_EXPORT hipblasStatus_t
    hipblasGetMatrix(int rows, int cols, int elemSize, const void* AP, int lda, void* BP, int ldb);

/*! \brief asynchronously copy vector from host to device
    \details
    hipblasSetVectorAsync copies a vector from pinned host memory to device memory asynchronously.
    Memory on the host must be allocated with hipHostMalloc or the transfer will be synchronous.
    @param[in]
    n           [int]
                number of elements in the vector
    @param[in]
    elemSize   [int]
                number of bytes per element in the matrix
    @param[in]
    x           pointer to vector on the host
    @param[in]
    incx        [int]
                specifies the increment for the elements of the vector
    @param[out]
    y           pointer to vector on the device
    @param[in]
    incy        [int]
                specifies the increment for the elements of the vector
    @param[in]
    stream      specifies the stream into which this transfer request is queued
     ********************************************************************/
HIPBLAS_EXPORT hipblasStatus_t hipblasSetVectorAsync(
    int n, int elemSize, const void* x, int incx, void* y, int incy, hipStream_t stream);

/*! \brief asynchronously copy vector from device to host
    \details
    hipblasGetVectorAsync copies a vector from pinned host memory to device memory asynchronously.
    Memory on the host must be allocated with hipHostMalloc or the transfer will be synchronous.
    @param[in]
    n           [int]
                number of elements in the vector
    @param[in]
    elemSize   [int]
                number of bytes per element in the matrix
    @param[in]
    x           pointer to vector on the device
    @param[in]
    incx        [int]
                specifies the increment for the elements of the vector
    @param[out]
    y           pointer to vector on the host
    @param[in]
    incy        [int]
                specifies the increment for the elements of the vector
    @param[in]
    stream      specifies the stream into which this transfer request is queued
     ********************************************************************/
HIPBLAS_EXPORT hipblasStatus_t hipblasGetVectorAsync(
    int n, int elemSize, const void* x, int incx, void* y, int incy, hipStream_t stream);

/*! \brief asynchronously copy matrix from host to device
    \details
    hipblasSetMatrixAsync copies a matrix from pinned host memory to device memory asynchronously.
    Memory on the host must be allocated with hipHostMalloc or the transfer will be synchronous.
    @param[in]
    rows        [int]
                number of rows in matrices
    @param[in]
    cols        [int]
                number of columns in matrices
    @param[in]
    elemSize   [int]
                number of bytes per element in the matrix
    @param[in]
    AP           pointer to matrix on the host
    @param[in]
    lda         [int]
                specifies the leading dimension of A, lda >= rows
    @param[out]
    BP           pointer to matrix on the GPU
    @param[in]
    ldb         [int]
                specifies the leading dimension of B, ldb >= rows
    @param[in]
    stream      specifies the stream into which this transfer request is queued
     ********************************************************************/
HIPBLAS_EXPORT hipblasStatus_t hipblasSetMatrixAsync(int         rows,
                                                     int         cols,
                                                     int         elemSize,
                                                     const void* AP,
                                                     int         lda,
                                                     void*       BP,
                                                     int         ldb,
                                                     hipStream_t stream);
/*! \brief asynchronously copy matrix from device to host
    \details
    hipblasGetMatrixAsync copies a matrix from device memory to pinned host memory asynchronously.
    Memory on the host must be allocated with hipHostMalloc or the transfer will be synchronous.
    @param[in]
    rows        [int]
                number of rows in matrices
    @param[in]
    cols        [int]
                number of columns in matrices
    @param[in]
    elemSize   [int]
                number of bytes per element in the matrix
    @param[in]
    AP          pointer to matrix on the GPU
    @param[in]
    lda         [int]
                specifies the leading dimension of A, lda >= rows
    @param[out]
    BP           pointer to matrix on the host
    @param[in]
    ldb         [int]
                specifies the leading dimension of B, ldb >= rows
    @param[in]
    stream      specifies the stream into which this transfer request is queued
     ********************************************************************/
HIPBLAS_EXPORT hipblasStatus_t hipblasGetMatrixAsync(int         rows,
                                                     int         cols,
                                                     int         elemSize,
                                                     const void* AP,
                                                     int         lda,
                                                     void*       BP,
                                                     int         ldb,
                                                     hipStream_t stream);

/*! \brief Set hipblasSetAtomicsMode*/
HIPBLAS_EXPORT hipblasStatus_t hipblasSetAtomicsMode(hipblasHandle_t      handle,
                                                     hipblasAtomicsMode_t atomics_mode);

/*! \brief Get hipblasSetAtomicsMode*/
HIPBLAS_EXPORT hipblasStatus_t hipblasGetAtomicsMode(hipblasHandle_t       handle,
                                                     hipblasAtomicsMode_t* atomics_mode);

/*
 * ===========================================================================
 *    level 1 BLAS
 * ===========================================================================
 */

/*! @{
    \brief BLAS Level 1 API

    \details
    amax finds the first index of the element of maximum magnitude of a vector x.

    - Supported precisions in rocBLAS : s,d,c,z.
    - Supported precisions in cuBLAS  : s,d,c,z.

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    n         [int]
              the number of elements in x.
    @param[in]
    x         device pointer storing vector x.
    @param[in]
    incx      [int]
              specifies the increment for the elements of y.
    @param[inout]
    result
              device pointer or host pointer to store the amax index.
              return is 0.0 if n, incx<=0.

    ********************************************************************/
HIPBLAS_EXPORT hipblasStatus_t
    hipblasIsamax(hipblasHandle_t handle, int n, const float* x, int incx, int* result);

HIPBLAS_EXPORT hipblasStatus_t
    hipblasIdamax(hipblasHandle_t handle, int n, const double* x, int incx, int* result);

HIPBLAS_EXPORT hipblasStatus_t
    hipblasIcamax(hipblasHandle_t handle, int n, const hipblasComplex* x, int incx, int* result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIzamax(
    hipblasHandle_t handle, int n, const hipblasDoubleComplex* x, int incx, int* result);

HIPBLAS_EXPORT hipblasStatus_t
    hipblasIcamax_v2(hipblasHandle_t handle, int n, const hipComplex* x, int incx, int* result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIzamax_v2(
    hipblasHandle_t handle, int n, const hipDoubleComplex* x, int incx, int* result);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasIsamax_64(
    hipblasHandle_t handle, int64_t n, const float* x, int64_t incx, int64_t* result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIdamax_64(
    hipblasHandle_t handle, int64_t n, const double* x, int64_t incx, int64_t* result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIcamax_64(
    hipblasHandle_t handle, int64_t n, const hipblasComplex* x, int64_t incx, int64_t* result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIzamax_64(hipblasHandle_t             handle,
                                                int64_t                     n,
                                                const hipblasDoubleComplex* x,
                                                int64_t                     incx,
                                                int64_t*                    result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIcamax_v2_64(
    hipblasHandle_t handle, int64_t n, const hipComplex* x, int64_t incx, int64_t* result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIzamax_v2_64(
    hipblasHandle_t handle, int64_t n, const hipDoubleComplex* x, int64_t incx, int64_t* result);
//! @}

/*! @{
    \brief BLAS Level 1 API

    \details
     amaxBatched finds the first index of the element of maximum magnitude of each vector x_i in a batch, for i = 1, ..., batchCount.

    - Supported precisions in rocBLAS : s,d,c,z.
    - Supported precisions in cuBLAS  : No support.

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    n         [int]
              number of elements in each vector x_i
    @param[in]
    x         device array of device pointers storing each vector x_i.
    @param[in]
    incx      [int]
              specifies the increment for the elements of each x_i. incx must be > 0.
    @param[in]
    batchCount [int]
              number of instances in the batch, must be > 0.
    @param[out]
    result
              device or host array of pointers of batchCount size for results.
              return is 0 if n, incx<=0.

    ********************************************************************/
HIPBLAS_EXPORT hipblasStatus_t hipblasIsamaxBatched(
    hipblasHandle_t handle, int n, const float* const x[], int incx, int batchCount, int* result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIdamaxBatched(
    hipblasHandle_t handle, int n, const double* const x[], int incx, int batchCount, int* result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIcamaxBatched(hipblasHandle_t             handle,
                                                    int                         n,
                                                    const hipblasComplex* const x[],
                                                    int                         incx,
                                                    int                         batchCount,
                                                    int*                        result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIzamaxBatched(hipblasHandle_t                   handle,
                                                    int                               n,
                                                    const hipblasDoubleComplex* const x[],
                                                    int                               incx,
                                                    int                               batchCount,
                                                    int*                              result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIcamaxBatched_v2(hipblasHandle_t         handle,
                                                       int                     n,
                                                       const hipComplex* const x[],
                                                       int                     incx,
                                                       int                     batchCount,
                                                       int*                    result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIzamaxBatched_v2(hipblasHandle_t               handle,
                                                       int                           n,
                                                       const hipDoubleComplex* const x[],
                                                       int                           incx,
                                                       int                           batchCount,
                                                       int*                          result);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasIsamaxBatched_64(hipblasHandle_t    handle,
                                                       int64_t            n,
                                                       const float* const x[],
                                                       int64_t            incx,
                                                       int64_t            batchCount,
                                                       int64_t*           result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIdamaxBatched_64(hipblasHandle_t     handle,
                                                       int64_t             n,
                                                       const double* const x[],
                                                       int64_t             incx,
                                                       int64_t             batchCount,
                                                       int64_t*            result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIcamaxBatched_64(hipblasHandle_t             handle,
                                                       int64_t                     n,
                                                       const hipblasComplex* const x[],
                                                       int64_t                     incx,
                                                       int64_t                     batchCount,
                                                       int64_t*                    result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIzamaxBatched_64(hipblasHandle_t                   handle,
                                                       int64_t                           n,
                                                       const hipblasDoubleComplex* const x[],
                                                       int64_t                           incx,
                                                       int64_t                           batchCount,
                                                       int64_t*                          result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIcamaxBatched_v2_64(hipblasHandle_t         handle,
                                                          int64_t                 n,
                                                          const hipComplex* const x[],
                                                          int64_t                 incx,
                                                          int64_t                 batchCount,
                                                          int64_t*                result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIzamaxBatched_v2_64(hipblasHandle_t               handle,
                                                          int64_t                       n,
                                                          const hipDoubleComplex* const x[],
                                                          int64_t                       incx,
                                                          int64_t                       batchCount,
                                                          int64_t*                      result);
//! @}

/*! @{
    \brief BLAS Level 1 API

    \details
     amaxStridedBatched finds the first index of the element of maximum magnitude of each vector x_i in a batch, for i = 1, ..., batchCount.

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    n         [int]
              number of elements in each vector x_i
    @param[in]
    x         device pointer to the first vector x_1.
    @param[in]
    incx      [int]
              specifies the increment for the elements of each x_i. incx must be > 0.
    @param[in]
    stridex   [hipblasStride]
              specifies the pointer increment between one x_i and the next x_(i + 1).
    @param[in]
    batchCount [int]
              number of instances in the batch
    @param[out]
    result
              device or host pointer for storing contiguous batchCount results.
              return is 0 if n <= 0, incx<=0.

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasIsamaxStridedBatched(hipblasHandle_t handle,
                                                           int             n,
                                                           const float*    x,
                                                           int             incx,
                                                           hipblasStride   stridex,
                                                           int             batchCount,
                                                           int*            result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIdamaxStridedBatched(hipblasHandle_t handle,
                                                           int             n,
                                                           const double*   x,
                                                           int             incx,
                                                           hipblasStride   stridex,
                                                           int             batchCount,
                                                           int*            result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIcamaxStridedBatched(hipblasHandle_t       handle,
                                                           int                   n,
                                                           const hipblasComplex* x,
                                                           int                   incx,
                                                           hipblasStride         stridex,
                                                           int                   batchCount,
                                                           int*                  result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIzamaxStridedBatched(hipblasHandle_t             handle,
                                                           int                         n,
                                                           const hipblasDoubleComplex* x,
                                                           int                         incx,
                                                           hipblasStride               stridex,
                                                           int                         batchCount,
                                                           int*                        result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIcamaxStridedBatched_v2(hipblasHandle_t   handle,
                                                              int               n,
                                                              const hipComplex* x,
                                                              int               incx,
                                                              hipblasStride     stridex,
                                                              int               batchCount,
                                                              int*              result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIzamaxStridedBatched_v2(hipblasHandle_t         handle,
                                                              int                     n,
                                                              const hipDoubleComplex* x,
                                                              int                     incx,
                                                              hipblasStride           stridex,
                                                              int                     batchCount,
                                                              int*                    result);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasIsamaxStridedBatched_64(hipblasHandle_t handle,
                                                              int64_t         n,
                                                              const float*    x,
                                                              int64_t         incx,
                                                              hipblasStride   stridex,
                                                              int64_t         batchCount,
                                                              int64_t*        result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIdamaxStridedBatched_64(hipblasHandle_t handle,
                                                              int64_t         n,
                                                              const double*   x,
                                                              int64_t         incx,
                                                              hipblasStride   stridex,
                                                              int64_t         batchCount,
                                                              int64_t*        result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIcamaxStridedBatched_64(hipblasHandle_t       handle,
                                                              int64_t               n,
                                                              const hipblasComplex* x,
                                                              int64_t               incx,
                                                              hipblasStride         stridex,
                                                              int64_t               batchCount,
                                                              int64_t*              result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIzamaxStridedBatched_64(hipblasHandle_t             handle,
                                                              int64_t                     n,
                                                              const hipblasDoubleComplex* x,
                                                              int64_t                     incx,
                                                              hipblasStride               stridex,
                                                              int64_t  batchCount,
                                                              int64_t* result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIcamaxStridedBatched_v2_64(hipblasHandle_t   handle,
                                                                 int64_t           n,
                                                                 const hipComplex* x,
                                                                 int64_t           incx,
                                                                 hipblasStride     stridex,
                                                                 int64_t           batchCount,
                                                                 int64_t*          result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIzamaxStridedBatched_v2_64(hipblasHandle_t         handle,
                                                                 int64_t                 n,
                                                                 const hipDoubleComplex* x,
                                                                 int64_t                 incx,
                                                                 hipblasStride           stridex,
                                                                 int64_t                 batchCount,
                                                                 int64_t*                result);
//! @}

/*! @{
    \brief BLAS Level 1 API

    \details
    amin finds the first index of the element of minimum magnitude of a vector x.

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : s,d,c,z

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    n         [int]
              the number of elements in x.
    @param[in]
    x         device pointer storing vector x.
    @param[in]
    incx      [int]
              specifies the increment for the elements of y.
    @param[inout]
    result
              device pointer or host pointer to store the amin index.
              return is 0.0 if n, incx<=0.

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t
    hipblasIsamin(hipblasHandle_t handle, int n, const float* x, int incx, int* result);

HIPBLAS_EXPORT hipblasStatus_t
    hipblasIdamin(hipblasHandle_t handle, int n, const double* x, int incx, int* result);

HIPBLAS_EXPORT hipblasStatus_t
    hipblasIcamin(hipblasHandle_t handle, int n, const hipblasComplex* x, int incx, int* result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIzamin(
    hipblasHandle_t handle, int n, const hipblasDoubleComplex* x, int incx, int* result);

HIPBLAS_EXPORT hipblasStatus_t
    hipblasIcamin_v2(hipblasHandle_t handle, int n, const hipComplex* x, int incx, int* result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIzamin_v2(
    hipblasHandle_t handle, int n, const hipDoubleComplex* x, int incx, int* result);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasIsamin_64(
    hipblasHandle_t handle, int64_t n, const float* x, int64_t incx, int64_t* result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIdamin_64(
    hipblasHandle_t handle, int64_t n, const double* x, int64_t incx, int64_t* result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIcamin_64(
    hipblasHandle_t handle, int64_t n, const hipblasComplex* x, int64_t incx, int64_t* result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIzamin_64(hipblasHandle_t             handle,
                                                int64_t                     n,
                                                const hipblasDoubleComplex* x,
                                                int64_t                     incx,
                                                int64_t*                    result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIcamin_v2_64(
    hipblasHandle_t handle, int64_t n, const hipComplex* x, int64_t incx, int64_t* result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIzamin_v2_64(
    hipblasHandle_t handle, int64_t n, const hipDoubleComplex* x, int64_t incx, int64_t* result);
//! @}

/*! @{
    \brief BLAS Level 1 API

    \details
    aminBatched finds the first index of the element of minimum magnitude of each vector x_i in a batch, for i = 1, ..., batchCount.

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    n         [int]
              number of elements in each vector x_i
    @param[in]
    x         device array of device pointers storing each vector x_i.
    @param[in]
    incx      [int]
              specifies the increment for the elements of each x_i. incx must be > 0.
    @param[in]
    batchCount [int]
              number of instances in the batch, must be > 0.
    @param[out]
    result
              device or host pointers to array of batchCount size for results.
              return is 0 if n, incx<=0.

    ********************************************************************/
HIPBLAS_EXPORT hipblasStatus_t hipblasIsaminBatched(
    hipblasHandle_t handle, int n, const float* const x[], int incx, int batchCount, int* result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIdaminBatched(
    hipblasHandle_t handle, int n, const double* const x[], int incx, int batchCount, int* result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIcaminBatched(hipblasHandle_t             handle,
                                                    int                         n,
                                                    const hipblasComplex* const x[],
                                                    int                         incx,
                                                    int                         batchCount,
                                                    int*                        result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIzaminBatched(hipblasHandle_t                   handle,
                                                    int                               n,
                                                    const hipblasDoubleComplex* const x[],
                                                    int                               incx,
                                                    int                               batchCount,
                                                    int*                              result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIcaminBatched_v2(hipblasHandle_t         handle,
                                                       int                     n,
                                                       const hipComplex* const x[],
                                                       int                     incx,
                                                       int                     batchCount,
                                                       int*                    result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIzaminBatched_v2(hipblasHandle_t               handle,
                                                       int                           n,
                                                       const hipDoubleComplex* const x[],
                                                       int                           incx,
                                                       int                           batchCount,
                                                       int*                          result);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasIsaminBatched_64(hipblasHandle_t    handle,
                                                       int64_t            n,
                                                       const float* const x[],
                                                       int64_t            incx,
                                                       int64_t            batchCount,
                                                       int64_t*           result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIdaminBatched_64(hipblasHandle_t     handle,
                                                       int64_t             n,
                                                       const double* const x[],
                                                       int64_t             incx,
                                                       int64_t             batchCount,
                                                       int64_t*            result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIcaminBatched_64(hipblasHandle_t             handle,
                                                       int64_t                     n,
                                                       const hipblasComplex* const x[],
                                                       int64_t                     incx,
                                                       int64_t                     batchCount,
                                                       int64_t*                    result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIzaminBatched_64(hipblasHandle_t                   handle,
                                                       int64_t                           n,
                                                       const hipblasDoubleComplex* const x[],
                                                       int64_t                           incx,
                                                       int64_t                           batchCount,
                                                       int64_t*                          result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIcaminBatched_v2_64(hipblasHandle_t         handle,
                                                          int64_t                 n,
                                                          const hipComplex* const x[],
                                                          int64_t                 incx,
                                                          int64_t                 batchCount,
                                                          int64_t*                result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIzaminBatched_v2_64(hipblasHandle_t               handle,
                                                          int64_t                       n,
                                                          const hipDoubleComplex* const x[],
                                                          int64_t                       incx,
                                                          int64_t                       batchCount,
                                                          int64_t*                      result);
//! @}

/*! @{
    \brief BLAS Level 1 API

    \details
     aminStridedBatched finds the first index of the element of minimum magnitude of each vector x_i in a batch, for i = 1, ..., batchCount.

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    n         [int]
              number of elements in each vector x_i
    @param[in]
    x         device pointer to the first vector x_1.
    @param[in]
    incx      [int]
              specifies the increment for the elements of each x_i. incx must be > 0.
    @param[in]
    stridex   [hipblasStride]
              specifies the pointer increment between one x_i and the next x_(i + 1)
    @param[in]
    batchCount [int]
              number of instances in the batch
    @param[out]
    result
              device or host pointer to array for storing contiguous batchCount results.
              return is 0 if n <= 0, incx<=0.

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasIsaminStridedBatched(hipblasHandle_t handle,
                                                           int             n,
                                                           const float*    x,
                                                           int             incx,
                                                           hipblasStride   stridex,
                                                           int             batchCount,
                                                           int*            result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIdaminStridedBatched(hipblasHandle_t handle,
                                                           int             n,
                                                           const double*   x,
                                                           int             incx,
                                                           hipblasStride   stridex,
                                                           int             batchCount,
                                                           int*            result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIcaminStridedBatched(hipblasHandle_t       handle,
                                                           int                   n,
                                                           const hipblasComplex* x,
                                                           int                   incx,
                                                           hipblasStride         stridex,
                                                           int                   batchCount,
                                                           int*                  result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIzaminStridedBatched(hipblasHandle_t             handle,
                                                           int                         n,
                                                           const hipblasDoubleComplex* x,
                                                           int                         incx,
                                                           hipblasStride               stridex,
                                                           int                         batchCount,
                                                           int*                        result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIcaminStridedBatched_v2(hipblasHandle_t   handle,
                                                              int               n,
                                                              const hipComplex* x,
                                                              int               incx,
                                                              hipblasStride     stridex,
                                                              int               batchCount,
                                                              int*              result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIzaminStridedBatched_v2(hipblasHandle_t         handle,
                                                              int                     n,
                                                              const hipDoubleComplex* x,
                                                              int                     incx,
                                                              hipblasStride           stridex,
                                                              int                     batchCount,
                                                              int*                    result);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasIsaminStridedBatched_64(hipblasHandle_t handle,
                                                              int64_t         n,
                                                              const float*    x,
                                                              int64_t         incx,
                                                              hipblasStride   stridex,
                                                              int64_t         batchCount,
                                                              int64_t*        result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIdaminStridedBatched_64(hipblasHandle_t handle,
                                                              int64_t         n,
                                                              const double*   x,
                                                              int64_t         incx,
                                                              hipblasStride   stridex,
                                                              int64_t         batchCount,
                                                              int64_t*        result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIcaminStridedBatched_64(hipblasHandle_t       handle,
                                                              int64_t               n,
                                                              const hipblasComplex* x,
                                                              int64_t               incx,
                                                              hipblasStride         stridex,
                                                              int64_t               batchCount,
                                                              int64_t*              result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIzaminStridedBatched_64(hipblasHandle_t             handle,
                                                              int64_t                     n,
                                                              const hipblasDoubleComplex* x,
                                                              int64_t                     incx,
                                                              hipblasStride               stridex,
                                                              int64_t  batchCount,
                                                              int64_t* result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIcaminStridedBatched_v2_64(hipblasHandle_t   handle,
                                                                 int64_t           n,
                                                                 const hipComplex* x,
                                                                 int64_t           incx,
                                                                 hipblasStride     stridex,
                                                                 int64_t           batchCount,
                                                                 int64_t*          result);

HIPBLAS_EXPORT hipblasStatus_t hipblasIzaminStridedBatched_v2_64(hipblasHandle_t         handle,
                                                                 int64_t                 n,
                                                                 const hipDoubleComplex* x,
                                                                 int64_t                 incx,
                                                                 hipblasStride           stridex,
                                                                 int64_t                 batchCount,
                                                                 int64_t*                result);
//! @}

/*! @{
    \brief BLAS Level 1 API

    \details
    asum computes the sum of the magnitudes of elements of a real vector x,
         or the sum of magnitudes of the real and imaginary parts of elements if x is a complex vector.

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : s,d,c,z

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    n         [int]
              the number of elements in x and y.
    @param[in]
    x         device pointer storing vector x.
    @param[in]
    incx      [int]
              specifies the increment for the elements of x. incx must be > 0.
    @param[inout]
    result
              device pointer or host pointer to store the asum product.
              return is 0.0 if n <= 0.

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t
    hipblasSasum(hipblasHandle_t handle, int n, const float* x, int incx, float* result);

HIPBLAS_EXPORT hipblasStatus_t
    hipblasDasum(hipblasHandle_t handle, int n, const double* x, int incx, double* result);

HIPBLAS_EXPORT hipblasStatus_t
    hipblasScasum(hipblasHandle_t handle, int n, const hipblasComplex* x, int incx, float* result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDzasum(
    hipblasHandle_t handle, int n, const hipblasDoubleComplex* x, int incx, double* result);

HIPBLAS_EXPORT hipblasStatus_t
    hipblasScasum_v2(hipblasHandle_t handle, int n, const hipComplex* x, int incx, float* result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDzasum_v2(
    hipblasHandle_t handle, int n, const hipDoubleComplex* x, int incx, double* result);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t
    hipblasSasum_64(hipblasHandle_t handle, int64_t n, const float* x, int64_t incx, float* result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDasum_64(
    hipblasHandle_t handle, int64_t n, const double* x, int64_t incx, double* result);

HIPBLAS_EXPORT hipblasStatus_t hipblasScasum_64(
    hipblasHandle_t handle, int64_t n, const hipblasComplex* x, int64_t incx, float* result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDzasum_64(
    hipblasHandle_t handle, int64_t n, const hipblasDoubleComplex* x, int64_t incx, double* result);

HIPBLAS_EXPORT hipblasStatus_t hipblasScasum_v2_64(
    hipblasHandle_t handle, int64_t n, const hipComplex* x, int64_t incx, float* result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDzasum_v2_64(
    hipblasHandle_t handle, int64_t n, const hipDoubleComplex* x, int64_t incx, double* result);
//! @}

/*! @{
    \brief BLAS Level 1 API

    \details
    asumBatched computes the sum of the magnitudes of the elements in a batch of real vectors x_i,
        or the sum of magnitudes of the real and imaginary parts of elements if x_i is a complex
        vector, for i = 1, ..., batchCount.

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    n         [int]
              number of elements in each vector x_i
    @param[in]
    x         device array of device pointers storing each vector x_i.
    @param[in]
    incx      [int]
              specifies the increment for the elements of each x_i. incx must be > 0.
    @param[in]
    batchCount [int]
              number of instances in the batch.
    @param[out]
    result
              device array or host array of batchCount size for results.
              return is 0.0 if n, incx<=0.

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSasumBatched(
    hipblasHandle_t handle, int n, const float* const x[], int incx, int batchCount, float* result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDasumBatched(hipblasHandle_t     handle,
                                                   int                 n,
                                                   const double* const x[],
                                                   int                 incx,
                                                   int                 batchCount,
                                                   double*             result);

HIPBLAS_EXPORT hipblasStatus_t hipblasScasumBatched(hipblasHandle_t             handle,
                                                    int                         n,
                                                    const hipblasComplex* const x[],
                                                    int                         incx,
                                                    int                         batchCount,
                                                    float*                      result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDzasumBatched(hipblasHandle_t                   handle,
                                                    int                               n,
                                                    const hipblasDoubleComplex* const x[],
                                                    int                               incx,
                                                    int                               batchCount,
                                                    double*                           result);

HIPBLAS_EXPORT hipblasStatus_t hipblasScasumBatched_v2(hipblasHandle_t         handle,
                                                       int                     n,
                                                       const hipComplex* const x[],
                                                       int                     incx,
                                                       int                     batchCount,
                                                       float*                  result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDzasumBatched_v2(hipblasHandle_t               handle,
                                                       int                           n,
                                                       const hipDoubleComplex* const x[],
                                                       int                           incx,
                                                       int                           batchCount,
                                                       double*                       result);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSasumBatched_64(hipblasHandle_t    handle,
                                                      int64_t            n,
                                                      const float* const x[],
                                                      int64_t            incx,
                                                      int64_t            batchCount,
                                                      float*             result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDasumBatched_64(hipblasHandle_t     handle,
                                                      int64_t             n,
                                                      const double* const x[],
                                                      int64_t             incx,
                                                      int64_t             batchCount,
                                                      double*             result);

HIPBLAS_EXPORT hipblasStatus_t hipblasScasumBatched_64(hipblasHandle_t             handle,
                                                       int64_t                     n,
                                                       const hipblasComplex* const x[],
                                                       int64_t                     incx,
                                                       int64_t                     batchCount,
                                                       float*                      result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDzasumBatched_64(hipblasHandle_t                   handle,
                                                       int64_t                           n,
                                                       const hipblasDoubleComplex* const x[],
                                                       int64_t                           incx,
                                                       int64_t                           batchCount,
                                                       double*                           result);

HIPBLAS_EXPORT hipblasStatus_t hipblasScasumBatched_v2_64(hipblasHandle_t         handle,
                                                          int64_t                 n,
                                                          const hipComplex* const x[],
                                                          int64_t                 incx,
                                                          int64_t                 batchCount,
                                                          float*                  result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDzasumBatched_v2_64(hipblasHandle_t               handle,
                                                          int64_t                       n,
                                                          const hipDoubleComplex* const x[],
                                                          int64_t                       incx,
                                                          int64_t                       batchCount,
                                                          double*                       result);
//! @}

/*! @{
    \brief BLAS Level 1 API

    \details
    asumStridedBatched computes the sum of the magnitudes of elements of a real vectors x_i,
        or the sum of magnitudes of the real and imaginary parts of elements if x_i is a complex
        vector, for i = 1, ..., batchCount.

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    n         [int]
              number of elements in each vector x_i
    @param[in]
    x         device pointer to the first vector x_1.
    @param[in]
    incx      [int]
              specifies the increment for the elements of each x_i. incx must be > 0.
    @param[in]
    stridex   [hipblasStride]
              stride from the start of one vector (x_i) and the next one (x_i+1).
              There are no restrictions placed on stride_x, however the user should
              take care to ensure that stride_x is of appropriate size, for a typical
              case this means stride_x >= n * incx.
    @param[in]
    batchCount [int]
              number of instances in the batch
    @param[out]
    result
              device pointer or host pointer to array for storing contiguous batchCount results.
              return is 0.0 if n, incx<=0.

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSasumStridedBatched(hipblasHandle_t handle,
                                                          int             n,
                                                          const float*    x,
                                                          int             incx,
                                                          hipblasStride   stridex,
                                                          int             batchCount,
                                                          float*          result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDasumStridedBatched(hipblasHandle_t handle,
                                                          int             n,
                                                          const double*   x,
                                                          int             incx,
                                                          hipblasStride   stridex,
                                                          int             batchCount,
                                                          double*         result);

HIPBLAS_EXPORT hipblasStatus_t hipblasScasumStridedBatched(hipblasHandle_t       handle,
                                                           int                   n,
                                                           const hipblasComplex* x,
                                                           int                   incx,
                                                           hipblasStride         stridex,
                                                           int                   batchCount,
                                                           float*                result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDzasumStridedBatched(hipblasHandle_t             handle,
                                                           int                         n,
                                                           const hipblasDoubleComplex* x,
                                                           int                         incx,
                                                           hipblasStride               stridex,
                                                           int                         batchCount,
                                                           double*                     result);

HIPBLAS_EXPORT hipblasStatus_t hipblasScasumStridedBatched_v2(hipblasHandle_t   handle,
                                                              int               n,
                                                              const hipComplex* x,
                                                              int               incx,
                                                              hipblasStride     stridex,
                                                              int               batchCount,
                                                              float*            result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDzasumStridedBatched_v2(hipblasHandle_t         handle,
                                                              int                     n,
                                                              const hipDoubleComplex* x,
                                                              int                     incx,
                                                              hipblasStride           stridex,
                                                              int                     batchCount,
                                                              double*                 result);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSasumStridedBatched_64(hipblasHandle_t handle,
                                                             int64_t         n,
                                                             const float*    x,
                                                             int64_t         incx,
                                                             hipblasStride   stridex,
                                                             int64_t         batchCount,
                                                             float*          result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDasumStridedBatched_64(hipblasHandle_t handle,
                                                             int64_t         n,
                                                             const double*   x,
                                                             int64_t         incx,
                                                             hipblasStride   stridex,
                                                             int64_t         batchCount,
                                                             double*         result);

HIPBLAS_EXPORT hipblasStatus_t hipblasScasumStridedBatched_64(hipblasHandle_t       handle,
                                                              int64_t               n,
                                                              const hipblasComplex* x,
                                                              int64_t               incx,
                                                              hipblasStride         stridex,
                                                              int64_t               batchCount,
                                                              float*                result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDzasumStridedBatched_64(hipblasHandle_t             handle,
                                                              int64_t                     n,
                                                              const hipblasDoubleComplex* x,
                                                              int64_t                     incx,
                                                              hipblasStride               stridex,
                                                              int64_t batchCount,
                                                              double* result);

HIPBLAS_EXPORT hipblasStatus_t hipblasScasumStridedBatched_v2_64(hipblasHandle_t   handle,
                                                                 int64_t           n,
                                                                 const hipComplex* x,
                                                                 int64_t           incx,
                                                                 hipblasStride     stridex,
                                                                 int64_t           batchCount,
                                                                 float*            result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDzasumStridedBatched_v2_64(hipblasHandle_t         handle,
                                                                 int64_t                 n,
                                                                 const hipDoubleComplex* x,
                                                                 int64_t                 incx,
                                                                 hipblasStride           stridex,
                                                                 int64_t                 batchCount,
                                                                 double*                 result);
//! @}

/*! @{
    \brief BLAS Level 1 API

    \details
    axpy   computes constant alpha multiplied by vector x, plus vector y

        y := alpha * x + y

    - Supported precisions in rocBLAS : h,s,d,c,z
    - Supported precisions in cuBLAS  : s,d,c,z

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    n         [int]
              the number of elements in x and y.
    @param[in]
    alpha     device pointer or host pointer to specify the scalar alpha.
    @param[in]
    x         device pointer storing vector x.
    @param[in]
    incx      [int]
              specifies the increment for the elements of x.
    @param[out]
    y         device pointer storing vector y.
    @param[inout]
    incy      [int]
              specifies the increment for the elements of y.

    ********************************************************************/
HIPBLAS_EXPORT hipblasStatus_t hipblasHaxpy(hipblasHandle_t    handle,
                                            int                n,
                                            const hipblasHalf* alpha,
                                            const hipblasHalf* x,
                                            int                incx,
                                            hipblasHalf*       y,
                                            int                incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasSaxpy(hipblasHandle_t handle,
                                            int             n,
                                            const float*    alpha,
                                            const float*    x,
                                            int             incx,
                                            float*          y,
                                            int             incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasDaxpy(hipblasHandle_t handle,
                                            int             n,
                                            const double*   alpha,
                                            const double*   x,
                                            int             incx,
                                            double*         y,
                                            int             incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasCaxpy(hipblasHandle_t       handle,
                                            int                   n,
                                            const hipblasComplex* alpha,
                                            const hipblasComplex* x,
                                            int                   incx,
                                            hipblasComplex*       y,
                                            int                   incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasZaxpy(hipblasHandle_t             handle,
                                            int                         n,
                                            const hipblasDoubleComplex* alpha,
                                            const hipblasDoubleComplex* x,
                                            int                         incx,
                                            hipblasDoubleComplex*       y,
                                            int                         incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasCaxpy_v2(hipblasHandle_t   handle,
                                               int               n,
                                               const hipComplex* alpha,
                                               const hipComplex* x,
                                               int               incx,
                                               hipComplex*       y,
                                               int               incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasZaxpy_v2(hipblasHandle_t         handle,
                                               int                     n,
                                               const hipDoubleComplex* alpha,
                                               const hipDoubleComplex* x,
                                               int                     incx,
                                               hipDoubleComplex*       y,
                                               int                     incy);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasHaxpy_64(hipblasHandle_t    handle,
                                               int64_t            n,
                                               const hipblasHalf* alpha,
                                               const hipblasHalf* x,
                                               int64_t            incx,
                                               hipblasHalf*       y,
                                               int64_t            incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasSaxpy_64(hipblasHandle_t handle,
                                               int64_t         n,
                                               const float*    alpha,
                                               const float*    x,
                                               int64_t         incx,
                                               float*          y,
                                               int64_t         incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasDaxpy_64(hipblasHandle_t handle,
                                               int64_t         n,
                                               const double*   alpha,
                                               const double*   x,
                                               int64_t         incx,
                                               double*         y,
                                               int64_t         incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasCaxpy_64(hipblasHandle_t       handle,
                                               int64_t               n,
                                               const hipblasComplex* alpha,
                                               const hipblasComplex* x,
                                               int64_t               incx,
                                               hipblasComplex*       y,
                                               int64_t               incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasZaxpy_64(hipblasHandle_t             handle,
                                               int64_t                     n,
                                               const hipblasDoubleComplex* alpha,
                                               const hipblasDoubleComplex* x,
                                               int64_t                     incx,
                                               hipblasDoubleComplex*       y,
                                               int64_t                     incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasCaxpy_v2_64(hipblasHandle_t   handle,
                                                  int64_t           n,
                                                  const hipComplex* alpha,
                                                  const hipComplex* x,
                                                  int64_t           incx,
                                                  hipComplex*       y,
                                                  int64_t           incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasZaxpy_v2_64(hipblasHandle_t         handle,
                                                  int64_t                 n,
                                                  const hipDoubleComplex* alpha,
                                                  const hipDoubleComplex* x,
                                                  int64_t                 incx,
                                                  hipDoubleComplex*       y,
                                                  int64_t                 incy);

//! @}

/*! @{
    \brief BLAS Level 1 API

    \details
    axpyBatched   compute y := alpha * x + y over a set of batched vectors.

    - Supported precisions in rocBLAS : h,s,d,c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    n         [int]
              the number of elements in x and y.
    @param[in]
    alpha     specifies the scalar alpha.
    @param[in]
    x         pointer storing vector x on the GPU.
    @param[in]
    incx      [int]
              specifies the increment for the elements of x.
    @param[out]
    y         pointer storing vector y on the GPU.
    @param[inout]
    incy      [int]
              specifies the increment for the elements of y.

    @param[in]
    batchCount [int]
              number of instances in the batch

    ********************************************************************/
HIPBLAS_EXPORT hipblasStatus_t hipblasHaxpyBatched(hipblasHandle_t          handle,
                                                   int                      n,
                                                   const hipblasHalf*       alpha,
                                                   const hipblasHalf* const x[],
                                                   int                      incx,
                                                   hipblasHalf* const       y[],
                                                   int                      incy,
                                                   int                      batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasSaxpyBatched(hipblasHandle_t    handle,
                                                   int                n,
                                                   const float*       alpha,
                                                   const float* const x[],
                                                   int                incx,
                                                   float* const       y[],
                                                   int                incy,
                                                   int                batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDaxpyBatched(hipblasHandle_t     handle,
                                                   int                 n,
                                                   const double*       alpha,
                                                   const double* const x[],
                                                   int                 incx,
                                                   double* const       y[],
                                                   int                 incy,
                                                   int                 batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCaxpyBatched(hipblasHandle_t             handle,
                                                   int                         n,
                                                   const hipblasComplex*       alpha,
                                                   const hipblasComplex* const x[],
                                                   int                         incx,
                                                   hipblasComplex* const       y[],
                                                   int                         incy,
                                                   int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZaxpyBatched(hipblasHandle_t                   handle,
                                                   int                               n,
                                                   const hipblasDoubleComplex*       alpha,
                                                   const hipblasDoubleComplex* const x[],
                                                   int                               incx,
                                                   hipblasDoubleComplex* const       y[],
                                                   int                               incy,
                                                   int                               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCaxpyBatched_v2(hipblasHandle_t         handle,
                                                      int                     n,
                                                      const hipComplex*       alpha,
                                                      const hipComplex* const x[],
                                                      int                     incx,
                                                      hipComplex* const       y[],
                                                      int                     incy,
                                                      int                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZaxpyBatched_v2(hipblasHandle_t               handle,
                                                      int                           n,
                                                      const hipDoubleComplex*       alpha,
                                                      const hipDoubleComplex* const x[],
                                                      int                           incx,
                                                      hipDoubleComplex* const       y[],
                                                      int                           incy,
                                                      int                           batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasHaxpyBatched_64(hipblasHandle_t          handle,
                                                      int64_t                  n,
                                                      const hipblasHalf*       alpha,
                                                      const hipblasHalf* const x[],
                                                      int64_t                  incx,
                                                      hipblasHalf* const       y[],
                                                      int64_t                  incy,
                                                      int64_t                  batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasSaxpyBatched_64(hipblasHandle_t    handle,
                                                      int64_t            n,
                                                      const float*       alpha,
                                                      const float* const x[],
                                                      int64_t            incx,
                                                      float* const       y[],
                                                      int64_t            incy,
                                                      int64_t            batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDaxpyBatched_64(hipblasHandle_t     handle,
                                                      int64_t             n,
                                                      const double*       alpha,
                                                      const double* const x[],
                                                      int64_t             incx,
                                                      double* const       y[],
                                                      int64_t             incy,
                                                      int64_t             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCaxpyBatched_64(hipblasHandle_t             handle,
                                                      int64_t                     n,
                                                      const hipblasComplex*       alpha,
                                                      const hipblasComplex* const x[],
                                                      int64_t                     incx,
                                                      hipblasComplex* const       y[],
                                                      int64_t                     incy,
                                                      int64_t                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZaxpyBatched_64(hipblasHandle_t                   handle,
                                                      int64_t                           n,
                                                      const hipblasDoubleComplex*       alpha,
                                                      const hipblasDoubleComplex* const x[],
                                                      int64_t                           incx,
                                                      hipblasDoubleComplex* const       y[],
                                                      int64_t                           incy,
                                                      int64_t                           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCaxpyBatched_v2_64(hipblasHandle_t         handle,
                                                         int64_t                 n,
                                                         const hipComplex*       alpha,
                                                         const hipComplex* const x[],
                                                         int64_t                 incx,
                                                         hipComplex* const       y[],
                                                         int64_t                 incy,
                                                         int64_t                 batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZaxpyBatched_v2_64(hipblasHandle_t               handle,
                                                         int64_t                       n,
                                                         const hipDoubleComplex*       alpha,
                                                         const hipDoubleComplex* const x[],
                                                         int64_t                       incx,
                                                         hipDoubleComplex* const       y[],
                                                         int64_t                       incy,
                                                         int64_t                       batchCount);

//! @}

/*! @{
    \brief BLAS Level 1 API

    \details
    axpyStridedBatched   compute y := alpha * x + y over a set of strided batched vectors.

    - Supported precisions in rocBLAS : h,s,d,c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    n         [int]
    @param[in]
    alpha     specifies the scalar alpha.
    @param[in]
    x         pointer storing vector x on the GPU.
    @param[in]
    incx      [int]
              specifies the increment for the elements of x.
    @param[in]
    stridex   [hipblasStride]
              specifies the increment between vectors of x.
    @param[out]
    y         pointer storing vector y on the GPU.
    @param[inout]
    incy      [int]
              specifies the increment for the elements of y.
    @param[in]
    stridey   [hipblasStride]
              specifies the increment between vectors of y.

    @param[in]
    batchCount [int]
              number of instances in the batch

            ********************************************************************/
HIPBLAS_EXPORT hipblasStatus_t hipblasHaxpyStridedBatched(hipblasHandle_t    handle,
                                                          int                n,
                                                          const hipblasHalf* alpha,
                                                          const hipblasHalf* x,
                                                          int                incx,
                                                          hipblasStride      stridex,
                                                          hipblasHalf*       y,
                                                          int                incy,
                                                          hipblasStride      stridey,
                                                          int                batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasSaxpyStridedBatched(hipblasHandle_t handle,
                                                          int             n,
                                                          const float*    alpha,
                                                          const float*    x,
                                                          int             incx,
                                                          hipblasStride   stridex,
                                                          float*          y,
                                                          int             incy,
                                                          hipblasStride   stridey,
                                                          int             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDaxpyStridedBatched(hipblasHandle_t handle,
                                                          int             n,
                                                          const double*   alpha,
                                                          const double*   x,
                                                          int             incx,
                                                          hipblasStride   stridex,
                                                          double*         y,
                                                          int             incy,
                                                          hipblasStride   stridey,
                                                          int             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCaxpyStridedBatched(hipblasHandle_t       handle,
                                                          int                   n,
                                                          const hipblasComplex* alpha,
                                                          const hipblasComplex* x,
                                                          int                   incx,
                                                          hipblasStride         stridex,
                                                          hipblasComplex*       y,
                                                          int                   incy,
                                                          hipblasStride         stridey,
                                                          int                   batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZaxpyStridedBatched(hipblasHandle_t             handle,
                                                          int                         n,
                                                          const hipblasDoubleComplex* alpha,
                                                          const hipblasDoubleComplex* x,
                                                          int                         incx,
                                                          hipblasStride               stridex,
                                                          hipblasDoubleComplex*       y,
                                                          int                         incy,
                                                          hipblasStride               stridey,
                                                          int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCaxpyStridedBatched_v2(hipblasHandle_t   handle,
                                                             int               n,
                                                             const hipComplex* alpha,
                                                             const hipComplex* x,
                                                             int               incx,
                                                             hipblasStride     stridex,
                                                             hipComplex*       y,
                                                             int               incy,
                                                             hipblasStride     stridey,
                                                             int               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZaxpyStridedBatched_v2(hipblasHandle_t         handle,
                                                             int                     n,
                                                             const hipDoubleComplex* alpha,
                                                             const hipDoubleComplex* x,
                                                             int                     incx,
                                                             hipblasStride           stridex,
                                                             hipDoubleComplex*       y,
                                                             int                     incy,
                                                             hipblasStride           stridey,
                                                             int                     batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasHaxpyStridedBatched_64(hipblasHandle_t    handle,
                                                             int64_t            n,
                                                             const hipblasHalf* alpha,
                                                             const hipblasHalf* x,
                                                             int64_t            incx,
                                                             hipblasStride      stridex,
                                                             hipblasHalf*       y,
                                                             int64_t            incy,
                                                             hipblasStride      stridey,
                                                             int64_t            batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasSaxpyStridedBatched_64(hipblasHandle_t handle,
                                                             int64_t         n,
                                                             const float*    alpha,
                                                             const float*    x,
                                                             int64_t         incx,
                                                             hipblasStride   stridex,
                                                             float*          y,
                                                             int64_t         incy,
                                                             hipblasStride   stridey,
                                                             int64_t         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDaxpyStridedBatched_64(hipblasHandle_t handle,
                                                             int64_t         n,
                                                             const double*   alpha,
                                                             const double*   x,
                                                             int64_t         incx,
                                                             hipblasStride   stridex,
                                                             double*         y,
                                                             int64_t         incy,
                                                             hipblasStride   stridey,
                                                             int64_t         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCaxpyStridedBatched_64(hipblasHandle_t       handle,
                                                             int64_t               n,
                                                             const hipblasComplex* alpha,
                                                             const hipblasComplex* x,
                                                             int64_t               incx,
                                                             hipblasStride         stridex,
                                                             hipblasComplex*       y,
                                                             int64_t               incy,
                                                             hipblasStride         stridey,
                                                             int64_t               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZaxpyStridedBatched_64(hipblasHandle_t             handle,
                                                             int64_t                     n,
                                                             const hipblasDoubleComplex* alpha,
                                                             const hipblasDoubleComplex* x,
                                                             int64_t                     incx,
                                                             hipblasStride               stridex,
                                                             hipblasDoubleComplex*       y,
                                                             int64_t                     incy,
                                                             hipblasStride               stridey,
                                                             int64_t batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCaxpyStridedBatched_v2_64(hipblasHandle_t   handle,
                                                                int64_t           n,
                                                                const hipComplex* alpha,
                                                                const hipComplex* x,
                                                                int64_t           incx,
                                                                hipblasStride     stridex,
                                                                hipComplex*       y,
                                                                int64_t           incy,
                                                                hipblasStride     stridey,
                                                                int64_t           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZaxpyStridedBatched_v2_64(hipblasHandle_t         handle,
                                                                int64_t                 n,
                                                                const hipDoubleComplex* alpha,
                                                                const hipDoubleComplex* x,
                                                                int64_t                 incx,
                                                                hipblasStride           stridex,
                                                                hipDoubleComplex*       y,
                                                                int64_t                 incy,
                                                                hipblasStride           stridey,
                                                                int64_t                 batchCount);

//! @}

/*! @{
    \brief BLAS Level 1 API

    \details
    copy  copies each element x[i] into y[i], for  i = 1 , ... , n

        y := x,

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : s,d,c,z

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    n         [int]
              the number of elements in x to be copied to y.
    @param[in]
    x         device pointer storing vector x.
    @param[in]
    incx      [int]
              specifies the increment for the elements of x.
    @param[out]
    y         device pointer storing vector y.
    @param[in]
    incy      [int]
              specifies the increment for the elements of y.

            ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t
    hipblasScopy(hipblasHandle_t handle, int n, const float* x, int incx, float* y, int incy);

HIPBLAS_EXPORT hipblasStatus_t
    hipblasDcopy(hipblasHandle_t handle, int n, const double* x, int incx, double* y, int incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasCcopy(
    hipblasHandle_t handle, int n, const hipblasComplex* x, int incx, hipblasComplex* y, int incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasZcopy(hipblasHandle_t             handle,
                                            int                         n,
                                            const hipblasDoubleComplex* x,
                                            int                         incx,
                                            hipblasDoubleComplex*       y,
                                            int                         incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasCcopy_v2(
    hipblasHandle_t handle, int n, const hipComplex* x, int incx, hipComplex* y, int incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasZcopy_v2(hipblasHandle_t         handle,
                                               int                     n,
                                               const hipDoubleComplex* x,
                                               int                     incx,
                                               hipDoubleComplex*       y,
                                               int                     incy);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasScopy_64(
    hipblasHandle_t handle, int64_t n, const float* x, int64_t incx, float* y, int64_t incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasDcopy_64(
    hipblasHandle_t handle, int64_t n, const double* x, int64_t incx, double* y, int64_t incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasCcopy_64(hipblasHandle_t       handle,
                                               int64_t               n,
                                               const hipblasComplex* x,
                                               int64_t               incx,
                                               hipblasComplex*       y,
                                               int64_t               incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasZcopy_64(hipblasHandle_t             handle,
                                               int64_t                     n,
                                               const hipblasDoubleComplex* x,
                                               int64_t                     incx,
                                               hipblasDoubleComplex*       y,
                                               int64_t                     incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasCcopy_v2_64(hipblasHandle_t   handle,
                                                  int64_t           n,
                                                  const hipComplex* x,
                                                  int64_t           incx,
                                                  hipComplex*       y,
                                                  int64_t           incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasZcopy_v2_64(hipblasHandle_t         handle,
                                                  int64_t                 n,
                                                  const hipDoubleComplex* x,
                                                  int64_t                 incx,
                                                  hipDoubleComplex*       y,
                                                  int64_t                 incy);
//! @}

/*! @{
    \brief BLAS Level 1 API

    \details
    copyBatched copies each element x_i[j] into y_i[j], for  j = 1 , ... , n; i = 1 , ... , batchCount

        y_i := x_i,

    where (x_i, y_i) is the i-th instance of the batch.
    x_i and y_i are vectors.

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    n         [int]
              the number of elements in each x_i to be copied to y_i.
    @param[in]
    x         device array of device pointers storing each vector x_i.
    @param[in]
    incx      [int]
              specifies the increment for the elements of each vector x_i.
    @param[out]
    y         device array of device pointers storing each vector y_i.
    @param[in]
    incy      [int]
              specifies the increment for the elements of each vector y_i.
    @param[in]
    batchCount [int]
                number of instances in the batch

            ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasScopyBatched(hipblasHandle_t    handle,
                                                   int                n,
                                                   const float* const x[],
                                                   int                incx,
                                                   float* const       y[],
                                                   int                incy,
                                                   int                batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDcopyBatched(hipblasHandle_t     handle,
                                                   int                 n,
                                                   const double* const x[],
                                                   int                 incx,
                                                   double* const       y[],
                                                   int                 incy,
                                                   int                 batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCcopyBatched(hipblasHandle_t             handle,
                                                   int                         n,
                                                   const hipblasComplex* const x[],
                                                   int                         incx,
                                                   hipblasComplex* const       y[],
                                                   int                         incy,
                                                   int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZcopyBatched(hipblasHandle_t                   handle,
                                                   int                               n,
                                                   const hipblasDoubleComplex* const x[],
                                                   int                               incx,
                                                   hipblasDoubleComplex* const       y[],
                                                   int                               incy,
                                                   int                               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCcopyBatched_v2(hipblasHandle_t         handle,
                                                      int                     n,
                                                      const hipComplex* const x[],
                                                      int                     incx,
                                                      hipComplex* const       y[],
                                                      int                     incy,
                                                      int                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZcopyBatched_v2(hipblasHandle_t               handle,
                                                      int                           n,
                                                      const hipDoubleComplex* const x[],
                                                      int                           incx,
                                                      hipDoubleComplex* const       y[],
                                                      int                           incy,
                                                      int                           batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasScopyBatched_64(hipblasHandle_t    handle,
                                                      int64_t            n,
                                                      const float* const x[],
                                                      int64_t            incx,
                                                      float* const       y[],
                                                      int64_t            incy,
                                                      int64_t            batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDcopyBatched_64(hipblasHandle_t     handle,
                                                      int64_t             n,
                                                      const double* const x[],
                                                      int64_t             incx,
                                                      double* const       y[],
                                                      int64_t             incy,
                                                      int64_t             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCcopyBatched_64(hipblasHandle_t             handle,
                                                      int64_t                     n,
                                                      const hipblasComplex* const x[],
                                                      int64_t                     incx,
                                                      hipblasComplex* const       y[],
                                                      int64_t                     incy,
                                                      int64_t                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZcopyBatched_64(hipblasHandle_t                   handle,
                                                      int64_t                           n,
                                                      const hipblasDoubleComplex* const x[],
                                                      int64_t                           incx,
                                                      hipblasDoubleComplex* const       y[],
                                                      int64_t                           incy,
                                                      int64_t                           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCcopyBatched_v2_64(hipblasHandle_t         handle,
                                                         int64_t                 n,
                                                         const hipComplex* const x[],
                                                         int64_t                 incx,
                                                         hipComplex* const       y[],
                                                         int64_t                 incy,
                                                         int64_t                 batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZcopyBatched_v2_64(hipblasHandle_t               handle,
                                                         int64_t                       n,
                                                         const hipDoubleComplex* const x[],
                                                         int64_t                       incx,
                                                         hipDoubleComplex* const       y[],
                                                         int64_t                       incy,
                                                         int64_t                       batchCount);
//! @}

/*! @{
    \brief BLAS Level 1 API

    \details
    copyStridedBatched copies each element x_i[j] into y_i[j], for  j = 1 , ... , n; i = 1 , ... , batchCount

        y_i := x_i,

    where (x_i, y_i) is the i-th instance of the batch.
    x_i and y_i are vectors.

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    n         [int]
              the number of elements in each x_i to be copied to y_i.
    @param[in]
    x         device pointer to the first vector (x_1) in the batch.
    @param[in]
    incx      [int]
              specifies the increments for the elements of vectors x_i.
    @param[in]
    stridex     [hipblasStride]
                stride from the start of one vector (x_i) and the next one (x_i+1).
                There are no restrictions placed on stride_x, however the user should
                take care to ensure that stride_x is of appropriate size, for a typical
                case this means stride_x >= n * incx.
    @param[out]
    y         device pointer to the first vector (y_1) in the batch.
    @param[in]
    incy      [int]
              specifies the increment for the elements of vectors y_i.
    @param[in]
    stridey     [hipblasStride]
                stride from the start of one vector (y_i) and the next one (y_i+1).
                There are no restrictions placed on stride_y, however the user should
                take care to ensure that stride_y is of appropriate size, for a typical
                case this means stride_y >= n * incy. stridey should be non zero.
    @param[in]
    batchCount [int]
                number of instances in the batch

            ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasScopyStridedBatched(hipblasHandle_t handle,
                                                          int             n,
                                                          const float*    x,
                                                          int             incx,
                                                          hipblasStride   stridex,
                                                          float*          y,
                                                          int             incy,
                                                          hipblasStride   stridey,
                                                          int             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDcopyStridedBatched(hipblasHandle_t handle,
                                                          int             n,
                                                          const double*   x,
                                                          int             incx,
                                                          hipblasStride   stridex,
                                                          double*         y,
                                                          int             incy,
                                                          hipblasStride   stridey,
                                                          int             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCcopyStridedBatched(hipblasHandle_t       handle,
                                                          int                   n,
                                                          const hipblasComplex* x,
                                                          int                   incx,
                                                          hipblasStride         stridex,
                                                          hipblasComplex*       y,
                                                          int                   incy,
                                                          hipblasStride         stridey,
                                                          int                   batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZcopyStridedBatched(hipblasHandle_t             handle,
                                                          int                         n,
                                                          const hipblasDoubleComplex* x,
                                                          int                         incx,
                                                          hipblasStride               stridex,
                                                          hipblasDoubleComplex*       y,
                                                          int                         incy,
                                                          hipblasStride               stridey,
                                                          int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCcopyStridedBatched_v2(hipblasHandle_t   handle,
                                                             int               n,
                                                             const hipComplex* x,
                                                             int               incx,
                                                             hipblasStride     stridex,
                                                             hipComplex*       y,
                                                             int               incy,
                                                             hipblasStride     stridey,
                                                             int               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZcopyStridedBatched_v2(hipblasHandle_t         handle,
                                                             int                     n,
                                                             const hipDoubleComplex* x,
                                                             int                     incx,
                                                             hipblasStride           stridex,
                                                             hipDoubleComplex*       y,
                                                             int                     incy,
                                                             hipblasStride           stridey,
                                                             int                     batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasScopyStridedBatched_64(hipblasHandle_t handle,
                                                             int64_t         n,
                                                             const float*    x,
                                                             int64_t         incx,
                                                             hipblasStride   stridex,
                                                             float*          y,
                                                             int64_t         incy,
                                                             hipblasStride   stridey,
                                                             int64_t         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDcopyStridedBatched_64(hipblasHandle_t handle,
                                                             int64_t         n,
                                                             const double*   x,
                                                             int64_t         incx,
                                                             hipblasStride   stridex,
                                                             double*         y,
                                                             int64_t         incy,
                                                             hipblasStride   stridey,
                                                             int64_t         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCcopyStridedBatched_64(hipblasHandle_t       handle,
                                                             int64_t               n,
                                                             const hipblasComplex* x,
                                                             int64_t               incx,
                                                             hipblasStride         stridex,
                                                             hipblasComplex*       y,
                                                             int64_t               incy,
                                                             hipblasStride         stridey,
                                                             int64_t               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZcopyStridedBatched_64(hipblasHandle_t             handle,
                                                             int64_t                     n,
                                                             const hipblasDoubleComplex* x,
                                                             int64_t                     incx,
                                                             hipblasStride               stridex,
                                                             hipblasDoubleComplex*       y,
                                                             int64_t                     incy,
                                                             hipblasStride               stridey,
                                                             int64_t batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCcopyStridedBatched_v2_64(hipblasHandle_t   handle,
                                                                int64_t           n,
                                                                const hipComplex* x,
                                                                int64_t           incx,
                                                                hipblasStride     stridex,
                                                                hipComplex*       y,
                                                                int64_t           incy,
                                                                hipblasStride     stridey,
                                                                int64_t           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZcopyStridedBatched_v2_64(hipblasHandle_t         handle,
                                                                int64_t                 n,
                                                                const hipDoubleComplex* x,
                                                                int64_t                 incx,
                                                                hipblasStride           stridex,
                                                                hipDoubleComplex*       y,
                                                                int64_t                 incy,
                                                                hipblasStride           stridey,
                                                                int64_t                 batchCount);
//! @}

/*! @{
    \brief BLAS Level 1 API

    \details
    dot(u)  performs the dot product of vectors x and y

        result = x * y;

    dotc  performs the dot product of the conjugate of complex vector x and complex vector y

        result = conjugate (x) * y;

    - Supported precisions in rocBLAS : h,bf,s,d,c,z
    - Supported precisions in cuBLAS  : s,d,c,z

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    n         [int]
              the number of elements in x and y.
    @param[in]
    x         device pointer storing vector x.
    @param[in]
    incx      [int]
              specifies the increment for the elements of y.
    @param[in]
    y         device pointer storing vector y.
    @param[in]
    incy      [int]
              specifies the increment for the elements of y.
    @param[inout]
    result
              device pointer or host pointer to store the dot product.
              return is 0.0 if n <= 0.

            ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasHdot(hipblasHandle_t    handle,
                                           int                n,
                                           const hipblasHalf* x,
                                           int                incx,
                                           const hipblasHalf* y,
                                           int                incy,
                                           hipblasHalf*       result);

HIPBLAS_EXPORT hipblasStatus_t hipblasBfdot(hipblasHandle_t        handle,
                                            int                    n,
                                            const hipblasBfloat16* x,
                                            int                    incx,
                                            const hipblasBfloat16* y,
                                            int                    incy,
                                            hipblasBfloat16*       result);

HIPBLAS_EXPORT hipblasStatus_t hipblasSdot(hipblasHandle_t handle,
                                           int             n,
                                           const float*    x,
                                           int             incx,
                                           const float*    y,
                                           int             incy,
                                           float*          result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDdot(hipblasHandle_t handle,
                                           int             n,
                                           const double*   x,
                                           int             incx,
                                           const double*   y,
                                           int             incy,
                                           double*         result);

HIPBLAS_EXPORT hipblasStatus_t hipblasCdotc(hipblasHandle_t       handle,
                                            int                   n,
                                            const hipblasComplex* x,
                                            int                   incx,
                                            const hipblasComplex* y,
                                            int                   incy,
                                            hipblasComplex*       result);

HIPBLAS_EXPORT hipblasStatus_t hipblasCdotu(hipblasHandle_t       handle,
                                            int                   n,
                                            const hipblasComplex* x,
                                            int                   incx,
                                            const hipblasComplex* y,
                                            int                   incy,
                                            hipblasComplex*       result);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdotc(hipblasHandle_t             handle,
                                            int                         n,
                                            const hipblasDoubleComplex* x,
                                            int                         incx,
                                            const hipblasDoubleComplex* y,
                                            int                         incy,
                                            hipblasDoubleComplex*       result);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdotu(hipblasHandle_t             handle,
                                            int                         n,
                                            const hipblasDoubleComplex* x,
                                            int                         incx,
                                            const hipblasDoubleComplex* y,
                                            int                         incy,
                                            hipblasDoubleComplex*       result);

HIPBLAS_EXPORT hipblasStatus_t hipblasCdotc_v2(hipblasHandle_t   handle,
                                               int               n,
                                               const hipComplex* x,
                                               int               incx,
                                               const hipComplex* y,
                                               int               incy,
                                               hipComplex*       result);

HIPBLAS_EXPORT hipblasStatus_t hipblasCdotu_v2(hipblasHandle_t   handle,
                                               int               n,
                                               const hipComplex* x,
                                               int               incx,
                                               const hipComplex* y,
                                               int               incy,
                                               hipComplex*       result);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdotc_v2(hipblasHandle_t         handle,
                                               int                     n,
                                               const hipDoubleComplex* x,
                                               int                     incx,
                                               const hipDoubleComplex* y,
                                               int                     incy,
                                               hipDoubleComplex*       result);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdotu_v2(hipblasHandle_t         handle,
                                               int                     n,
                                               const hipDoubleComplex* x,
                                               int                     incx,
                                               const hipDoubleComplex* y,
                                               int                     incy,
                                               hipDoubleComplex*       result);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasHdot_64(hipblasHandle_t    handle,
                                              int64_t            n,
                                              const hipblasHalf* x,
                                              int64_t            incx,
                                              const hipblasHalf* y,
                                              int64_t            incy,
                                              hipblasHalf*       result);

HIPBLAS_EXPORT hipblasStatus_t hipblasBfdot_64(hipblasHandle_t        handle,
                                               int64_t                n,
                                               const hipblasBfloat16* x,
                                               int64_t                incx,
                                               const hipblasBfloat16* y,
                                               int64_t                incy,
                                               hipblasBfloat16*       result);

HIPBLAS_EXPORT hipblasStatus_t hipblasSdot_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              const float*    x,
                                              int64_t         incx,
                                              const float*    y,
                                              int64_t         incy,
                                              float*          result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDdot_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              const double*   x,
                                              int64_t         incx,
                                              const double*   y,
                                              int64_t         incy,
                                              double*         result);

HIPBLAS_EXPORT hipblasStatus_t hipblasCdotc_64(hipblasHandle_t       handle,
                                               int64_t               n,
                                               const hipblasComplex* x,
                                               int64_t               incx,
                                               const hipblasComplex* y,
                                               int64_t               incy,
                                               hipblasComplex*       result);

HIPBLAS_EXPORT hipblasStatus_t hipblasCdotu_64(hipblasHandle_t       handle,
                                               int64_t               n,
                                               const hipblasComplex* x,
                                               int64_t               incx,
                                               const hipblasComplex* y,
                                               int64_t               incy,
                                               hipblasComplex*       result);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdotc_64(hipblasHandle_t             handle,
                                               int64_t                     n,
                                               const hipblasDoubleComplex* x,
                                               int64_t                     incx,
                                               const hipblasDoubleComplex* y,
                                               int64_t                     incy,
                                               hipblasDoubleComplex*       result);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdotu_64(hipblasHandle_t             handle,
                                               int64_t                     n,
                                               const hipblasDoubleComplex* x,
                                               int64_t                     incx,
                                               const hipblasDoubleComplex* y,
                                               int64_t                     incy,
                                               hipblasDoubleComplex*       result);

HIPBLAS_EXPORT hipblasStatus_t hipblasCdotc_v2_64(hipblasHandle_t   handle,
                                                  int64_t           n,
                                                  const hipComplex* x,
                                                  int64_t           incx,
                                                  const hipComplex* y,
                                                  int64_t           incy,
                                                  hipComplex*       result);

HIPBLAS_EXPORT hipblasStatus_t hipblasCdotu_v2_64(hipblasHandle_t   handle,
                                                  int64_t           n,
                                                  const hipComplex* x,
                                                  int64_t           incx,
                                                  const hipComplex* y,
                                                  int64_t           incy,
                                                  hipComplex*       result);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdotc_v2_64(hipblasHandle_t         handle,
                                                  int64_t                 n,
                                                  const hipDoubleComplex* x,
                                                  int64_t                 incx,
                                                  const hipDoubleComplex* y,
                                                  int64_t                 incy,
                                                  hipDoubleComplex*       result);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdotu_v2_64(hipblasHandle_t         handle,
                                                  int64_t                 n,
                                                  const hipDoubleComplex* x,
                                                  int64_t                 incx,
                                                  const hipDoubleComplex* y,
                                                  int64_t                 incy,
                                                  hipDoubleComplex*       result);
//! @}

/*! @{
    \brief BLAS Level 1 API

    \details
    dotBatched(u) performs a batch of dot products of vectors x and y

        result_i = x_i * y_i;

    dotcBatched  performs a batch of dot products of the conjugate of complex vector x and complex vector y

        result_i = conjugate (x_i) * y_i;

    where (x_i, y_i) is the i-th instance of the batch.
    x_i and y_i are vectors, for i = 1, ..., batchCount

    - Supported precisions in rocBLAS : h,bf,s,d,c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    n         [int]
              the number of elements in each x_i and y_i.
    @param[in]
    x         device array of device pointers storing each vector x_i.
    @param[in]
    incx      [int]
              specifies the increment for the elements of each x_i.
    @param[in]
    y         device array of device pointers storing each vector y_i.
    @param[in]
    incy      [int]
              specifies the increment for the elements of each y_i.
    @param[in]
    batchCount [int]
                number of instances in the batch
    @param[inout]
    result
              device array or host array of batchCount size to store the dot products of each batch.
              return 0.0 for each element if n <= 0.

            ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasHdotBatched(hipblasHandle_t          handle,
                                                  int                      n,
                                                  const hipblasHalf* const x[],
                                                  int                      incx,
                                                  const hipblasHalf* const y[],
                                                  int                      incy,
                                                  int                      batchCount,
                                                  hipblasHalf*             result);

HIPBLAS_EXPORT hipblasStatus_t hipblasBfdotBatched(hipblasHandle_t              handle,
                                                   int                          n,
                                                   const hipblasBfloat16* const x[],
                                                   int                          incx,
                                                   const hipblasBfloat16* const y[],
                                                   int                          incy,
                                                   int                          batchCount,
                                                   hipblasBfloat16*             result);

HIPBLAS_EXPORT hipblasStatus_t hipblasSdotBatched(hipblasHandle_t    handle,
                                                  int                n,
                                                  const float* const x[],
                                                  int                incx,
                                                  const float* const y[],
                                                  int                incy,
                                                  int                batchCount,
                                                  float*             result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDdotBatched(hipblasHandle_t     handle,
                                                  int                 n,
                                                  const double* const x[],
                                                  int                 incx,
                                                  const double* const y[],
                                                  int                 incy,
                                                  int                 batchCount,
                                                  double*             result);

HIPBLAS_EXPORT hipblasStatus_t hipblasCdotcBatched(hipblasHandle_t             handle,
                                                   int                         n,
                                                   const hipblasComplex* const x[],
                                                   int                         incx,
                                                   const hipblasComplex* const y[],
                                                   int                         incy,
                                                   int                         batchCount,
                                                   hipblasComplex*             result);

HIPBLAS_EXPORT hipblasStatus_t hipblasCdotuBatched(hipblasHandle_t             handle,
                                                   int                         n,
                                                   const hipblasComplex* const x[],
                                                   int                         incx,
                                                   const hipblasComplex* const y[],
                                                   int                         incy,
                                                   int                         batchCount,
                                                   hipblasComplex*             result);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdotcBatched(hipblasHandle_t                   handle,
                                                   int                               n,
                                                   const hipblasDoubleComplex* const x[],
                                                   int                               incx,
                                                   const hipblasDoubleComplex* const y[],
                                                   int                               incy,
                                                   int                               batchCount,
                                                   hipblasDoubleComplex*             result);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdotuBatched(hipblasHandle_t                   handle,
                                                   int                               n,
                                                   const hipblasDoubleComplex* const x[],
                                                   int                               incx,
                                                   const hipblasDoubleComplex* const y[],
                                                   int                               incy,
                                                   int                               batchCount,
                                                   hipblasDoubleComplex*             result);

HIPBLAS_EXPORT hipblasStatus_t hipblasCdotcBatched_v2(hipblasHandle_t         handle,
                                                      int                     n,
                                                      const hipComplex* const x[],
                                                      int                     incx,
                                                      const hipComplex* const y[],
                                                      int                     incy,
                                                      int                     batchCount,
                                                      hipComplex*             result);

HIPBLAS_EXPORT hipblasStatus_t hipblasCdotuBatched_v2(hipblasHandle_t         handle,
                                                      int                     n,
                                                      const hipComplex* const x[],
                                                      int                     incx,
                                                      const hipComplex* const y[],
                                                      int                     incy,
                                                      int                     batchCount,
                                                      hipComplex*             result);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdotcBatched_v2(hipblasHandle_t               handle,
                                                      int                           n,
                                                      const hipDoubleComplex* const x[],
                                                      int                           incx,
                                                      const hipDoubleComplex* const y[],
                                                      int                           incy,
                                                      int                           batchCount,
                                                      hipDoubleComplex*             result);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdotuBatched_v2(hipblasHandle_t               handle,
                                                      int                           n,
                                                      const hipDoubleComplex* const x[],
                                                      int                           incx,
                                                      const hipDoubleComplex* const y[],
                                                      int                           incy,
                                                      int                           batchCount,
                                                      hipDoubleComplex*             result);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasHdotBatched_64(hipblasHandle_t          handle,
                                                     int64_t                  n,
                                                     const hipblasHalf* const x[],
                                                     int64_t                  incx,
                                                     const hipblasHalf* const y[],
                                                     int64_t                  incy,
                                                     int64_t                  batchCount,
                                                     hipblasHalf*             result);

HIPBLAS_EXPORT hipblasStatus_t hipblasBfdotBatched_64(hipblasHandle_t              handle,
                                                      int64_t                      n,
                                                      const hipblasBfloat16* const x[],
                                                      int64_t                      incx,
                                                      const hipblasBfloat16* const y[],
                                                      int64_t                      incy,
                                                      int64_t                      batchCount,
                                                      hipblasBfloat16*             result);

HIPBLAS_EXPORT hipblasStatus_t hipblasSdotBatched_64(hipblasHandle_t    handle,
                                                     int64_t            n,
                                                     const float* const x[],
                                                     int64_t            incx,
                                                     const float* const y[],
                                                     int64_t            incy,
                                                     int64_t            batchCount,
                                                     float*             result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDdotBatched_64(hipblasHandle_t     handle,
                                                     int64_t             n,
                                                     const double* const x[],
                                                     int64_t             incx,
                                                     const double* const y[],
                                                     int64_t             incy,
                                                     int64_t             batchCount,
                                                     double*             result);

HIPBLAS_EXPORT hipblasStatus_t hipblasCdotcBatched_64(hipblasHandle_t             handle,
                                                      int64_t                     n,
                                                      const hipblasComplex* const x[],
                                                      int64_t                     incx,
                                                      const hipblasComplex* const y[],
                                                      int64_t                     incy,
                                                      int64_t                     batchCount,
                                                      hipblasComplex*             result);

HIPBLAS_EXPORT hipblasStatus_t hipblasCdotuBatched_64(hipblasHandle_t             handle,
                                                      int64_t                     n,
                                                      const hipblasComplex* const x[],
                                                      int64_t                     incx,
                                                      const hipblasComplex* const y[],
                                                      int64_t                     incy,
                                                      int64_t                     batchCount,
                                                      hipblasComplex*             result);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdotcBatched_64(hipblasHandle_t                   handle,
                                                      int64_t                           n,
                                                      const hipblasDoubleComplex* const x[],
                                                      int64_t                           incx,
                                                      const hipblasDoubleComplex* const y[],
                                                      int64_t                           incy,
                                                      int64_t                           batchCount,
                                                      hipblasDoubleComplex*             result);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdotuBatched_64(hipblasHandle_t                   handle,
                                                      int64_t                           n,
                                                      const hipblasDoubleComplex* const x[],
                                                      int64_t                           incx,
                                                      const hipblasDoubleComplex* const y[],
                                                      int64_t                           incy,
                                                      int64_t                           batchCount,
                                                      hipblasDoubleComplex*             result);

HIPBLAS_EXPORT hipblasStatus_t hipblasCdotcBatched_v2_64(hipblasHandle_t         handle,
                                                         int64_t                 n,
                                                         const hipComplex* const x[],
                                                         int64_t                 incx,
                                                         const hipComplex* const y[],
                                                         int64_t                 incy,
                                                         int64_t                 batchCount,
                                                         hipComplex*             result);

HIPBLAS_EXPORT hipblasStatus_t hipblasCdotuBatched_v2_64(hipblasHandle_t         handle,
                                                         int64_t                 n,
                                                         const hipComplex* const x[],
                                                         int64_t                 incx,
                                                         const hipComplex* const y[],
                                                         int64_t                 incy,
                                                         int64_t                 batchCount,
                                                         hipComplex*             result);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdotcBatched_v2_64(hipblasHandle_t               handle,
                                                         int64_t                       n,
                                                         const hipDoubleComplex* const x[],
                                                         int64_t                       incx,
                                                         const hipDoubleComplex* const y[],
                                                         int64_t                       incy,
                                                         int64_t                       batchCount,
                                                         hipDoubleComplex*             result);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdotuBatched_v2_64(hipblasHandle_t               handle,
                                                         int64_t                       n,
                                                         const hipDoubleComplex* const x[],
                                                         int64_t                       incx,
                                                         const hipDoubleComplex* const y[],
                                                         int64_t                       incy,
                                                         int64_t                       batchCount,
                                                         hipDoubleComplex*             result);
//! @}

/*! @{
    \brief BLAS Level 1 API

    \details
    dotStridedBatched(u)  performs a batch of dot products of vectors x and y

        result_i = x_i * y_i;

    dotcStridedBatched  performs a batch of dot products of the conjugate of complex vector x and complex vector y

        result_i = conjugate (x_i) * y_i;

    where (x_i, y_i) is the i-th instance of the batch.
    x_i and y_i are vectors, for i = 1, ..., batchCount

    - Supported precisions in rocBLAS : h,bf,s,d,c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    n         [int]
              the number of elements in each x_i and y_i.
    @param[in]
    x         device pointer to the first vector (x_1) in the batch.
    @param[in]
    incx      [int]
              specifies the increment for the elements of each x_i.
    @param[in]
    stridex     [hipblasStride]
                stride from the start of one vector (x_i) and the next one (x_i+1)
    @param[in]
    y         device pointer to the first vector (y_1) in the batch.
    @param[in]
    incy      [int]
              specifies the increment for the elements of each y_i.
    @param[in]
    stridey     [hipblasStride]
                stride from the start of one vector (y_i) and the next one (y_i+1)
    @param[in]
    batchCount [int]
                number of instances in the batch
    @param[inout]
    result
              device array or host array of batchCount size to store the dot products of each batch.
              return 0.0 for each element if n <= 0.

            ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasHdotStridedBatched(hipblasHandle_t    handle,
                                                         int                n,
                                                         const hipblasHalf* x,
                                                         int                incx,
                                                         hipblasStride      stridex,
                                                         const hipblasHalf* y,
                                                         int                incy,
                                                         hipblasStride      stridey,
                                                         int                batchCount,
                                                         hipblasHalf*       result);

HIPBLAS_EXPORT hipblasStatus_t hipblasBfdotStridedBatched(hipblasHandle_t        handle,
                                                          int                    n,
                                                          const hipblasBfloat16* x,
                                                          int                    incx,
                                                          hipblasStride          stridex,
                                                          const hipblasBfloat16* y,
                                                          int                    incy,
                                                          hipblasStride          stridey,
                                                          int                    batchCount,
                                                          hipblasBfloat16*       result);

HIPBLAS_EXPORT hipblasStatus_t hipblasSdotStridedBatched(hipblasHandle_t handle,
                                                         int             n,
                                                         const float*    x,
                                                         int             incx,
                                                         hipblasStride   stridex,
                                                         const float*    y,
                                                         int             incy,
                                                         hipblasStride   stridey,
                                                         int             batchCount,
                                                         float*          result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDdotStridedBatched(hipblasHandle_t handle,
                                                         int             n,
                                                         const double*   x,
                                                         int             incx,
                                                         hipblasStride   stridex,
                                                         const double*   y,
                                                         int             incy,
                                                         hipblasStride   stridey,
                                                         int             batchCount,
                                                         double*         result);

HIPBLAS_EXPORT hipblasStatus_t hipblasCdotcStridedBatched(hipblasHandle_t       handle,
                                                          int                   n,
                                                          const hipblasComplex* x,
                                                          int                   incx,
                                                          hipblasStride         stridex,
                                                          const hipblasComplex* y,
                                                          int                   incy,
                                                          hipblasStride         stridey,
                                                          int                   batchCount,
                                                          hipblasComplex*       result);

HIPBLAS_EXPORT hipblasStatus_t hipblasCdotuStridedBatched(hipblasHandle_t       handle,
                                                          int                   n,
                                                          const hipblasComplex* x,
                                                          int                   incx,
                                                          hipblasStride         stridex,
                                                          const hipblasComplex* y,
                                                          int                   incy,
                                                          hipblasStride         stridey,
                                                          int                   batchCount,
                                                          hipblasComplex*       result);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdotcStridedBatched(hipblasHandle_t             handle,
                                                          int                         n,
                                                          const hipblasDoubleComplex* x,
                                                          int                         incx,
                                                          hipblasStride               stridex,
                                                          const hipblasDoubleComplex* y,
                                                          int                         incy,
                                                          hipblasStride               stridey,
                                                          int                         batchCount,
                                                          hipblasDoubleComplex*       result);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdotuStridedBatched(hipblasHandle_t             handle,
                                                          int                         n,
                                                          const hipblasDoubleComplex* x,
                                                          int                         incx,
                                                          hipblasStride               stridex,
                                                          const hipblasDoubleComplex* y,
                                                          int                         incy,
                                                          hipblasStride               stridey,
                                                          int                         batchCount,
                                                          hipblasDoubleComplex*       result);

HIPBLAS_EXPORT hipblasStatus_t hipblasCdotcStridedBatched_v2(hipblasHandle_t   handle,
                                                             int               n,
                                                             const hipComplex* x,
                                                             int               incx,
                                                             hipblasStride     stridex,
                                                             const hipComplex* y,
                                                             int               incy,
                                                             hipblasStride     stridey,
                                                             int               batchCount,
                                                             hipComplex*       result);

HIPBLAS_EXPORT hipblasStatus_t hipblasCdotuStridedBatched_v2(hipblasHandle_t   handle,
                                                             int               n,
                                                             const hipComplex* x,
                                                             int               incx,
                                                             hipblasStride     stridex,
                                                             const hipComplex* y,
                                                             int               incy,
                                                             hipblasStride     stridey,
                                                             int               batchCount,
                                                             hipComplex*       result);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdotcStridedBatched_v2(hipblasHandle_t         handle,
                                                             int                     n,
                                                             const hipDoubleComplex* x,
                                                             int                     incx,
                                                             hipblasStride           stridex,
                                                             const hipDoubleComplex* y,
                                                             int                     incy,
                                                             hipblasStride           stridey,
                                                             int                     batchCount,
                                                             hipDoubleComplex*       result);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdotuStridedBatched_v2(hipblasHandle_t         handle,
                                                             int                     n,
                                                             const hipDoubleComplex* x,
                                                             int                     incx,
                                                             hipblasStride           stridex,
                                                             const hipDoubleComplex* y,
                                                             int                     incy,
                                                             hipblasStride           stridey,
                                                             int                     batchCount,
                                                             hipDoubleComplex*       result);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasHdotStridedBatched_64(hipblasHandle_t    handle,
                                                            int64_t            n,
                                                            const hipblasHalf* x,
                                                            int64_t            incx,
                                                            hipblasStride      stridex,
                                                            const hipblasHalf* y,
                                                            int64_t            incy,
                                                            hipblasStride      stridey,
                                                            int64_t            batchCount,
                                                            hipblasHalf*       result);

HIPBLAS_EXPORT hipblasStatus_t hipblasBfdotStridedBatched_64(hipblasHandle_t        handle,
                                                             int64_t                n,
                                                             const hipblasBfloat16* x,
                                                             int64_t                incx,
                                                             hipblasStride          stridex,
                                                             const hipblasBfloat16* y,
                                                             int64_t                incy,
                                                             hipblasStride          stridey,
                                                             int64_t                batchCount,
                                                             hipblasBfloat16*       result);

HIPBLAS_EXPORT hipblasStatus_t hipblasSdotStridedBatched_64(hipblasHandle_t handle,
                                                            int64_t         n,
                                                            const float*    x,
                                                            int64_t         incx,
                                                            hipblasStride   stridex,
                                                            const float*    y,
                                                            int64_t         incy,
                                                            hipblasStride   stridey,
                                                            int64_t         batchCount,
                                                            float*          result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDdotStridedBatched_64(hipblasHandle_t handle,
                                                            int64_t         n,
                                                            const double*   x,
                                                            int64_t         incx,
                                                            hipblasStride   stridex,
                                                            const double*   y,
                                                            int64_t         incy,
                                                            hipblasStride   stridey,
                                                            int64_t         batchCount,
                                                            double*         result);

HIPBLAS_EXPORT hipblasStatus_t hipblasCdotcStridedBatched_64(hipblasHandle_t       handle,
                                                             int64_t               n,
                                                             const hipblasComplex* x,
                                                             int64_t               incx,
                                                             hipblasStride         stridex,
                                                             const hipblasComplex* y,
                                                             int64_t               incy,
                                                             hipblasStride         stridey,
                                                             int64_t               batchCount,
                                                             hipblasComplex*       result);

HIPBLAS_EXPORT hipblasStatus_t hipblasCdotuStridedBatched_64(hipblasHandle_t       handle,
                                                             int64_t               n,
                                                             const hipblasComplex* x,
                                                             int64_t               incx,
                                                             hipblasStride         stridex,
                                                             const hipblasComplex* y,
                                                             int64_t               incy,
                                                             hipblasStride         stridey,
                                                             int64_t               batchCount,
                                                             hipblasComplex*       result);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdotcStridedBatched_64(hipblasHandle_t             handle,
                                                             int64_t                     n,
                                                             const hipblasDoubleComplex* x,
                                                             int64_t                     incx,
                                                             hipblasStride               stridex,
                                                             const hipblasDoubleComplex* y,
                                                             int64_t                     incy,
                                                             hipblasStride               stridey,
                                                             int64_t                     batchCount,
                                                             hipblasDoubleComplex*       result);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdotuStridedBatched_64(hipblasHandle_t             handle,
                                                             int64_t                     n,
                                                             const hipblasDoubleComplex* x,
                                                             int64_t                     incx,
                                                             hipblasStride               stridex,
                                                             const hipblasDoubleComplex* y,
                                                             int64_t                     incy,
                                                             hipblasStride               stridey,
                                                             int64_t                     batchCount,
                                                             hipblasDoubleComplex*       result);

HIPBLAS_EXPORT hipblasStatus_t hipblasCdotcStridedBatched_v2_64(hipblasHandle_t   handle,
                                                                int64_t           n,
                                                                const hipComplex* x,
                                                                int64_t           incx,
                                                                hipblasStride     stridex,
                                                                const hipComplex* y,
                                                                int64_t           incy,
                                                                hipblasStride     stridey,
                                                                int64_t           batchCount,
                                                                hipComplex*       result);

HIPBLAS_EXPORT hipblasStatus_t hipblasCdotuStridedBatched_v2_64(hipblasHandle_t   handle,
                                                                int64_t           n,
                                                                const hipComplex* x,
                                                                int64_t           incx,
                                                                hipblasStride     stridex,
                                                                const hipComplex* y,
                                                                int64_t           incy,
                                                                hipblasStride     stridey,
                                                                int64_t           batchCount,
                                                                hipComplex*       result);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdotcStridedBatched_v2_64(hipblasHandle_t         handle,
                                                                int64_t                 n,
                                                                const hipDoubleComplex* x,
                                                                int64_t                 incx,
                                                                hipblasStride           stridex,
                                                                const hipDoubleComplex* y,
                                                                int64_t                 incy,
                                                                hipblasStride           stridey,
                                                                int64_t                 batchCount,
                                                                hipDoubleComplex*       result);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdotuStridedBatched_v2_64(hipblasHandle_t         handle,
                                                                int64_t                 n,
                                                                const hipDoubleComplex* x,
                                                                int64_t                 incx,
                                                                hipblasStride           stridex,
                                                                const hipDoubleComplex* y,
                                                                int64_t                 incy,
                                                                hipblasStride           stridey,
                                                                int64_t                 batchCount,
                                                                hipDoubleComplex*       result);
//! @}

/*! @{
    \brief BLAS Level 1 API

    \details
    nrm2 computes the euclidean norm of a real or complex vector

              result := sqrt( x'*x ) for real vectors
              result := sqrt( x**H*x ) for complex vectors

    - Supported precisions in rocBLAS : s,d,c,z,sc,dz
    - Supported precisions in cuBLAS  : s,d,sc,dz

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    n         [int]
              the number of elements in x.
    @param[in]
    x         device pointer storing vector x.
    @param[in]
    incx      [int]
              specifies the increment for the elements of y.
    @param[inout]
    result
              device pointer or host pointer to store the nrm2 product.
              return is 0.0 if n, incx<=0.

            ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t
    hipblasSnrm2(hipblasHandle_t handle, int n, const float* x, int incx, float* result);

HIPBLAS_EXPORT hipblasStatus_t
    hipblasDnrm2(hipblasHandle_t handle, int n, const double* x, int incx, double* result);

HIPBLAS_EXPORT hipblasStatus_t
    hipblasScnrm2(hipblasHandle_t handle, int n, const hipblasComplex* x, int incx, float* result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDznrm2(
    hipblasHandle_t handle, int n, const hipblasDoubleComplex* x, int incx, double* result);

HIPBLAS_EXPORT hipblasStatus_t
    hipblasScnrm2_v2(hipblasHandle_t handle, int n, const hipComplex* x, int incx, float* result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDznrm2_v2(
    hipblasHandle_t handle, int n, const hipDoubleComplex* x, int incx, double* result);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t
    hipblasSnrm2_64(hipblasHandle_t handle, int64_t n, const float* x, int64_t incx, float* result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDnrm2_64(
    hipblasHandle_t handle, int64_t n, const double* x, int64_t incx, double* result);

HIPBLAS_EXPORT hipblasStatus_t hipblasScnrm2_64(
    hipblasHandle_t handle, int64_t n, const hipblasComplex* x, int64_t incx, float* result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDznrm2_64(
    hipblasHandle_t handle, int64_t n, const hipblasDoubleComplex* x, int64_t incx, double* result);

HIPBLAS_EXPORT hipblasStatus_t hipblasScnrm2_v2_64(
    hipblasHandle_t handle, int64_t n, const hipComplex* x, int64_t incx, float* result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDznrm2_v2_64(
    hipblasHandle_t handle, int64_t n, const hipDoubleComplex* x, int64_t incx, double* result);

//! @}

/*! @{
    \brief BLAS Level 1 API

    \details
    nrm2Batched computes the euclidean norm over a batch of real or complex vectors

              result := sqrt( x_i'*x_i ) for real vectors x, for i = 1, ..., batchCount
              result := sqrt( x_i**H*x_i ) for complex vectors x, for i = 1, ..., batchCount

    - Supported precisions in rocBLAS : s,d,c,z,sc,dz
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    n         [int]
              number of elements in each x_i.
    @param[in]
    x         device array of device pointers storing each vector x_i.
    @param[in]
    incx      [int]
              specifies the increment for the elements of each x_i. incx must be > 0.
    @param[in]
    batchCount [int]
              number of instances in the batch
    @param[out]
    result
              device pointer or host pointer to array of batchCount size for nrm2 results.
              return is 0.0 for each element if n <= 0, incx<=0.

            ********************************************************************/
HIPBLAS_EXPORT hipblasStatus_t hipblasSnrm2Batched(
    hipblasHandle_t handle, int n, const float* const x[], int incx, int batchCount, float* result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDnrm2Batched(hipblasHandle_t     handle,
                                                   int                 n,
                                                   const double* const x[],
                                                   int                 incx,
                                                   int                 batchCount,
                                                   double*             result);

HIPBLAS_EXPORT hipblasStatus_t hipblasScnrm2Batched(hipblasHandle_t             handle,
                                                    int                         n,
                                                    const hipblasComplex* const x[],
                                                    int                         incx,
                                                    int                         batchCount,
                                                    float*                      result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDznrm2Batched(hipblasHandle_t                   handle,
                                                    int                               n,
                                                    const hipblasDoubleComplex* const x[],
                                                    int                               incx,
                                                    int                               batchCount,
                                                    double*                           result);

HIPBLAS_EXPORT hipblasStatus_t hipblasScnrm2Batched_v2(hipblasHandle_t         handle,
                                                       int                     n,
                                                       const hipComplex* const x[],
                                                       int                     incx,
                                                       int                     batchCount,
                                                       float*                  result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDznrm2Batched_v2(hipblasHandle_t               handle,
                                                       int                           n,
                                                       const hipDoubleComplex* const x[],
                                                       int                           incx,
                                                       int                           batchCount,
                                                       double*                       result);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSnrm2Batched_64(hipblasHandle_t    handle,
                                                      int64_t            n,
                                                      const float* const x[],
                                                      int64_t            incx,
                                                      int64_t            batchCount,
                                                      float*             result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDnrm2Batched_64(hipblasHandle_t     handle,
                                                      int64_t             n,
                                                      const double* const x[],
                                                      int64_t             incx,
                                                      int64_t             batchCount,
                                                      double*             result);

HIPBLAS_EXPORT hipblasStatus_t hipblasScnrm2Batched_64(hipblasHandle_t             handle,
                                                       int64_t                     n,
                                                       const hipblasComplex* const x[],
                                                       int64_t                     incx,
                                                       int64_t                     batchCount,
                                                       float*                      result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDznrm2Batched_64(hipblasHandle_t                   handle,
                                                       int64_t                           n,
                                                       const hipblasDoubleComplex* const x[],
                                                       int64_t                           incx,
                                                       int64_t                           batchCount,
                                                       double*                           result);

HIPBLAS_EXPORT hipblasStatus_t hipblasScnrm2Batched_v2_64(hipblasHandle_t         handle,
                                                          int64_t                 n,
                                                          const hipComplex* const x[],
                                                          int64_t                 incx,
                                                          int64_t                 batchCount,
                                                          float*                  result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDznrm2Batched_v2_64(hipblasHandle_t               handle,
                                                          int64_t                       n,
                                                          const hipDoubleComplex* const x[],
                                                          int64_t                       incx,
                                                          int64_t                       batchCount,
                                                          double*                       result);

//! @}

/*! @{
    \brief BLAS Level 1 API

    \details
    nrm2StridedBatched computes the euclidean norm over a batch of real or complex vectors

              := sqrt( x_i'*x_i ) for real vectors x, for i = 1, ..., batchCount
              := sqrt( x_i**H*x_i ) for complex vectors, for i = 1, ..., batchCount

    - Supported precisions in rocBLAS : s,d,c,z,sc,dz
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    n         [int]
              number of elements in each x_i.
    @param[in]
    x         device pointer to the first vector x_1.
    @param[in]
    incx      [int]
              specifies the increment for the elements of each x_i. incx must be > 0.
    @param[in]
    stridex   [hipblasStride]
              stride from the start of one vector (x_i) and the next one (x_i+1).
              There are no restrictions placed on stride_x, however the user should
              take care to ensure that stride_x is of appropriate size, for a typical
              case this means stride_x >= n * incx.
    @param[in]
    batchCount [int]
              number of instances in the batch
    @param[out]
    result
              device pointer or host pointer to array for storing contiguous batchCount results.
              return is 0.0 for each element if n <= 0, incx<=0.

            ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSnrm2StridedBatched(hipblasHandle_t handle,
                                                          int             n,
                                                          const float*    x,
                                                          int             incx,
                                                          hipblasStride   stridex,
                                                          int             batchCount,
                                                          float*          result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDnrm2StridedBatched(hipblasHandle_t handle,
                                                          int             n,
                                                          const double*   x,
                                                          int             incx,
                                                          hipblasStride   stridex,
                                                          int             batchCount,
                                                          double*         result);

HIPBLAS_EXPORT hipblasStatus_t hipblasScnrm2StridedBatched(hipblasHandle_t       handle,
                                                           int                   n,
                                                           const hipblasComplex* x,
                                                           int                   incx,
                                                           hipblasStride         stridex,
                                                           int                   batchCount,
                                                           float*                result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDznrm2StridedBatched(hipblasHandle_t             handle,
                                                           int                         n,
                                                           const hipblasDoubleComplex* x,
                                                           int                         incx,
                                                           hipblasStride               stridex,
                                                           int                         batchCount,
                                                           double*                     result);

HIPBLAS_EXPORT hipblasStatus_t hipblasScnrm2StridedBatched_v2(hipblasHandle_t   handle,
                                                              int               n,
                                                              const hipComplex* x,
                                                              int               incx,
                                                              hipblasStride     stridex,
                                                              int               batchCount,
                                                              float*            result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDznrm2StridedBatched_v2(hipblasHandle_t         handle,
                                                              int                     n,
                                                              const hipDoubleComplex* x,
                                                              int                     incx,
                                                              hipblasStride           stridex,
                                                              int                     batchCount,
                                                              double*                 result);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSnrm2StridedBatched_64(hipblasHandle_t handle,
                                                             int64_t         n,
                                                             const float*    x,
                                                             int64_t         incx,
                                                             hipblasStride   stridex,
                                                             int64_t         batchCount,
                                                             float*          result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDnrm2StridedBatched_64(hipblasHandle_t handle,
                                                             int64_t         n,
                                                             const double*   x,
                                                             int64_t         incx,
                                                             hipblasStride   stridex,
                                                             int64_t         batchCount,
                                                             double*         result);

HIPBLAS_EXPORT hipblasStatus_t hipblasScnrm2StridedBatched_64(hipblasHandle_t       handle,
                                                              int64_t               n,
                                                              const hipblasComplex* x,
                                                              int64_t               incx,
                                                              hipblasStride         stridex,
                                                              int64_t               batchCount,
                                                              float*                result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDznrm2StridedBatched_64(hipblasHandle_t             handle,
                                                              int64_t                     n,
                                                              const hipblasDoubleComplex* x,
                                                              int64_t                     incx,
                                                              hipblasStride               stridex,
                                                              int64_t batchCount,
                                                              double* result);

HIPBLAS_EXPORT hipblasStatus_t hipblasScnrm2StridedBatched_v2_64(hipblasHandle_t   handle,
                                                                 int64_t           n,
                                                                 const hipComplex* x,
                                                                 int64_t           incx,
                                                                 hipblasStride     stridex,
                                                                 int64_t           batchCount,
                                                                 float*            result);

HIPBLAS_EXPORT hipblasStatus_t hipblasDznrm2StridedBatched_v2_64(hipblasHandle_t         handle,
                                                                 int64_t                 n,
                                                                 const hipDoubleComplex* x,
                                                                 int64_t                 incx,
                                                                 hipblasStride           stridex,
                                                                 int64_t                 batchCount,
                                                                 double*                 result);
//! @}

/*! @{
    \brief BLAS Level 1 API

    \details
    rot applies the Givens rotation matrix defined by c=cos(alpha) and s=sin(alpha) to vectors x and y.
        Scalars c and s may be stored in either host or device memory, location is specified by calling hipblasSetPointerMode.

    - Supported precisions in rocBLAS : s,d,c,z,sc,dz
    - Supported precisions in cuBLAS  : s,d,c,z,cs,zd

    @param[in]
    handle  [hipblasHandle_t]
            handle to the hipblas library context queue.
    @param[in]
    n       [int]
            number of elements in the x and y vectors.
    @param[inout]
    x       device pointer storing vector x.
    @param[in]
    incx    [int]
            specifies the increment between elements of x.
    @param[inout]
    y       device pointer storing vector y.
    @param[in]
    incy    [int]
            specifies the increment between elements of y.
    @param[in]
    c       device pointer or host pointer storing scalar cosine component of the rotation matrix.
    @param[in]
    s       device pointer or host pointer storing scalar sine component of the rotation matrix.

            ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSrot(hipblasHandle_t handle,
                                           int             n,
                                           float*          x,
                                           int             incx,
                                           float*          y,
                                           int             incy,
                                           const float*    c,
                                           const float*    s);

HIPBLAS_EXPORT hipblasStatus_t hipblasDrot(hipblasHandle_t handle,
                                           int             n,
                                           double*         x,
                                           int             incx,
                                           double*         y,
                                           int             incy,
                                           const double*   c,
                                           const double*   s);

HIPBLAS_EXPORT hipblasStatus_t hipblasCrot(hipblasHandle_t       handle,
                                           int                   n,
                                           hipblasComplex*       x,
                                           int                   incx,
                                           hipblasComplex*       y,
                                           int                   incy,
                                           const float*          c,
                                           const hipblasComplex* s);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsrot(hipblasHandle_t handle,
                                            int             n,
                                            hipblasComplex* x,
                                            int             incx,
                                            hipblasComplex* y,
                                            int             incy,
                                            const float*    c,
                                            const float*    s);

HIPBLAS_EXPORT hipblasStatus_t hipblasZrot(hipblasHandle_t             handle,
                                           int                         n,
                                           hipblasDoubleComplex*       x,
                                           int                         incx,
                                           hipblasDoubleComplex*       y,
                                           int                         incy,
                                           const double*               c,
                                           const hipblasDoubleComplex* s);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdrot(hipblasHandle_t       handle,
                                            int                   n,
                                            hipblasDoubleComplex* x,
                                            int                   incx,
                                            hipblasDoubleComplex* y,
                                            int                   incy,
                                            const double*         c,
                                            const double*         s);

HIPBLAS_EXPORT hipblasStatus_t hipblasCrot_v2(hipblasHandle_t   handle,
                                              int               n,
                                              hipComplex*       x,
                                              int               incx,
                                              hipComplex*       y,
                                              int               incy,
                                              const float*      c,
                                              const hipComplex* s);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsrot_v2(hipblasHandle_t handle,
                                               int             n,
                                               hipComplex*     x,
                                               int             incx,
                                               hipComplex*     y,
                                               int             incy,
                                               const float*    c,
                                               const float*    s);

HIPBLAS_EXPORT hipblasStatus_t hipblasZrot_v2(hipblasHandle_t         handle,
                                              int                     n,
                                              hipDoubleComplex*       x,
                                              int                     incx,
                                              hipDoubleComplex*       y,
                                              int                     incy,
                                              const double*           c,
                                              const hipDoubleComplex* s);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdrot_v2(hipblasHandle_t   handle,
                                               int               n,
                                               hipDoubleComplex* x,
                                               int               incx,
                                               hipDoubleComplex* y,
                                               int               incy,
                                               const double*     c,
                                               const double*     s);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSrot_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              float*          x,
                                              int64_t         incx,
                                              float*          y,
                                              int64_t         incy,
                                              const float*    c,
                                              const float*    s);

HIPBLAS_EXPORT hipblasStatus_t hipblasDrot_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              double*         x,
                                              int64_t         incx,
                                              double*         y,
                                              int64_t         incy,
                                              const double*   c,
                                              const double*   s);

HIPBLAS_EXPORT hipblasStatus_t hipblasCrot_64(hipblasHandle_t       handle,
                                              int64_t               n,
                                              hipblasComplex*       x,
                                              int64_t               incx,
                                              hipblasComplex*       y,
                                              int64_t               incy,
                                              const float*          c,
                                              const hipblasComplex* s);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsrot_64(hipblasHandle_t handle,
                                               int64_t         n,
                                               hipblasComplex* x,
                                               int64_t         incx,
                                               hipblasComplex* y,
                                               int64_t         incy,
                                               const float*    c,
                                               const float*    s);

HIPBLAS_EXPORT hipblasStatus_t hipblasZrot_64(hipblasHandle_t             handle,
                                              int64_t                     n,
                                              hipblasDoubleComplex*       x,
                                              int64_t                     incx,
                                              hipblasDoubleComplex*       y,
                                              int64_t                     incy,
                                              const double*               c,
                                              const hipblasDoubleComplex* s);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdrot_64(hipblasHandle_t       handle,
                                               int64_t               n,
                                               hipblasDoubleComplex* x,
                                               int64_t               incx,
                                               hipblasDoubleComplex* y,
                                               int64_t               incy,
                                               const double*         c,
                                               const double*         s);

HIPBLAS_EXPORT hipblasStatus_t hipblasCrot_v2_64(hipblasHandle_t   handle,
                                                 int64_t           n,
                                                 hipComplex*       x,
                                                 int64_t           incx,
                                                 hipComplex*       y,
                                                 int64_t           incy,
                                                 const float*      c,
                                                 const hipComplex* s);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsrot_v2_64(hipblasHandle_t handle,
                                                  int64_t         n,
                                                  hipComplex*     x,
                                                  int64_t         incx,
                                                  hipComplex*     y,
                                                  int64_t         incy,
                                                  const float*    c,
                                                  const float*    s);

HIPBLAS_EXPORT hipblasStatus_t hipblasZrot_v2_64(hipblasHandle_t         handle,
                                                 int64_t                 n,
                                                 hipDoubleComplex*       x,
                                                 int64_t                 incx,
                                                 hipDoubleComplex*       y,
                                                 int64_t                 incy,
                                                 const double*           c,
                                                 const hipDoubleComplex* s);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdrot_v2_64(hipblasHandle_t   handle,
                                                  int64_t           n,
                                                  hipDoubleComplex* x,
                                                  int64_t           incx,
                                                  hipDoubleComplex* y,
                                                  int64_t           incy,
                                                  const double*     c,
                                                  const double*     s);
//! @}

/*! @{
    \brief BLAS Level 1 API

    \details
    rotBatched applies the Givens rotation matrix defined by c=cos(alpha) and s=sin(alpha) to batched vectors x_i and y_i, for i = 1, ..., batchCount.
        Scalars c and s may be stored in either host or device memory, location is specified by calling hipblasSetPointerMode.

    - Supported precisions in rocBLAS : s,d,sc,dz
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle  [hipblasHandle_t]
            handle to the hipblas library context queue.
    @param[in]
    n       [int]
            number of elements in each x_i and y_i vectors.
    @param[inout]
    x       device array of device pointers storing each vector x_i.
    @param[in]
    incx    [int]
            specifies the increment between elements of each x_i.
    @param[inout]
    y       device array of device pointers storing each vector y_i.
    @param[in]
    incy    [int]
            specifies the increment between elements of each y_i.
    @param[in]
    c       device pointer or host pointer to scalar cosine component of the rotation matrix.
    @param[in]
    s       device pointer or host pointer to scalar sine component of the rotation matrix.
    @param[in]
    batchCount [int]
                the number of x and y arrays, i.e. the number of batches.

            ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSrotBatched(hipblasHandle_t handle,
                                                  int             n,
                                                  float* const    x[],
                                                  int             incx,
                                                  float* const    y[],
                                                  int             incy,
                                                  const float*    c,
                                                  const float*    s,
                                                  int             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDrotBatched(hipblasHandle_t handle,
                                                  int             n,
                                                  double* const   x[],
                                                  int             incx,
                                                  double* const   y[],
                                                  int             incy,
                                                  const double*   c,
                                                  const double*   s,
                                                  int             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCrotBatched(hipblasHandle_t       handle,
                                                  int                   n,
                                                  hipblasComplex* const x[],
                                                  int                   incx,
                                                  hipblasComplex* const y[],
                                                  int                   incy,
                                                  const float*          c,
                                                  const hipblasComplex* s,
                                                  int                   batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsrotBatched(hipblasHandle_t       handle,
                                                   int                   n,
                                                   hipblasComplex* const x[],
                                                   int                   incx,
                                                   hipblasComplex* const y[],
                                                   int                   incy,
                                                   const float*          c,
                                                   const float*          s,
                                                   int                   batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZrotBatched(hipblasHandle_t             handle,
                                                  int                         n,
                                                  hipblasDoubleComplex* const x[],
                                                  int                         incx,
                                                  hipblasDoubleComplex* const y[],
                                                  int                         incy,
                                                  const double*               c,
                                                  const hipblasDoubleComplex* s,
                                                  int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdrotBatched(hipblasHandle_t             handle,
                                                   int                         n,
                                                   hipblasDoubleComplex* const x[],
                                                   int                         incx,
                                                   hipblasDoubleComplex* const y[],
                                                   int                         incy,
                                                   const double*               c,
                                                   const double*               s,
                                                   int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCrotBatched_v2(hipblasHandle_t   handle,
                                                     int               n,
                                                     hipComplex* const x[],
                                                     int               incx,
                                                     hipComplex* const y[],
                                                     int               incy,
                                                     const float*      c,
                                                     const hipComplex* s,
                                                     int               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsrotBatched_v2(hipblasHandle_t   handle,
                                                      int               n,
                                                      hipComplex* const x[],
                                                      int               incx,
                                                      hipComplex* const y[],
                                                      int               incy,
                                                      const float*      c,
                                                      const float*      s,
                                                      int               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZrotBatched_v2(hipblasHandle_t         handle,
                                                     int                     n,
                                                     hipDoubleComplex* const x[],
                                                     int                     incx,
                                                     hipDoubleComplex* const y[],
                                                     int                     incy,
                                                     const double*           c,
                                                     const hipDoubleComplex* s,
                                                     int                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdrotBatched_v2(hipblasHandle_t         handle,
                                                      int                     n,
                                                      hipDoubleComplex* const x[],
                                                      int                     incx,
                                                      hipDoubleComplex* const y[],
                                                      int                     incy,
                                                      const double*           c,
                                                      const double*           s,
                                                      int                     batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSrotBatched_64(hipblasHandle_t handle,
                                                     int64_t         n,
                                                     float* const    x[],
                                                     int64_t         incx,
                                                     float* const    y[],
                                                     int64_t         incy,
                                                     const float*    c,
                                                     const float*    s,
                                                     int64_t         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDrotBatched_64(hipblasHandle_t handle,
                                                     int64_t         n,
                                                     double* const   x[],
                                                     int64_t         incx,
                                                     double* const   y[],
                                                     int64_t         incy,
                                                     const double*   c,
                                                     const double*   s,
                                                     int64_t         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCrotBatched_64(hipblasHandle_t       handle,
                                                     int64_t               n,
                                                     hipblasComplex* const x[],
                                                     int64_t               incx,
                                                     hipblasComplex* const y[],
                                                     int64_t               incy,
                                                     const float*          c,
                                                     const hipblasComplex* s,
                                                     int64_t               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsrotBatched_64(hipblasHandle_t       handle,
                                                      int64_t               n,
                                                      hipblasComplex* const x[],
                                                      int64_t               incx,
                                                      hipblasComplex* const y[],
                                                      int64_t               incy,
                                                      const float*          c,
                                                      const float*          s,
                                                      int64_t               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZrotBatched_64(hipblasHandle_t             handle,
                                                     int64_t                     n,
                                                     hipblasDoubleComplex* const x[],
                                                     int64_t                     incx,
                                                     hipblasDoubleComplex* const y[],
                                                     int64_t                     incy,
                                                     const double*               c,
                                                     const hipblasDoubleComplex* s,
                                                     int64_t                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdrotBatched_64(hipblasHandle_t             handle,
                                                      int64_t                     n,
                                                      hipblasDoubleComplex* const x[],
                                                      int64_t                     incx,
                                                      hipblasDoubleComplex* const y[],
                                                      int64_t                     incy,
                                                      const double*               c,
                                                      const double*               s,
                                                      int64_t                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCrotBatched_v2_64(hipblasHandle_t   handle,
                                                        int64_t           n,
                                                        hipComplex* const x[],
                                                        int64_t           incx,
                                                        hipComplex* const y[],
                                                        int64_t           incy,
                                                        const float*      c,
                                                        const hipComplex* s,
                                                        int64_t           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsrotBatched_v2_64(hipblasHandle_t   handle,
                                                         int64_t           n,
                                                         hipComplex* const x[],
                                                         int64_t           incx,
                                                         hipComplex* const y[],
                                                         int64_t           incy,
                                                         const float*      c,
                                                         const float*      s,
                                                         int64_t           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZrotBatched_v2_64(hipblasHandle_t         handle,
                                                        int64_t                 n,
                                                        hipDoubleComplex* const x[],
                                                        int64_t                 incx,
                                                        hipDoubleComplex* const y[],
                                                        int64_t                 incy,
                                                        const double*           c,
                                                        const hipDoubleComplex* s,
                                                        int64_t                 batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdrotBatched_v2_64(hipblasHandle_t         handle,
                                                         int64_t                 n,
                                                         hipDoubleComplex* const x[],
                                                         int64_t                 incx,
                                                         hipDoubleComplex* const y[],
                                                         int64_t                 incy,
                                                         const double*           c,
                                                         const double*           s,
                                                         int64_t                 batchCount);
//! @}

/*! @{
    \brief BLAS Level 1 API

    \details
    rotStridedBatched applies the Givens rotation matrix defined by c=cos(alpha) and s=sin(alpha) to strided batched vectors x_i and y_i, for i = 1, ..., batchCount.
        Scalars c and s may be stored in either host or device memory, location is specified by calling hipblasSetPointerMode.

    - Supported precisions in rocBLAS : s,d,sc,dz
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle  [hipblasHandle_t]
            handle to the hipblas library context queue.
    @param[in]
    n       [int]
            number of elements in each x_i and y_i vectors.
    @param[inout]
    x       device pointer to the first vector x_1.
    @param[in]
    incx    [int]
            specifies the increment between elements of each x_i.
    @param[in]
    stridex [hipblasStride]
             specifies the increment from the beginning of x_i to the beginning of x_(i+1)
    @param[inout]
    y       device pointer to the first vector y_1.
    @param[in]
    incy    [int]
            specifies the increment between elements of each y_i.
    @param[in]
    stridey  [hipblasStride]
             specifies the increment from the beginning of y_i to the beginning of y_(i+1)
    @param[in]
    c       device pointer or host pointer to scalar cosine component of the rotation matrix.
    @param[in]
    s       device pointer or host pointer to scalar sine component of the rotation matrix.
    @param[in]
    batchCount [int]
            the number of x and y arrays, i.e. the number of batches.

            ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSrotStridedBatched(hipblasHandle_t handle,
                                                         int             n,
                                                         float*          x,
                                                         int             incx,
                                                         hipblasStride   stridex,
                                                         float*          y,
                                                         int             incy,
                                                         hipblasStride   stridey,
                                                         const float*    c,
                                                         const float*    s,
                                                         int             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDrotStridedBatched(hipblasHandle_t handle,
                                                         int             n,
                                                         double*         x,
                                                         int             incx,
                                                         hipblasStride   stridex,
                                                         double*         y,
                                                         int             incy,
                                                         hipblasStride   stridey,
                                                         const double*   c,
                                                         const double*   s,
                                                         int             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCrotStridedBatched(hipblasHandle_t       handle,
                                                         int                   n,
                                                         hipblasComplex*       x,
                                                         int                   incx,
                                                         hipblasStride         stridex,
                                                         hipblasComplex*       y,
                                                         int                   incy,
                                                         hipblasStride         stridey,
                                                         const float*          c,
                                                         const hipblasComplex* s,
                                                         int                   batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsrotStridedBatched(hipblasHandle_t handle,
                                                          int             n,
                                                          hipblasComplex* x,
                                                          int             incx,
                                                          hipblasStride   stridex,
                                                          hipblasComplex* y,
                                                          int             incy,
                                                          hipblasStride   stridey,
                                                          const float*    c,
                                                          const float*    s,
                                                          int             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZrotStridedBatched(hipblasHandle_t             handle,
                                                         int                         n,
                                                         hipblasDoubleComplex*       x,
                                                         int                         incx,
                                                         hipblasStride               stridex,
                                                         hipblasDoubleComplex*       y,
                                                         int                         incy,
                                                         hipblasStride               stridey,
                                                         const double*               c,
                                                         const hipblasDoubleComplex* s,
                                                         int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdrotStridedBatched(hipblasHandle_t       handle,
                                                          int                   n,
                                                          hipblasDoubleComplex* x,
                                                          int                   incx,
                                                          hipblasStride         stridex,
                                                          hipblasDoubleComplex* y,
                                                          int                   incy,
                                                          hipblasStride         stridey,
                                                          const double*         c,
                                                          const double*         s,
                                                          int                   batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCrotStridedBatched_v2(hipblasHandle_t   handle,
                                                            int               n,
                                                            hipComplex*       x,
                                                            int               incx,
                                                            hipblasStride     stridex,
                                                            hipComplex*       y,
                                                            int               incy,
                                                            hipblasStride     stridey,
                                                            const float*      c,
                                                            const hipComplex* s,
                                                            int               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsrotStridedBatched_v2(hipblasHandle_t handle,
                                                             int             n,
                                                             hipComplex*     x,
                                                             int             incx,
                                                             hipblasStride   stridex,
                                                             hipComplex*     y,
                                                             int             incy,
                                                             hipblasStride   stridey,
                                                             const float*    c,
                                                             const float*    s,
                                                             int             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZrotStridedBatched_v2(hipblasHandle_t         handle,
                                                            int                     n,
                                                            hipDoubleComplex*       x,
                                                            int                     incx,
                                                            hipblasStride           stridex,
                                                            hipDoubleComplex*       y,
                                                            int                     incy,
                                                            hipblasStride           stridey,
                                                            const double*           c,
                                                            const hipDoubleComplex* s,
                                                            int                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdrotStridedBatched_v2(hipblasHandle_t   handle,
                                                             int               n,
                                                             hipDoubleComplex* x,
                                                             int               incx,
                                                             hipblasStride     stridex,
                                                             hipDoubleComplex* y,
                                                             int               incy,
                                                             hipblasStride     stridey,
                                                             const double*     c,
                                                             const double*     s,
                                                             int               batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSrotStridedBatched_64(hipblasHandle_t handle,
                                                            int64_t         n,
                                                            float*          x,
                                                            int64_t         incx,
                                                            hipblasStride   stridex,
                                                            float*          y,
                                                            int64_t         incy,
                                                            hipblasStride   stridey,
                                                            const float*    c,
                                                            const float*    s,
                                                            int64_t         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDrotStridedBatched_64(hipblasHandle_t handle,
                                                            int64_t         n,
                                                            double*         x,
                                                            int64_t         incx,
                                                            hipblasStride   stridex,
                                                            double*         y,
                                                            int64_t         incy,
                                                            hipblasStride   stridey,
                                                            const double*   c,
                                                            const double*   s,
                                                            int64_t         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCrotStridedBatched_64(hipblasHandle_t       handle,
                                                            int64_t               n,
                                                            hipblasComplex*       x,
                                                            int64_t               incx,
                                                            hipblasStride         stridex,
                                                            hipblasComplex*       y,
                                                            int64_t               incy,
                                                            hipblasStride         stridey,
                                                            const float*          c,
                                                            const hipblasComplex* s,
                                                            int64_t               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsrotStridedBatched_64(hipblasHandle_t handle,
                                                             int64_t         n,
                                                             hipblasComplex* x,
                                                             int64_t         incx,
                                                             hipblasStride   stridex,
                                                             hipblasComplex* y,
                                                             int64_t         incy,
                                                             hipblasStride   stridey,
                                                             const float*    c,
                                                             const float*    s,
                                                             int64_t         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZrotStridedBatched_64(hipblasHandle_t             handle,
                                                            int64_t                     n,
                                                            hipblasDoubleComplex*       x,
                                                            int64_t                     incx,
                                                            hipblasStride               stridex,
                                                            hipblasDoubleComplex*       y,
                                                            int64_t                     incy,
                                                            hipblasStride               stridey,
                                                            const double*               c,
                                                            const hipblasDoubleComplex* s,
                                                            int64_t                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdrotStridedBatched_64(hipblasHandle_t       handle,
                                                             int64_t               n,
                                                             hipblasDoubleComplex* x,
                                                             int64_t               incx,
                                                             hipblasStride         stridex,
                                                             hipblasDoubleComplex* y,
                                                             int64_t               incy,
                                                             hipblasStride         stridey,
                                                             const double*         c,
                                                             const double*         s,
                                                             int64_t               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCrotStridedBatched_v2_64(hipblasHandle_t   handle,
                                                               int64_t           n,
                                                               hipComplex*       x,
                                                               int64_t           incx,
                                                               hipblasStride     stridex,
                                                               hipComplex*       y,
                                                               int64_t           incy,
                                                               hipblasStride     stridey,
                                                               const float*      c,
                                                               const hipComplex* s,
                                                               int64_t           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsrotStridedBatched_v2_64(hipblasHandle_t handle,
                                                                int64_t         n,
                                                                hipComplex*     x,
                                                                int64_t         incx,
                                                                hipblasStride   stridex,
                                                                hipComplex*     y,
                                                                int64_t         incy,
                                                                hipblasStride   stridey,
                                                                const float*    c,
                                                                const float*    s,
                                                                int64_t         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZrotStridedBatched_v2_64(hipblasHandle_t         handle,
                                                               int64_t                 n,
                                                               hipDoubleComplex*       x,
                                                               int64_t                 incx,
                                                               hipblasStride           stridex,
                                                               hipDoubleComplex*       y,
                                                               int64_t                 incy,
                                                               hipblasStride           stridey,
                                                               const double*           c,
                                                               const hipDoubleComplex* s,
                                                               int64_t                 batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdrotStridedBatched_v2_64(hipblasHandle_t   handle,
                                                                int64_t           n,
                                                                hipDoubleComplex* x,
                                                                int64_t           incx,
                                                                hipblasStride     stridex,
                                                                hipDoubleComplex* y,
                                                                int64_t           incy,
                                                                hipblasStride     stridey,
                                                                const double*     c,
                                                                const double*     s,
                                                                int64_t           batchCount);
//! @}

/*! @{
    \brief BLAS Level 1 API

    \details
    rotg creates the Givens rotation matrix for the vector (a b).
         Scalars c and s and arrays a and b may be stored in either host or device memory, location is specified by calling hipblasSetPointerMode.
         If the pointer mode is set to HIPBLAS_POINTER_MODE_HOST, this function blocks the CPU until the GPU has finished and the results are available in host memory.
         If the pointer mode is set to HIPBLAS_POINTER_MODE_DEVICE, this function returns immediately and synchronization is required to read the results.

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : s,d,c,z

    @param[in]
    handle  [hipblasHandle_t]
            handle to the hipblas library context queue.
    @param[inout]
    a       device pointer or host pointer to input vector element, overwritten with r.
    @param[inout]
    b       device pointer or host pointer to input vector element, overwritten with z.
    @param[inout]
    c       device pointer or host pointer to cosine element of Givens rotation.
    @param[inout]
    s       device pointer or host pointer sine element of Givens rotation.

            ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t
    hipblasSrotg(hipblasHandle_t handle, float* a, float* b, float* c, float* s);

HIPBLAS_EXPORT hipblasStatus_t
    hipblasDrotg(hipblasHandle_t handle, double* a, double* b, double* c, double* s);

HIPBLAS_EXPORT hipblasStatus_t hipblasCrotg(
    hipblasHandle_t handle, hipblasComplex* a, hipblasComplex* b, float* c, hipblasComplex* s);

HIPBLAS_EXPORT hipblasStatus_t hipblasZrotg(hipblasHandle_t       handle,
                                            hipblasDoubleComplex* a,
                                            hipblasDoubleComplex* b,
                                            double*               c,
                                            hipblasDoubleComplex* s);

HIPBLAS_EXPORT hipblasStatus_t
    hipblasCrotg_v2(hipblasHandle_t handle, hipComplex* a, hipComplex* b, float* c, hipComplex* s);

HIPBLAS_EXPORT hipblasStatus_t hipblasZrotg_v2(hipblasHandle_t   handle,
                                               hipDoubleComplex* a,
                                               hipDoubleComplex* b,
                                               double*           c,
                                               hipDoubleComplex* s);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t
    hipblasSrotg_64(hipblasHandle_t handle, float* a, float* b, float* c, float* s);

HIPBLAS_EXPORT hipblasStatus_t
    hipblasDrotg_64(hipblasHandle_t handle, double* a, double* b, double* c, double* s);

HIPBLAS_EXPORT hipblasStatus_t hipblasCrotg_64(
    hipblasHandle_t handle, hipblasComplex* a, hipblasComplex* b, float* c, hipblasComplex* s);

HIPBLAS_EXPORT hipblasStatus_t hipblasZrotg_64(hipblasHandle_t       handle,
                                               hipblasDoubleComplex* a,
                                               hipblasDoubleComplex* b,
                                               double*               c,
                                               hipblasDoubleComplex* s);

HIPBLAS_EXPORT hipblasStatus_t hipblasCrotg_v2_64(
    hipblasHandle_t handle, hipComplex* a, hipComplex* b, float* c, hipComplex* s);

HIPBLAS_EXPORT hipblasStatus_t hipblasZrotg_v2_64(hipblasHandle_t   handle,
                                                  hipDoubleComplex* a,
                                                  hipDoubleComplex* b,
                                                  double*           c,
                                                  hipDoubleComplex* s);
//! @}

/*! @{
    \brief BLAS Level 1 API

    \details
    rotgBatched creates the Givens rotation matrix for the batched vectors (a_i b_i), for i = 1, ..., batchCount.
         a, b, c, and s may be stored in either host or device memory, location is specified by calling hipblasSetPointerMode.
         If the pointer mode is set to HIPBLAS_POINTER_MODE_HOST, this function blocks the CPU until the GPU has finished and the results are available in host memory.
         If the pointer mode is set to HIPBLAS_POINTER_MODE_DEVICE, this function returns immediately and synchronization is required to read the results.

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle  [hipblasHandle_t]
            handle to the hipblas library context queue.
    @param[inout]
    a       device array of device pointers storing each single input vector element a_i, overwritten with r_i.
    @param[inout]
    b       device array of device pointers storing each single input vector element b_i, overwritten with z_i.
    @param[inout]
    c       device array of device pointers storing each cosine element of Givens rotation for the batch.
    @param[inout]
    s       device array of device pointers storing each sine element of Givens rotation for the batch.
    @param[in]
    batchCount [int]
                number of batches (length of arrays a, b, c, and s).

            ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSrotgBatched(hipblasHandle_t handle,
                                                   float* const    a[],
                                                   float* const    b[],
                                                   float* const    c[],
                                                   float* const    s[],
                                                   int             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDrotgBatched(hipblasHandle_t handle,
                                                   double* const   a[],
                                                   double* const   b[],
                                                   double* const   c[],
                                                   double* const   s[],
                                                   int             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCrotgBatched(hipblasHandle_t       handle,
                                                   hipblasComplex* const a[],
                                                   hipblasComplex* const b[],
                                                   float* const          c[],
                                                   hipblasComplex* const s[],
                                                   int                   batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZrotgBatched(hipblasHandle_t             handle,
                                                   hipblasDoubleComplex* const a[],
                                                   hipblasDoubleComplex* const b[],
                                                   double* const               c[],
                                                   hipblasDoubleComplex* const s[],
                                                   int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCrotgBatched_v2(hipblasHandle_t   handle,
                                                      hipComplex* const a[],
                                                      hipComplex* const b[],
                                                      float* const      c[],
                                                      hipComplex* const s[],
                                                      int               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZrotgBatched_v2(hipblasHandle_t         handle,
                                                      hipDoubleComplex* const a[],
                                                      hipDoubleComplex* const b[],
                                                      double* const           c[],
                                                      hipDoubleComplex* const s[],
                                                      int                     batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSrotgBatched_64(hipblasHandle_t handle,
                                                      float* const    a[],
                                                      float* const    b[],
                                                      float* const    c[],
                                                      float* const    s[],
                                                      int64_t         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDrotgBatched_64(hipblasHandle_t handle,
                                                      double* const   a[],
                                                      double* const   b[],
                                                      double* const   c[],
                                                      double* const   s[],
                                                      int64_t         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCrotgBatched_64(hipblasHandle_t       handle,
                                                      hipblasComplex* const a[],
                                                      hipblasComplex* const b[],
                                                      float* const          c[],
                                                      hipblasComplex* const s[],
                                                      int64_t               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZrotgBatched_64(hipblasHandle_t             handle,
                                                      hipblasDoubleComplex* const a[],
                                                      hipblasDoubleComplex* const b[],
                                                      double* const               c[],
                                                      hipblasDoubleComplex* const s[],
                                                      int64_t                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCrotgBatched_v2_64(hipblasHandle_t   handle,
                                                         hipComplex* const a[],
                                                         hipComplex* const b[],
                                                         float* const      c[],
                                                         hipComplex* const s[],
                                                         int64_t           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZrotgBatched_v2_64(hipblasHandle_t         handle,
                                                         hipDoubleComplex* const a[],
                                                         hipDoubleComplex* const b[],
                                                         double* const           c[],
                                                         hipDoubleComplex* const s[],
                                                         int64_t                 batchCount);
//! @}

/*! @{
    \brief BLAS Level 1 API

    \details
    rotgStridedBatched creates the Givens rotation matrix for the strided batched vectors (a_i b_i), for i = 1, ..., batchCount.
         a, b, c, and s may be stored in either host or device memory, location is specified by calling hipblasSetPointerMode.
         If the pointer mode is set to HIPBLAS_POINTER_MODE_HOST, this function blocks the CPU until the GPU has finished and the results are available in host memory.
         If the pointer mode is set to HIPBLAS_POINTER_MODE_HOST, this function returns immediately and synchronization is required to read the results.

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle  [hipblasHandle_t]
            handle to the hipblas library context queue.
    @param[inout]
    a       device strided_batched pointer or host strided_batched pointer to first single input vector element a_1, overwritten with r.
    @param[in]
    stridea [hipblasStride]
             distance between elements of a in batch (distance between a_i and a_(i + 1))
    @param[inout]
    b       device strided_batched pointer or host strided_batched pointer to first single input vector element b_1, overwritten with z.
    @param[in]
    strideb [hipblasStride]
             distance between elements of b in batch (distance between b_i and b_(i + 1))
    @param[inout]
    c       device strided_batched pointer or host strided_batched pointer to first cosine element of Givens rotations c_1.
    @param[in]
    stridec [hipblasStride]
             distance between elements of c in batch (distance between c_i and c_(i + 1))
    @param[inout]
    s       device strided_batched pointer or host strided_batched pointer to sine element of Givens rotations s_1.
    @param[in]
    strides [hipblasStride]
             distance between elements of s in batch (distance between s_i and s_(i + 1))
    @param[in]
    batchCount [int]
                number of batches (length of arrays a, b, c, and s).

            ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSrotgStridedBatched(hipblasHandle_t handle,
                                                          float*          a,
                                                          hipblasStride   stridea,
                                                          float*          b,
                                                          hipblasStride   strideb,
                                                          float*          c,
                                                          hipblasStride   stridec,
                                                          float*          s,
                                                          hipblasStride   strides,
                                                          int             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDrotgStridedBatched(hipblasHandle_t handle,
                                                          double*         a,
                                                          hipblasStride   stridea,
                                                          double*         b,
                                                          hipblasStride   strideb,
                                                          double*         c,
                                                          hipblasStride   stridec,
                                                          double*         s,
                                                          hipblasStride   strides,
                                                          int             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCrotgStridedBatched(hipblasHandle_t handle,
                                                          hipblasComplex* a,
                                                          hipblasStride   stridea,
                                                          hipblasComplex* b,
                                                          hipblasStride   strideb,
                                                          float*          c,
                                                          hipblasStride   stridec,
                                                          hipblasComplex* s,
                                                          hipblasStride   strides,
                                                          int             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZrotgStridedBatched(hipblasHandle_t       handle,
                                                          hipblasDoubleComplex* a,
                                                          hipblasStride         stridea,
                                                          hipblasDoubleComplex* b,
                                                          hipblasStride         strideb,
                                                          double*               c,
                                                          hipblasStride         stridec,
                                                          hipblasDoubleComplex* s,
                                                          hipblasStride         strides,
                                                          int                   batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCrotgStridedBatched_v2(hipblasHandle_t handle,
                                                             hipComplex*     a,
                                                             hipblasStride   stridea,
                                                             hipComplex*     b,
                                                             hipblasStride   strideb,
                                                             float*          c,
                                                             hipblasStride   stridec,
                                                             hipComplex*     s,
                                                             hipblasStride   strides,
                                                             int             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZrotgStridedBatched_v2(hipblasHandle_t   handle,
                                                             hipDoubleComplex* a,
                                                             hipblasStride     stridea,
                                                             hipDoubleComplex* b,
                                                             hipblasStride     strideb,
                                                             double*           c,
                                                             hipblasStride     stridec,
                                                             hipDoubleComplex* s,
                                                             hipblasStride     strides,
                                                             int               batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSrotgStridedBatched_64(hipblasHandle_t handle,
                                                             float*          a,
                                                             hipblasStride   stridea,
                                                             float*          b,
                                                             hipblasStride   strideb,
                                                             float*          c,
                                                             hipblasStride   stridec,
                                                             float*          s,
                                                             hipblasStride   strides,
                                                             int64_t         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDrotgStridedBatched_64(hipblasHandle_t handle,
                                                             double*         a,
                                                             hipblasStride   stridea,
                                                             double*         b,
                                                             hipblasStride   strideb,
                                                             double*         c,
                                                             hipblasStride   stridec,
                                                             double*         s,
                                                             hipblasStride   strides,
                                                             int64_t         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCrotgStridedBatched_64(hipblasHandle_t handle,
                                                             hipblasComplex* a,
                                                             hipblasStride   stridea,
                                                             hipblasComplex* b,
                                                             hipblasStride   strideb,
                                                             float*          c,
                                                             hipblasStride   stridec,
                                                             hipblasComplex* s,
                                                             hipblasStride   strides,
                                                             int64_t         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZrotgStridedBatched_64(hipblasHandle_t       handle,
                                                             hipblasDoubleComplex* a,
                                                             hipblasStride         stridea,
                                                             hipblasDoubleComplex* b,
                                                             hipblasStride         strideb,
                                                             double*               c,
                                                             hipblasStride         stridec,
                                                             hipblasDoubleComplex* s,
                                                             hipblasStride         strides,
                                                             int64_t               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCrotgStridedBatched_v2_64(hipblasHandle_t handle,
                                                                hipComplex*     a,
                                                                hipblasStride   stridea,
                                                                hipComplex*     b,
                                                                hipblasStride   strideb,
                                                                float*          c,
                                                                hipblasStride   stridec,
                                                                hipComplex*     s,
                                                                hipblasStride   strides,
                                                                int64_t         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZrotgStridedBatched_v2_64(hipblasHandle_t   handle,
                                                                hipDoubleComplex* a,
                                                                hipblasStride     stridea,
                                                                hipDoubleComplex* b,
                                                                hipblasStride     strideb,
                                                                double*           c,
                                                                hipblasStride     stridec,
                                                                hipDoubleComplex* s,
                                                                hipblasStride     strides,
                                                                int64_t           batchCount);
//! @}

/*! @{
    \brief BLAS Level 1 API

    \details
    rotm applies the modified Givens rotation matrix defined by param to vectors x and y.

    - Supported precisions in rocBLAS : s,d
    - Supported precisions in cuBLAS  : s,d

    @param[in]
    handle  [hipblasHandle_t]
            handle to the hipblas library context queue.
    @param[in]
    n       [int]
            number of elements in the x and y vectors.
    @param[inout]
    x       device pointer storing vector x.
    @param[in]
    incx    [int]
            specifies the increment between elements of x.
    @param[inout]
    y       device pointer storing vector y.
    @param[in]
    incy    [int]
            specifies the increment between elements of y.
    @param[in]
    param   device vector or host vector of 5 elements defining the rotation.
            param[0] = flag
            param[1] = H11
            param[2] = H21
            param[3] = H12
            param[4] = H22
            The flag parameter defines the form of H:
            flag = -1 => H = ( H11 H12 H21 H22 )
            flag =  0 => H = ( 1.0 H12 H21 1.0 )
            flag =  1 => H = ( H11 1.0 -1.0 H22 )
            flag = -2 => H = ( 1.0 0.0 0.0 1.0 )
            param may be stored in either host or device memory, location is specified by calling hipblasSetPointerMode.

            ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSrotm(
    hipblasHandle_t handle, int n, float* x, int incx, float* y, int incy, const float* param);

HIPBLAS_EXPORT hipblasStatus_t hipblasDrotm(
    hipblasHandle_t handle, int n, double* x, int incx, double* y, int incy, const double* param);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSrotm_64(hipblasHandle_t handle,
                                               int64_t         n,
                                               float*          x,
                                               int64_t         incx,
                                               float*          y,
                                               int64_t         incy,
                                               const float*    param);

HIPBLAS_EXPORT hipblasStatus_t hipblasDrotm_64(hipblasHandle_t handle,
                                               int64_t         n,
                                               double*         x,
                                               int64_t         incx,
                                               double*         y,
                                               int64_t         incy,
                                               const double*   param);
//! @}

/*! @{
    \brief BLAS Level 1 API

    \details
    rotmBatched applies the modified Givens rotation matrix defined by param_i to batched vectors x_i and y_i, for i = 1, ..., batchCount.

    - Supported precisions in rocBLAS : s,d
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle  [hipblasHandle_t]
            handle to the hipblas library context queue.
    @param[in]
    n       [int]
            number of elements in the x and y vectors.
    @param[inout]
    x       device array of device pointers storing each vector x_i.
    @param[in]
    incx    [int]
            specifies the increment between elements of each x_i.
    @param[inout]
    y       device array of device pointers storing each vector y_1.
    @param[in]
    incy    [int]
            specifies the increment between elements of each y_i.
    @param[in]
    param   device array of device vectors of 5 elements defining the rotation.
            param[0] = flag
            param[1] = H11
            param[2] = H21
            param[3] = H12
            param[4] = H22
            The flag parameter defines the form of H:
            flag = -1 => H = ( H11 H12 H21 H22 )
            flag =  0 => H = ( 1.0 H12 H21 1.0 )
            flag =  1 => H = ( H11 1.0 -1.0 H22 )
            flag = -2 => H = ( 1.0 0.0 0.0 1.0 )
            param may ONLY be stored on the device for the batched version of this function.
    @param[in]
    batchCount [int]
                the number of x and y arrays, i.e. the number of batches.

            ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSrotmBatched(hipblasHandle_t    handle,
                                                   int                n,
                                                   float* const       x[],
                                                   int                incx,
                                                   float* const       y[],
                                                   int                incy,
                                                   const float* const param[],
                                                   int                batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDrotmBatched(hipblasHandle_t     handle,
                                                   int                 n,
                                                   double* const       x[],
                                                   int                 incx,
                                                   double* const       y[],
                                                   int                 incy,
                                                   const double* const param[],
                                                   int                 batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSrotmBatched_64(hipblasHandle_t    handle,
                                                      int64_t            n,
                                                      float* const       x[],
                                                      int64_t            incx,
                                                      float* const       y[],
                                                      int64_t            incy,
                                                      const float* const param[],
                                                      int64_t            batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDrotmBatched_64(hipblasHandle_t     handle,
                                                      int64_t             n,
                                                      double* const       x[],
                                                      int64_t             incx,
                                                      double* const       y[],
                                                      int64_t             incy,
                                                      const double* const param[],
                                                      int64_t             batchCount);
//! @}

/*! @{
    \brief BLAS Level 1 API

    \details
    rotmStridedBatched applies the modified Givens rotation matrix defined by param_i to strided batched vectors x_i and y_i, for i = 1, ..., batchCount

    - Supported precisions in rocBLAS : s,d
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle  [hipblasHandle_t]
            handle to the hipblas library context queue.
    @param[in]
    n       [int]
            number of elements in the x and y vectors.
    @param[inout]
    x       device pointer pointing to first strided batched vector x_1.
    @param[in]
    incx    [int]
            specifies the increment between elements of each x_i.
    @param[in]
    stridex [hipblasStride]
             specifies the increment between the beginning of x_i and x_(i + 1)
    @param[inout]
    y       device pointer pointing to first strided batched vector y_1.
    @param[in]
    incy    [int]
            specifies the increment between elements of each y_i.
    @param[in]
    stridey  [hipblasStride]
             specifies the increment between the beginning of y_i and y_(i + 1)
    @param[in]
    param   device pointer pointing to first array of 5 elements defining the rotation (param_1).
            param[0] = flag
            param[1] = H11
            param[2] = H21
            param[3] = H12
            param[4] = H22
            The flag parameter defines the form of H:
            flag = -1 => H = ( H11 H12 H21 H22 )
            flag =  0 => H = ( 1.0 H12 H21 1.0 )
            flag =  1 => H = ( H11 1.0 -1.0 H22 )
            flag = -2 => H = ( 1.0 0.0 0.0 1.0 )
            param may ONLY be stored on the device for the strided_batched version of this function.
    @param[in]
    strideParam [hipblasStride]
                 specifies the increment between the beginning of param_i and param_(i + 1)
    @param[in]
    batchCount [int]
                the number of x and y arrays, i.e. the number of batches.

            ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSrotmStridedBatched(hipblasHandle_t handle,
                                                          int             n,
                                                          float*          x,
                                                          int             incx,
                                                          hipblasStride   stridex,
                                                          float*          y,
                                                          int             incy,
                                                          hipblasStride   stridey,
                                                          const float*    param,
                                                          hipblasStride   strideParam,
                                                          int             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDrotmStridedBatched(hipblasHandle_t handle,
                                                          int             n,
                                                          double*         x,
                                                          int             incx,
                                                          hipblasStride   stridex,
                                                          double*         y,
                                                          int             incy,
                                                          hipblasStride   stridey,
                                                          const double*   param,
                                                          hipblasStride   strideParam,
                                                          int             batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSrotmStridedBatched_64(hipblasHandle_t handle,
                                                             int64_t         n,
                                                             float*          x,
                                                             int64_t         incx,
                                                             hipblasStride   stridex,
                                                             float*          y,
                                                             int64_t         incy,
                                                             hipblasStride   stridey,
                                                             const float*    param,
                                                             hipblasStride   strideParam,
                                                             int64_t         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDrotmStridedBatched_64(hipblasHandle_t handle,
                                                             int64_t         n,
                                                             double*         x,
                                                             int64_t         incx,
                                                             hipblasStride   stridex,
                                                             double*         y,
                                                             int64_t         incy,
                                                             hipblasStride   stridey,
                                                             const double*   param,
                                                             hipblasStride   strideParam,
                                                             int64_t         batchCount);
//! @}

/*! @{
    \brief BLAS Level 1 API

    \details
    rotmg creates the modified Givens rotation matrix for the vector (d1 * x1, d2 * y1).
          Parameters may be stored in either host or device memory, location is specified by calling hipblasSetPointerMode.
          If the pointer mode is set to HIPBLAS_POINTER_MODE_HOST, this function blocks the CPU until the GPU has finished and the results are available in host memory.
          If the pointer mode is set to HIPBLAS_POINTER_MODE_DEVICE, this function returns immediately and synchronization is required to read the results.

    - Supported precisions in rocBLAS : s,d
    - Supported precisions in cuBLAS  : s,d

    @param[in]
    handle  [hipblasHandle_t]
            handle to the hipblas library context queue.
    @param[inout]
    d1      device pointer or host pointer to input scalar that is overwritten.
    @param[inout]
    d2      device pointer or host pointer to input scalar that is overwritten.
    @param[inout]
    x1      device pointer or host pointer to input scalar that is overwritten.
    @param[in]
    y1      device pointer or host pointer to input scalar.
    @param[out]
    param   device vector or host vector of 5 elements defining the rotation.
            param[0] = flag
            param[1] = H11
            param[2] = H21
            param[3] = H12
            param[4] = H22
            The flag parameter defines the form of H:
            flag = -1 => H = ( H11 H12 H21 H22 )
            flag =  0 => H = ( 1.0 H12 H21 1.0 )
            flag =  1 => H = ( H11 1.0 -1.0 H22 )
            flag = -2 => H = ( 1.0 0.0 0.0 1.0 )
            param may be stored in either host or device memory, location is specified by calling hipblasSetPointerMode.

            ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSrotmg(
    hipblasHandle_t handle, float* d1, float* d2, float* x1, const float* y1, float* param);

HIPBLAS_EXPORT hipblasStatus_t hipblasDrotmg(
    hipblasHandle_t handle, double* d1, double* d2, double* x1, const double* y1, double* param);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSrotmg_64(
    hipblasHandle_t handle, float* d1, float* d2, float* x1, const float* y1, float* param);

HIPBLAS_EXPORT hipblasStatus_t hipblasDrotmg_64(
    hipblasHandle_t handle, double* d1, double* d2, double* x1, const double* y1, double* param);
//! @}

/*! @{
    \brief BLAS Level 1 API

    \details
    rotmgBatched creates the modified Givens rotation matrix for the batched vectors (d1_i * x1_i, d2_i * y1_i), for i = 1, ..., batchCount.
          Parameters may be stored in either host or device memory, location is specified by calling hipblasSetPointerMode.
          If the pointer mode is set to HIPBLAS_POINTER_MODE_HOST, this function blocks the CPU until the GPU has finished and the results are available in host memory.
          If the pointer mode is set to HIPBLAS_POINTER_MODE_DEVICE, this function returns immediately and synchronization is required to read the results.

    - Supported precisions in rocBLAS : s,d
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle  [hipblasHandle_t]
            handle to the hipblas library context queue.
    @param[inout]
    d1      device batched array or host batched array of input scalars that is overwritten.
    @param[inout]
    d2      device batched array or host batched array of input scalars that is overwritten.
    @param[inout]
    x1      device batched array or host batched array of input scalars that is overwritten.
    @param[in]
    y1      device batched array or host batched array of input scalars.
    @param[out]
    param   device batched array or host batched array of vectors of 5 elements defining the rotation.
            param[0] = flag
            param[1] = H11
            param[2] = H21
            param[3] = H12
            param[4] = H22
            The flag parameter defines the form of H:
            flag = -1 => H = ( H11 H12 H21 H22 )
            flag =  0 => H = ( 1.0 H12 H21 1.0 )
            flag =  1 => H = ( H11 1.0 -1.0 H22 )
            flag = -2 => H = ( 1.0 0.0 0.0 1.0 )
            param may be stored in either host or device memory, location is specified by calling hipblasSetPointerMode.
    @param[in]
    batchCount [int]
                the number of instances in the batch.

            ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSrotmgBatched(hipblasHandle_t    handle,
                                                    float* const       d1[],
                                                    float* const       d2[],
                                                    float* const       x1[],
                                                    const float* const y1[],
                                                    float* const       param[],
                                                    int                batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDrotmgBatched(hipblasHandle_t     handle,
                                                    double* const       d1[],
                                                    double* const       d2[],
                                                    double* const       x1[],
                                                    const double* const y1[],
                                                    double* const       param[],
                                                    int                 batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSrotmgBatched_64(hipblasHandle_t    handle,
                                                       float* const       d1[],
                                                       float* const       d2[],
                                                       float* const       x1[],
                                                       const float* const y1[],
                                                       float* const       param[],
                                                       int64_t            batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDrotmgBatched_64(hipblasHandle_t     handle,
                                                       double* const       d1[],
                                                       double* const       d2[],
                                                       double* const       x1[],
                                                       const double* const y1[],
                                                       double* const       param[],
                                                       int64_t             batchCount);
//! @}

/*! @{
    \brief BLAS Level 1 API

    \details
    rotmgStridedBatched creates the modified Givens rotation matrix for the strided batched vectors (d1_i * x1_i, d2_i * y1_i), for i = 1, ..., batchCount.
          Parameters may be stored in either host or device memory, location is specified by calling hipblasSetPointerMode.
          If the pointer mode is set to HIPBLAS_POINTER_MODE_HOST, this function blocks the CPU until the GPU has finished and the results are available in host memory.
          If the pointer mode is set to HIPBLAS_POINTER_MODE_DEVICE, this function returns immediately and synchronization is required to read the results.

    - Supported precisions in rocBLAS : s,d
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle  [hipblasHandle_t]
            handle to the hipblas library context queue.
    @param[inout]
    d1      device strided_batched array or host strided_batched array of input scalars that is overwritten.
    @param[in]
    strided1 [hipblasStride]
              specifies the increment between the beginning of d1_i and d1_(i+1)
    @param[inout]
    d2      device strided_batched array or host strided_batched array of input scalars that is overwritten.
    @param[in]
    strided2 [hipblasStride]
              specifies the increment between the beginning of d2_i and d2_(i+1)
    @param[inout]
    x1      device strided_batched array or host strided_batched array of input scalars that is overwritten.
    @param[in]
    stridex1 [hipblasStride]
              specifies the increment between the beginning of x1_i and x1_(i+1)
    @param[in]
    y1      device strided_batched array or host strided_batched array of input scalars.
    @param[in]
    stridey1 [hipblasStride]
              specifies the increment between the beginning of y1_i and y1_(i+1)
    @param[out]
    param   device stridedBatched array or host stridedBatched array of vectors of 5 elements defining the rotation.
            param[0] = flag
            param[1] = H11
            param[2] = H21
            param[3] = H12
            param[4] = H22
            The flag parameter defines the form of H:
            flag = -1 => H = ( H11 H12 H21 H22 )
            flag =  0 => H = ( 1.0 H12 H21 1.0 )
            flag =  1 => H = ( H11 1.0 -1.0 H22 )
            flag = -2 => H = ( 1.0 0.0 0.0 1.0 )
            param may be stored in either host or device memory, location is specified by calling hipblasSetPointerMode.
    @param[in]
    strideParam [hipblasStride]
                 specifies the increment between the beginning of param_i and param_(i + 1)
    @param[in]
    batchCount [int]
                the number of instances in the batch.

            ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSrotmgStridedBatched(hipblasHandle_t handle,
                                                           float*          d1,
                                                           hipblasStride   strided1,
                                                           float*          d2,
                                                           hipblasStride   strided2,
                                                           float*          x1,
                                                           hipblasStride   stridex1,
                                                           const float*    y1,
                                                           hipblasStride   stridey1,
                                                           float*          param,
                                                           hipblasStride   strideParam,
                                                           int             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDrotmgStridedBatched(hipblasHandle_t handle,
                                                           double*         d1,
                                                           hipblasStride   strided1,
                                                           double*         d2,
                                                           hipblasStride   strided2,
                                                           double*         x1,
                                                           hipblasStride   stridex1,
                                                           const double*   y1,
                                                           hipblasStride   stridey1,
                                                           double*         param,
                                                           hipblasStride   strideParam,
                                                           int             batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSrotmgStridedBatched_64(hipblasHandle_t handle,
                                                              float*          d1,
                                                              hipblasStride   strided1,
                                                              float*          d2,
                                                              hipblasStride   strided2,
                                                              float*          x1,
                                                              hipblasStride   stridex1,
                                                              const float*    y1,
                                                              hipblasStride   stridey1,
                                                              float*          param,
                                                              hipblasStride   strideParam,
                                                              int64_t         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDrotmgStridedBatched_64(hipblasHandle_t handle,
                                                              double*         d1,
                                                              hipblasStride   strided1,
                                                              double*         d2,
                                                              hipblasStride   strided2,
                                                              double*         x1,
                                                              hipblasStride   stridex1,
                                                              const double*   y1,
                                                              hipblasStride   stridey1,
                                                              double*         param,
                                                              hipblasStride   strideParam,
                                                              int64_t         batchCount);
//! @}

/*! @{
    \brief BLAS Level 1 API

    \details
    scal  scales each element of vector x with scalar alpha.

        x := alpha * x

    - Supported precisions in rocBLAS : s,d,c,z,cs,zd
    - Supported precisions in cuBLAS  : s,d,c,z,cs,zd

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    n         [int]
              the number of elements in x.
    @param[in]
    alpha     device pointer or host pointer for the scalar alpha.
    @param[inout]
    x         device pointer storing vector x.
    @param[in]
    incx      [int]
              specifies the increment for the elements of x.

            ********************************************************************/
HIPBLAS_EXPORT hipblasStatus_t
    hipblasSscal(hipblasHandle_t handle, int n, const float* alpha, float* x, int incx);

HIPBLAS_EXPORT hipblasStatus_t
    hipblasDscal(hipblasHandle_t handle, int n, const double* alpha, double* x, int incx);

HIPBLAS_EXPORT hipblasStatus_t hipblasCscal(
    hipblasHandle_t handle, int n, const hipblasComplex* alpha, hipblasComplex* x, int incx);

HIPBLAS_EXPORT hipblasStatus_t
    hipblasCsscal(hipblasHandle_t handle, int n, const float* alpha, hipblasComplex* x, int incx);

HIPBLAS_EXPORT hipblasStatus_t hipblasZscal(hipblasHandle_t             handle,
                                            int                         n,
                                            const hipblasDoubleComplex* alpha,
                                            hipblasDoubleComplex*       x,
                                            int                         incx);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdscal(
    hipblasHandle_t handle, int n, const double* alpha, hipblasDoubleComplex* x, int incx);

HIPBLAS_EXPORT hipblasStatus_t hipblasCscal_v2(
    hipblasHandle_t handle, int n, const hipComplex* alpha, hipComplex* x, int incx);

HIPBLAS_EXPORT hipblasStatus_t
    hipblasCsscal_v2(hipblasHandle_t handle, int n, const float* alpha, hipComplex* x, int incx);

HIPBLAS_EXPORT hipblasStatus_t hipblasZscal_v2(
    hipblasHandle_t handle, int n, const hipDoubleComplex* alpha, hipDoubleComplex* x, int incx);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdscal_v2(
    hipblasHandle_t handle, int n, const double* alpha, hipDoubleComplex* x, int incx);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t
    hipblasSscal_64(hipblasHandle_t handle, int64_t n, const float* alpha, float* x, int64_t incx);

HIPBLAS_EXPORT hipblasStatus_t hipblasDscal_64(
    hipblasHandle_t handle, int64_t n, const double* alpha, double* x, int64_t incx);

HIPBLAS_EXPORT hipblasStatus_t hipblasCscal_64(hipblasHandle_t       handle,
                                               int64_t               n,
                                               const hipblasComplex* alpha,
                                               hipblasComplex*       x,
                                               int64_t               incx);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsscal_64(
    hipblasHandle_t handle, int64_t n, const float* alpha, hipblasComplex* x, int64_t incx);

HIPBLAS_EXPORT hipblasStatus_t hipblasZscal_64(hipblasHandle_t             handle,
                                               int64_t                     n,
                                               const hipblasDoubleComplex* alpha,
                                               hipblasDoubleComplex*       x,
                                               int64_t                     incx);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdscal_64(
    hipblasHandle_t handle, int64_t n, const double* alpha, hipblasDoubleComplex* x, int64_t incx);

HIPBLAS_EXPORT hipblasStatus_t hipblasCscal_v2_64(
    hipblasHandle_t handle, int64_t n, const hipComplex* alpha, hipComplex* x, int64_t incx);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsscal_v2_64(
    hipblasHandle_t handle, int64_t n, const float* alpha, hipComplex* x, int64_t incx);

HIPBLAS_EXPORT hipblasStatus_t hipblasZscal_v2_64(hipblasHandle_t         handle,
                                                  int64_t                 n,
                                                  const hipDoubleComplex* alpha,
                                                  hipDoubleComplex*       x,
                                                  int64_t                 incx);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdscal_v2_64(
    hipblasHandle_t handle, int64_t n, const double* alpha, hipDoubleComplex* x, int64_t incx);
//! @}

/*! @{
    \brief BLAS Level 1 API
    \details
    scalBatched  scales each element of vector x_i with scalar alpha, for i = 1, ... , batchCount.

         x_i := alpha * x_i

     where (x_i) is the i-th instance of the batch.

    - Supported precisions in rocBLAS : s,d,c,z,cs,zd
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle      [hipblasHandle_t]
                handle to the hipblas library context queue.
    @param[in]
    n           [int]
                the number of elements in each x_i.
    @param[in]
    alpha       host pointer or device pointer for the scalar alpha.
    @param[inout]
    x           device array of device pointers storing each vector x_i.
    @param[in]
    incx        [int]
                specifies the increment for the elements of each x_i.
    @param[in]
    batchCount [int]
                specifies the number of batches in x.

             ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSscalBatched(
    hipblasHandle_t handle, int n, const float* alpha, float* const x[], int incx, int batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDscalBatched(hipblasHandle_t handle,
                                                   int             n,
                                                   const double*   alpha,
                                                   double* const   x[],
                                                   int             incx,
                                                   int             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCscalBatched(hipblasHandle_t       handle,
                                                   int                   n,
                                                   const hipblasComplex* alpha,
                                                   hipblasComplex* const x[],
                                                   int                   incx,
                                                   int                   batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZscalBatched(hipblasHandle_t             handle,
                                                   int                         n,
                                                   const hipblasDoubleComplex* alpha,
                                                   hipblasDoubleComplex* const x[],
                                                   int                         incx,
                                                   int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsscalBatched(hipblasHandle_t       handle,
                                                    int                   n,
                                                    const float*          alpha,
                                                    hipblasComplex* const x[],
                                                    int                   incx,
                                                    int                   batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdscalBatched(hipblasHandle_t             handle,
                                                    int                         n,
                                                    const double*               alpha,
                                                    hipblasDoubleComplex* const x[],
                                                    int                         incx,
                                                    int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCscalBatched_v2(hipblasHandle_t   handle,
                                                      int               n,
                                                      const hipComplex* alpha,
                                                      hipComplex* const x[],
                                                      int               incx,
                                                      int               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZscalBatched_v2(hipblasHandle_t         handle,
                                                      int                     n,
                                                      const hipDoubleComplex* alpha,
                                                      hipDoubleComplex* const x[],
                                                      int                     incx,
                                                      int                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsscalBatched_v2(hipblasHandle_t   handle,
                                                       int               n,
                                                       const float*      alpha,
                                                       hipComplex* const x[],
                                                       int               incx,
                                                       int               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdscalBatched_v2(hipblasHandle_t         handle,
                                                       int                     n,
                                                       const double*           alpha,
                                                       hipDoubleComplex* const x[],
                                                       int                     incx,
                                                       int                     batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSscalBatched_64(hipblasHandle_t handle,
                                                      int64_t         n,
                                                      const float*    alpha,
                                                      float* const    x[],
                                                      int64_t         incx,
                                                      int64_t         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDscalBatched_64(hipblasHandle_t handle,
                                                      int64_t         n,
                                                      const double*   alpha,
                                                      double* const   x[],
                                                      int64_t         incx,
                                                      int64_t         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCscalBatched_64(hipblasHandle_t       handle,
                                                      int64_t               n,
                                                      const hipblasComplex* alpha,
                                                      hipblasComplex* const x[],
                                                      int64_t               incx,
                                                      int64_t               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZscalBatched_64(hipblasHandle_t             handle,
                                                      int64_t                     n,
                                                      const hipblasDoubleComplex* alpha,
                                                      hipblasDoubleComplex* const x[],
                                                      int64_t                     incx,
                                                      int64_t                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsscalBatched_64(hipblasHandle_t       handle,
                                                       int64_t               n,
                                                       const float*          alpha,
                                                       hipblasComplex* const x[],
                                                       int64_t               incx,
                                                       int64_t               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdscalBatched_64(hipblasHandle_t             handle,
                                                       int64_t                     n,
                                                       const double*               alpha,
                                                       hipblasDoubleComplex* const x[],
                                                       int64_t                     incx,
                                                       int64_t                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCscalBatched_v2_64(hipblasHandle_t   handle,
                                                         int64_t           n,
                                                         const hipComplex* alpha,
                                                         hipComplex* const x[],
                                                         int64_t           incx,
                                                         int64_t           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZscalBatched_v2_64(hipblasHandle_t         handle,
                                                         int64_t                 n,
                                                         const hipDoubleComplex* alpha,
                                                         hipDoubleComplex* const x[],
                                                         int64_t                 incx,
                                                         int64_t                 batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsscalBatched_v2_64(hipblasHandle_t   handle,
                                                          int64_t           n,
                                                          const float*      alpha,
                                                          hipComplex* const x[],
                                                          int64_t           incx,
                                                          int64_t           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdscalBatched_v2_64(hipblasHandle_t         handle,
                                                          int64_t                 n,
                                                          const double*           alpha,
                                                          hipDoubleComplex* const x[],
                                                          int64_t                 incx,
                                                          int64_t                 batchCount);
//! @}

/*! @{
    \brief BLAS Level 1 API
    \details
    scalStridedBatched  scales each element of vector x_i with scalar alpha, for i = 1, ... , batchCount.

         x_i := alpha * x_i ,

     where (x_i) is the i-th instance of the batch.

    - Supported precisions in rocBLAS : s,d,c,z,cs,zd
    - Supported precisions in cuBLAS  : No support

     @param[in]
    handle      [hipblasHandle_t]
                handle to the hipblas library context queue.
    @param[in]
    n           [int]
                the number of elements in each x_i.
    @param[in]
    alpha       host pointer or device pointer for the scalar alpha.
    @param[inout]
    x           device pointer to the first vector (x_1) in the batch.
    @param[in]
    incx        [int]
                specifies the increment for the elements of x.
    @param[in]
    stridex     [hipblasStride]
                stride from the start of one vector (x_i) and the next one (x_i+1).
                There are no restrictions placed on stride_x, however the user should
                take care to ensure that stride_x is of appropriate size, for a typical
                case this means stride_x >= n * incx.
    @param[in]
    batchCount [int]
                specifies the number of batches in x.

             ********************************************************************/
HIPBLAS_EXPORT hipblasStatus_t hipblasSscalStridedBatched(hipblasHandle_t handle,
                                                          int             n,
                                                          const float*    alpha,
                                                          float*          x,
                                                          int             incx,
                                                          hipblasStride   stridex,
                                                          int             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDscalStridedBatched(hipblasHandle_t handle,
                                                          int             n,
                                                          const double*   alpha,
                                                          double*         x,
                                                          int             incx,
                                                          hipblasStride   stridex,
                                                          int             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCscalStridedBatched(hipblasHandle_t       handle,
                                                          int                   n,
                                                          const hipblasComplex* alpha,
                                                          hipblasComplex*       x,
                                                          int                   incx,
                                                          hipblasStride         stridex,
                                                          int                   batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZscalStridedBatched(hipblasHandle_t             handle,
                                                          int                         n,
                                                          const hipblasDoubleComplex* alpha,
                                                          hipblasDoubleComplex*       x,
                                                          int                         incx,
                                                          hipblasStride               stridex,
                                                          int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsscalStridedBatched(hipblasHandle_t handle,
                                                           int             n,
                                                           const float*    alpha,
                                                           hipblasComplex* x,
                                                           int             incx,
                                                           hipblasStride   stridex,
                                                           int             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdscalStridedBatched(hipblasHandle_t       handle,
                                                           int                   n,
                                                           const double*         alpha,
                                                           hipblasDoubleComplex* x,
                                                           int                   incx,
                                                           hipblasStride         stridex,
                                                           int                   batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCscalStridedBatched_v2(hipblasHandle_t   handle,
                                                             int               n,
                                                             const hipComplex* alpha,
                                                             hipComplex*       x,
                                                             int               incx,
                                                             hipblasStride     stridex,
                                                             int               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZscalStridedBatched_v2(hipblasHandle_t         handle,
                                                             int                     n,
                                                             const hipDoubleComplex* alpha,
                                                             hipDoubleComplex*       x,
                                                             int                     incx,
                                                             hipblasStride           stridex,
                                                             int                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsscalStridedBatched_v2(hipblasHandle_t handle,
                                                              int             n,
                                                              const float*    alpha,
                                                              hipComplex*     x,
                                                              int             incx,
                                                              hipblasStride   stridex,
                                                              int             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdscalStridedBatched_v2(hipblasHandle_t   handle,
                                                              int               n,
                                                              const double*     alpha,
                                                              hipDoubleComplex* x,
                                                              int               incx,
                                                              hipblasStride     stridex,
                                                              int               batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSscalStridedBatched_64(hipblasHandle_t handle,
                                                             int64_t         n,
                                                             const float*    alpha,
                                                             float*          x,
                                                             int64_t         incx,
                                                             hipblasStride   stridex,
                                                             int64_t         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDscalStridedBatched_64(hipblasHandle_t handle,
                                                             int64_t         n,
                                                             const double*   alpha,
                                                             double*         x,
                                                             int64_t         incx,
                                                             hipblasStride   stridex,
                                                             int64_t         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCscalStridedBatched_64(hipblasHandle_t       handle,
                                                             int64_t               n,
                                                             const hipblasComplex* alpha,
                                                             hipblasComplex*       x,
                                                             int64_t               incx,
                                                             hipblasStride         stridex,
                                                             int64_t               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZscalStridedBatched_64(hipblasHandle_t             handle,
                                                             int64_t                     n,
                                                             const hipblasDoubleComplex* alpha,
                                                             hipblasDoubleComplex*       x,
                                                             int64_t                     incx,
                                                             hipblasStride               stridex,
                                                             int64_t batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsscalStridedBatched_64(hipblasHandle_t handle,
                                                              int64_t         n,
                                                              const float*    alpha,
                                                              hipblasComplex* x,
                                                              int64_t         incx,
                                                              hipblasStride   stridex,
                                                              int64_t         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdscalStridedBatched_64(hipblasHandle_t       handle,
                                                              int64_t               n,
                                                              const double*         alpha,
                                                              hipblasDoubleComplex* x,
                                                              int64_t               incx,
                                                              hipblasStride         stridex,
                                                              int64_t               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCscalStridedBatched_v2_64(hipblasHandle_t   handle,
                                                                int64_t           n,
                                                                const hipComplex* alpha,
                                                                hipComplex*       x,
                                                                int64_t           incx,
                                                                hipblasStride     stridex,
                                                                int64_t           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZscalStridedBatched_v2_64(hipblasHandle_t         handle,
                                                                int64_t                 n,
                                                                const hipDoubleComplex* alpha,
                                                                hipDoubleComplex*       x,
                                                                int64_t                 incx,
                                                                hipblasStride           stridex,
                                                                int64_t                 batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsscalStridedBatched_v2_64(hipblasHandle_t handle,
                                                                 int64_t         n,
                                                                 const float*    alpha,
                                                                 hipComplex*     x,
                                                                 int64_t         incx,
                                                                 hipblasStride   stridex,
                                                                 int64_t         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZdscalStridedBatched_v2_64(hipblasHandle_t   handle,
                                                                 int64_t           n,
                                                                 const double*     alpha,
                                                                 hipDoubleComplex* x,
                                                                 int64_t           incx,
                                                                 hipblasStride     stridex,
                                                                 int64_t           batchCount);
//! @}

/*! @{
    \brief BLAS Level 1 API

    \details
    swap  interchanges vectors x and y.

        y := x; x := y

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : s,d,c,z

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    n         [int]
              the number of elements in x and y.
    @param[inout]
    x         device pointer storing vector x.
    @param[in]
    incx      [int]
              specifies the increment for the elements of x.
    @param[inout]
    y         device pointer storing vector y.
    @param[in]
    incy      [int]
              specifies the increment for the elements of y.

            ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t
    hipblasSswap(hipblasHandle_t handle, int n, float* x, int incx, float* y, int incy);

HIPBLAS_EXPORT hipblasStatus_t
    hipblasDswap(hipblasHandle_t handle, int n, double* x, int incx, double* y, int incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasCswap(
    hipblasHandle_t handle, int n, hipblasComplex* x, int incx, hipblasComplex* y, int incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasZswap(hipblasHandle_t       handle,
                                            int                   n,
                                            hipblasDoubleComplex* x,
                                            int                   incx,
                                            hipblasDoubleComplex* y,
                                            int                   incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasCswap_v2(
    hipblasHandle_t handle, int n, hipComplex* x, int incx, hipComplex* y, int incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasZswap_v2(
    hipblasHandle_t handle, int n, hipDoubleComplex* x, int incx, hipDoubleComplex* y, int incy);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSswap_64(
    hipblasHandle_t handle, int64_t n, float* x, int64_t incx, float* y, int64_t incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasDswap_64(
    hipblasHandle_t handle, int64_t n, double* x, int64_t incx, double* y, int64_t incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasCswap_64(hipblasHandle_t handle,
                                               int64_t         n,
                                               hipblasComplex* x,
                                               int64_t         incx,
                                               hipblasComplex* y,
                                               int64_t         incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasZswap_64(hipblasHandle_t       handle,
                                               int64_t               n,
                                               hipblasDoubleComplex* x,
                                               int64_t               incx,
                                               hipblasDoubleComplex* y,
                                               int64_t               incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasCswap_v2_64(
    hipblasHandle_t handle, int64_t n, hipComplex* x, int64_t incx, hipComplex* y, int64_t incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasZswap_v2_64(hipblasHandle_t   handle,
                                                  int64_t           n,
                                                  hipDoubleComplex* x,
                                                  int64_t           incx,
                                                  hipDoubleComplex* y,
                                                  int64_t           incy);
//! @}

/*! @{
    \brief BLAS Level 1 API

    \details
    swapBatched interchanges vectors x_i and y_i, for i = 1 , ... , batchCount

        y_i := x_i; x_i := y_i

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    n         [int]
              the number of elements in each x_i and y_i.
    @param[inout]
    x         device array of device pointers storing each vector x_i.
    @param[in]
    incx      [int]
              specifies the increment for the elements of each x_i.
    @param[inout]
    y         device array of device pointers storing each vector y_i.
    @param[in]
    incy      [int]
              specifies the increment for the elements of each y_i.
    @param[in]
    batchCount [int]
                number of instances in the batch.

            ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSswapBatched(hipblasHandle_t handle,
                                                   int             n,
                                                   float* const    x[],
                                                   int             incx,
                                                   float* const    y[],
                                                   int             incy,
                                                   int             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDswapBatched(hipblasHandle_t handle,
                                                   int             n,
                                                   double* const   x[],
                                                   int             incx,
                                                   double* const   y[],
                                                   int             incy,
                                                   int             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCswapBatched(hipblasHandle_t       handle,
                                                   int                   n,
                                                   hipblasComplex* const x[],
                                                   int                   incx,
                                                   hipblasComplex* const y[],
                                                   int                   incy,
                                                   int                   batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZswapBatched(hipblasHandle_t             handle,
                                                   int                         n,
                                                   hipblasDoubleComplex* const x[],
                                                   int                         incx,
                                                   hipblasDoubleComplex* const y[],
                                                   int                         incy,
                                                   int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCswapBatched_v2(hipblasHandle_t   handle,
                                                      int               n,
                                                      hipComplex* const x[],
                                                      int               incx,
                                                      hipComplex* const y[],
                                                      int               incy,
                                                      int               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZswapBatched_v2(hipblasHandle_t         handle,
                                                      int                     n,
                                                      hipDoubleComplex* const x[],
                                                      int                     incx,
                                                      hipDoubleComplex* const y[],
                                                      int                     incy,
                                                      int                     batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSswapBatched_64(hipblasHandle_t handle,
                                                      int64_t         n,
                                                      float* const    x[],
                                                      int64_t         incx,
                                                      float* const    y[],
                                                      int64_t         incy,
                                                      int64_t         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDswapBatched_64(hipblasHandle_t handle,
                                                      int64_t         n,
                                                      double* const   x[],
                                                      int64_t         incx,
                                                      double* const   y[],
                                                      int64_t         incy,
                                                      int64_t         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCswapBatched_64(hipblasHandle_t       handle,
                                                      int64_t               n,
                                                      hipblasComplex* const x[],
                                                      int64_t               incx,
                                                      hipblasComplex* const y[],
                                                      int64_t               incy,
                                                      int64_t               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZswapBatched_64(hipblasHandle_t             handle,
                                                      int64_t                     n,
                                                      hipblasDoubleComplex* const x[],
                                                      int64_t                     incx,
                                                      hipblasDoubleComplex* const y[],
                                                      int64_t                     incy,
                                                      int64_t                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCswapBatched_v2_64(hipblasHandle_t   handle,
                                                         int64_t           n,
                                                         hipComplex* const x[],
                                                         int64_t           incx,
                                                         hipComplex* const y[],
                                                         int64_t           incy,
                                                         int64_t           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZswapBatched_v2_64(hipblasHandle_t         handle,
                                                         int64_t                 n,
                                                         hipDoubleComplex* const x[],
                                                         int64_t                 incx,
                                                         hipDoubleComplex* const y[],
                                                         int64_t                 incy,
                                                         int64_t                 batchCount);
//! @}

/*! @{
    \brief BLAS Level 1 API

    \details
    swapStridedBatched interchanges vectors x_i and y_i, for i = 1 , ... , batchCount

        y_i := x_i; x_i := y_i

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    n         [int]
              the number of elements in each x_i and y_i.
    @param[inout]
    x         device pointer to the first vector x_1.
    @param[in]
    incx      [int]
              specifies the increment for the elements of x.
    @param[in]
    stridex   [hipblasStride]
              stride from the start of one vector (x_i) and the next one (x_i+1).
              There are no restrictions placed on stride_x, however the user should
              take care to ensure that stride_x is of appropriate size, for a typical
              case this means stride_x >= n * incx.
    @param[inout]
    y         device pointer to the first vector y_1.
    @param[in]
    incy      [int]
              specifies the increment for the elements of y.
    @param[in]
    stridey   [hipblasStride]
              stride from the start of one vector (y_i) and the next one (y_i+1).
              There are no restrictions placed on stride_x, however the user should
              take care to ensure that stride_y is of appropriate size, for a typical
              case this means stride_y >= n * incy. stridey should be non zero.
     @param[in]
     batchCount [int]
                 number of instances in the batch.

            ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSswapStridedBatched(hipblasHandle_t handle,
                                                          int             n,
                                                          float*          x,
                                                          int             incx,
                                                          hipblasStride   stridex,
                                                          float*          y,
                                                          int             incy,
                                                          hipblasStride   stridey,
                                                          int             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDswapStridedBatched(hipblasHandle_t handle,
                                                          int             n,
                                                          double*         x,
                                                          int             incx,
                                                          hipblasStride   stridex,
                                                          double*         y,
                                                          int             incy,
                                                          hipblasStride   stridey,
                                                          int             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCswapStridedBatched(hipblasHandle_t handle,
                                                          int             n,
                                                          hipblasComplex* x,
                                                          int             incx,
                                                          hipblasStride   stridex,
                                                          hipblasComplex* y,
                                                          int             incy,
                                                          hipblasStride   stridey,
                                                          int             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZswapStridedBatched(hipblasHandle_t       handle,
                                                          int                   n,
                                                          hipblasDoubleComplex* x,
                                                          int                   incx,
                                                          hipblasStride         stridex,
                                                          hipblasDoubleComplex* y,
                                                          int                   incy,
                                                          hipblasStride         stridey,
                                                          int                   batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCswapStridedBatched_v2(hipblasHandle_t handle,
                                                             int             n,
                                                             hipComplex*     x,
                                                             int             incx,
                                                             hipblasStride   stridex,
                                                             hipComplex*     y,
                                                             int             incy,
                                                             hipblasStride   stridey,
                                                             int             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZswapStridedBatched_v2(hipblasHandle_t   handle,
                                                             int               n,
                                                             hipDoubleComplex* x,
                                                             int               incx,
                                                             hipblasStride     stridex,
                                                             hipDoubleComplex* y,
                                                             int               incy,
                                                             hipblasStride     stridey,
                                                             int               batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSswapStridedBatched_64(hipblasHandle_t handle,
                                                             int64_t         n,
                                                             float*          x,
                                                             int64_t         incx,
                                                             hipblasStride   stridex,
                                                             float*          y,
                                                             int64_t         incy,
                                                             hipblasStride   stridey,
                                                             int64_t         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDswapStridedBatched_64(hipblasHandle_t handle,
                                                             int64_t         n,
                                                             double*         x,
                                                             int64_t         incx,
                                                             hipblasStride   stridex,
                                                             double*         y,
                                                             int64_t         incy,
                                                             hipblasStride   stridey,
                                                             int64_t         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCswapStridedBatched_64(hipblasHandle_t handle,
                                                             int64_t         n,
                                                             hipblasComplex* x,
                                                             int64_t         incx,
                                                             hipblasStride   stridex,
                                                             hipblasComplex* y,
                                                             int64_t         incy,
                                                             hipblasStride   stridey,
                                                             int64_t         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZswapStridedBatched_64(hipblasHandle_t       handle,
                                                             int64_t               n,
                                                             hipblasDoubleComplex* x,
                                                             int64_t               incx,
                                                             hipblasStride         stridex,
                                                             hipblasDoubleComplex* y,
                                                             int64_t               incy,
                                                             hipblasStride         stridey,
                                                             int64_t               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCswapStridedBatched_v2_64(hipblasHandle_t handle,
                                                                int64_t         n,
                                                                hipComplex*     x,
                                                                int64_t         incx,
                                                                hipblasStride   stridex,
                                                                hipComplex*     y,
                                                                int64_t         incy,
                                                                hipblasStride   stridey,
                                                                int64_t         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZswapStridedBatched_v2_64(hipblasHandle_t   handle,
                                                                int64_t           n,
                                                                hipDoubleComplex* x,
                                                                int64_t           incx,
                                                                hipblasStride     stridex,
                                                                hipDoubleComplex* y,
                                                                int64_t           incy,
                                                                hipblasStride     stridey,
                                                                int64_t           batchCount);
//! @}

/*
 * ===========================================================================
 *    level 2 BLAS
 * ===========================================================================
 */

/*! @{
    \brief BLAS Level 2 API

    \details
    gbmv performs one of the matrix-vector operations

        y := alpha*A*x    + beta*y,   or
        y := alpha*A**T*x + beta*y,   or
        y := alpha*A**H*x + beta*y,

    where alpha and beta are scalars, x and y are vectors and A is an
    m by n banded matrix with kl sub-diagonals and ku super-diagonals.

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : s,d,c,z

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    trans     [hipblasOperation_t]
              indicates whether matrix A is tranposed (conjugated) or not
    @param[in]
    m         [int]
              number of rows of matrix A
    @param[in]
    n         [int]
              number of columns of matrix A
    @param[in]
    kl        [int]
              number of sub-diagonals of A
    @param[in]
    ku        [int]
              number of super-diagonals of A
    @param[in]
    alpha     device pointer or host pointer to scalar alpha.
    @param[in]
        AP    device pointer storing banded matrix A.
              Leading (kl + ku + 1) by n part of the matrix contains the coefficients
              of the banded matrix. The leading diagonal resides in row (ku + 1) with
              the first super-diagonal above on the RHS of row ku. The first sub-diagonal
              resides below on the LHS of row ku + 2. This propagates up and down across
              sub/super-diagonals.
                Ex: (m = n = 7; ku = 2, kl = 2)
                1 2 3 0 0 0 0             0 0 3 3 3 3 3
                4 1 2 3 0 0 0             0 2 2 2 2 2 2
                5 4 1 2 3 0 0    ---->    1 1 1 1 1 1 1
                0 5 4 1 2 3 0             4 4 4 4 4 4 0
                0 0 5 4 1 2 0             5 5 5 5 5 0 0
                0 0 0 5 4 1 2             0 0 0 0 0 0 0
                0 0 0 0 5 4 1             0 0 0 0 0 0 0
              Note that the empty elements which don't correspond to data will not
              be referenced.
    @param[in]
    lda       [int]
              specifies the leading dimension of A. Must be >= (kl + ku + 1)
    @param[in]
    x         device pointer storing vector x.
    @param[in]
    incx      [int]
              specifies the increment for the elements of x.
    @param[in]
    beta      device pointer or host pointer to scalar beta.
    @param[inout]
    y         device pointer storing vector y.
    @param[in]
    incy      [int]
              specifies the increment for the elements of y.

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSgbmv(hipblasHandle_t    handle,
                                            hipblasOperation_t trans,
                                            int                m,
                                            int                n,
                                            int                kl,
                                            int                ku,
                                            const float*       alpha,
                                            const float*       AP,
                                            int                lda,
                                            const float*       x,
                                            int                incx,
                                            const float*       beta,
                                            float*             y,
                                            int                incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasDgbmv(hipblasHandle_t    handle,
                                            hipblasOperation_t trans,
                                            int                m,
                                            int                n,
                                            int                kl,
                                            int                ku,
                                            const double*      alpha,
                                            const double*      AP,
                                            int                lda,
                                            const double*      x,
                                            int                incx,
                                            const double*      beta,
                                            double*            y,
                                            int                incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgbmv(hipblasHandle_t       handle,
                                            hipblasOperation_t    trans,
                                            int                   m,
                                            int                   n,
                                            int                   kl,
                                            int                   ku,
                                            const hipblasComplex* alpha,
                                            const hipblasComplex* AP,
                                            int                   lda,
                                            const hipblasComplex* x,
                                            int                   incx,
                                            const hipblasComplex* beta,
                                            hipblasComplex*       y,
                                            int                   incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgbmv(hipblasHandle_t             handle,
                                            hipblasOperation_t          trans,
                                            int                         m,
                                            int                         n,
                                            int                         kl,
                                            int                         ku,
                                            const hipblasDoubleComplex* alpha,
                                            const hipblasDoubleComplex* AP,
                                            int                         lda,
                                            const hipblasDoubleComplex* x,
                                            int                         incx,
                                            const hipblasDoubleComplex* beta,
                                            hipblasDoubleComplex*       y,
                                            int                         incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgbmv_v2(hipblasHandle_t    handle,
                                               hipblasOperation_t trans,
                                               int                m,
                                               int                n,
                                               int                kl,
                                               int                ku,
                                               const hipComplex*  alpha,
                                               const hipComplex*  AP,
                                               int                lda,
                                               const hipComplex*  x,
                                               int                incx,
                                               const hipComplex*  beta,
                                               hipComplex*        y,
                                               int                incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgbmv_v2(hipblasHandle_t         handle,
                                               hipblasOperation_t      trans,
                                               int                     m,
                                               int                     n,
                                               int                     kl,
                                               int                     ku,
                                               const hipDoubleComplex* alpha,
                                               const hipDoubleComplex* AP,
                                               int                     lda,
                                               const hipDoubleComplex* x,
                                               int                     incx,
                                               const hipDoubleComplex* beta,
                                               hipDoubleComplex*       y,
                                               int                     incy);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSgbmv_64(hipblasHandle_t    handle,
                                               hipblasOperation_t trans,
                                               int64_t            m,
                                               int64_t            n,
                                               int64_t            kl,
                                               int64_t            ku,
                                               const float*       alpha,
                                               const float*       AP,
                                               int64_t            lda,
                                               const float*       x,
                                               int64_t            incx,
                                               const float*       beta,
                                               float*             y,
                                               int64_t            incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasDgbmv_64(hipblasHandle_t    handle,
                                               hipblasOperation_t trans,
                                               int64_t            m,
                                               int64_t            n,
                                               int64_t            kl,
                                               int64_t            ku,
                                               const double*      alpha,
                                               const double*      AP,
                                               int64_t            lda,
                                               const double*      x,
                                               int64_t            incx,
                                               const double*      beta,
                                               double*            y,
                                               int64_t            incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgbmv_64(hipblasHandle_t       handle,
                                               hipblasOperation_t    trans,
                                               int64_t               m,
                                               int64_t               n,
                                               int64_t               kl,
                                               int64_t               ku,
                                               const hipblasComplex* alpha,
                                               const hipblasComplex* AP,
                                               int64_t               lda,
                                               const hipblasComplex* x,
                                               int64_t               incx,
                                               const hipblasComplex* beta,
                                               hipblasComplex*       y,
                                               int64_t               incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgbmv_64(hipblasHandle_t             handle,
                                               hipblasOperation_t          trans,
                                               int64_t                     m,
                                               int64_t                     n,
                                               int64_t                     kl,
                                               int64_t                     ku,
                                               const hipblasDoubleComplex* alpha,
                                               const hipblasDoubleComplex* AP,
                                               int64_t                     lda,
                                               const hipblasDoubleComplex* x,
                                               int64_t                     incx,
                                               const hipblasDoubleComplex* beta,
                                               hipblasDoubleComplex*       y,
                                               int64_t                     incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgbmv_v2_64(hipblasHandle_t    handle,
                                                  hipblasOperation_t trans,
                                                  int64_t            m,
                                                  int64_t            n,
                                                  int64_t            kl,
                                                  int64_t            ku,
                                                  const hipComplex*  alpha,
                                                  const hipComplex*  AP,
                                                  int64_t            lda,
                                                  const hipComplex*  x,
                                                  int64_t            incx,
                                                  const hipComplex*  beta,
                                                  hipComplex*        y,
                                                  int64_t            incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgbmv_v2_64(hipblasHandle_t         handle,
                                                  hipblasOperation_t      trans,
                                                  int64_t                 m,
                                                  int64_t                 n,
                                                  int64_t                 kl,
                                                  int64_t                 ku,
                                                  const hipDoubleComplex* alpha,
                                                  const hipDoubleComplex* AP,
                                                  int64_t                 lda,
                                                  const hipDoubleComplex* x,
                                                  int64_t                 incx,
                                                  const hipDoubleComplex* beta,
                                                  hipDoubleComplex*       y,
                                                  int64_t                 incy);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    gbmvBatched performs one of the matrix-vector operations

        y_i := alpha*A_i*x_i    + beta*y_i,   or
        y_i := alpha*A_i**T*x_i + beta*y_i,   or
        y_i := alpha*A_i**H*x_i + beta*y_i,

    where (A_i, x_i, y_i) is the i-th instance of the batch.
    alpha and beta are scalars, x_i and y_i are vectors and A_i is an
    m by n banded matrix with kl sub-diagonals and ku super-diagonals,
    for i = 1, ..., batchCount.

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    trans     [hipblasOperation_t]
              indicates whether matrix A is tranposed (conjugated) or not
    @param[in]
    m         [int]
              number of rows of each matrix A_i
    @param[in]
    n         [int]
              number of columns of each matrix A_i
    @param[in]
    kl        [int]
              number of sub-diagonals of each A_i
    @param[in]
    ku        [int]
              number of super-diagonals of each A_i
    @param[in]
    alpha     device pointer or host pointer to scalar alpha.
    @param[in]
        AP    device array of device pointers storing each banded matrix A_i.
              Leading (kl + ku + 1) by n part of the matrix contains the coefficients
              of the banded matrix. The leading diagonal resides in row (ku + 1) with
              the first super-diagonal above on the RHS of row ku. The first sub-diagonal
              resides below on the LHS of row ku + 2. This propagates up and down across
              sub/super-diagonals.
                Ex: (m = n = 7; ku = 2, kl = 2)
                1 2 3 0 0 0 0             0 0 3 3 3 3 3
                4 1 2 3 0 0 0             0 2 2 2 2 2 2
                5 4 1 2 3 0 0    ---->    1 1 1 1 1 1 1
                0 5 4 1 2 3 0             4 4 4 4 4 4 0
                0 0 5 4 1 2 0             5 5 5 5 5 0 0
                0 0 0 5 4 1 2             0 0 0 0 0 0 0
                0 0 0 0 5 4 1             0 0 0 0 0 0 0
              Note that the empty elements which don't correspond to data will not
              be referenced.
    @param[in]
    lda       [int]
              specifies the leading dimension of each A_i. Must be >= (kl + ku + 1)
    @param[in]
    x         device array of device pointers storing each vector x_i.
    @param[in]
    incx      [int]
              specifies the increment for the elements of each x_i.
    @param[in]
    beta      device pointer or host pointer to scalar beta.
    @param[inout]
    y         device array of device pointers storing each vector y_i.
    @param[in]
    incy      [int]
              specifies the increment for the elements of each y_i.
    @param[in]
    batchCount [int]
                specifies the number of instances in the batch.

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSgbmvBatched(hipblasHandle_t    handle,
                                                   hipblasOperation_t trans,
                                                   int                m,
                                                   int                n,
                                                   int                kl,
                                                   int                ku,
                                                   const float*       alpha,
                                                   const float* const AP[],
                                                   int                lda,
                                                   const float* const x[],
                                                   int                incx,
                                                   const float*       beta,
                                                   float* const       y[],
                                                   int                incy,
                                                   int                batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDgbmvBatched(hipblasHandle_t     handle,
                                                   hipblasOperation_t  trans,
                                                   int                 m,
                                                   int                 n,
                                                   int                 kl,
                                                   int                 ku,
                                                   const double*       alpha,
                                                   const double* const AP[],
                                                   int                 lda,
                                                   const double* const x[],
                                                   int                 incx,
                                                   const double*       beta,
                                                   double* const       y[],
                                                   int                 incy,
                                                   int                 batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgbmvBatched(hipblasHandle_t             handle,
                                                   hipblasOperation_t          trans,
                                                   int                         m,
                                                   int                         n,
                                                   int                         kl,
                                                   int                         ku,
                                                   const hipblasComplex*       alpha,
                                                   const hipblasComplex* const AP[],
                                                   int                         lda,
                                                   const hipblasComplex* const x[],
                                                   int                         incx,
                                                   const hipblasComplex*       beta,
                                                   hipblasComplex* const       y[],
                                                   int                         incy,
                                                   int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgbmvBatched(hipblasHandle_t                   handle,
                                                   hipblasOperation_t                trans,
                                                   int                               m,
                                                   int                               n,
                                                   int                               kl,
                                                   int                               ku,
                                                   const hipblasDoubleComplex*       alpha,
                                                   const hipblasDoubleComplex* const AP[],
                                                   int                               lda,
                                                   const hipblasDoubleComplex* const x[],
                                                   int                               incx,
                                                   const hipblasDoubleComplex*       beta,
                                                   hipblasDoubleComplex* const       y[],
                                                   int                               incy,
                                                   int                               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgbmvBatched_v2(hipblasHandle_t         handle,
                                                      hipblasOperation_t      trans,
                                                      int                     m,
                                                      int                     n,
                                                      int                     kl,
                                                      int                     ku,
                                                      const hipComplex*       alpha,
                                                      const hipComplex* const AP[],
                                                      int                     lda,
                                                      const hipComplex* const x[],
                                                      int                     incx,
                                                      const hipComplex*       beta,
                                                      hipComplex* const       y[],
                                                      int                     incy,
                                                      int                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgbmvBatched_v2(hipblasHandle_t               handle,
                                                      hipblasOperation_t            trans,
                                                      int                           m,
                                                      int                           n,
                                                      int                           kl,
                                                      int                           ku,
                                                      const hipDoubleComplex*       alpha,
                                                      const hipDoubleComplex* const AP[],
                                                      int                           lda,
                                                      const hipDoubleComplex* const x[],
                                                      int                           incx,
                                                      const hipDoubleComplex*       beta,
                                                      hipDoubleComplex* const       y[],
                                                      int                           incy,
                                                      int                           batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSgbmvBatched_64(hipblasHandle_t    handle,
                                                      hipblasOperation_t trans,
                                                      int64_t            m,
                                                      int64_t            n,
                                                      int64_t            kl,
                                                      int64_t            ku,
                                                      const float*       alpha,
                                                      const float* const AP[],
                                                      int64_t            lda,
                                                      const float* const x[],
                                                      int64_t            incx,
                                                      const float*       beta,
                                                      float* const       y[],
                                                      int64_t            incy,
                                                      int64_t            batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDgbmvBatched_64(hipblasHandle_t     handle,
                                                      hipblasOperation_t  trans,
                                                      int64_t             m,
                                                      int64_t             n,
                                                      int64_t             kl,
                                                      int64_t             ku,
                                                      const double*       alpha,
                                                      const double* const AP[],
                                                      int64_t             lda,
                                                      const double* const x[],
                                                      int64_t             incx,
                                                      const double*       beta,
                                                      double* const       y[],
                                                      int64_t             incy,
                                                      int64_t             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgbmvBatched_64(hipblasHandle_t             handle,
                                                      hipblasOperation_t          trans,
                                                      int64_t                     m,
                                                      int64_t                     n,
                                                      int64_t                     kl,
                                                      int64_t                     ku,
                                                      const hipblasComplex*       alpha,
                                                      const hipblasComplex* const AP[],
                                                      int64_t                     lda,
                                                      const hipblasComplex* const x[],
                                                      int64_t                     incx,
                                                      const hipblasComplex*       beta,
                                                      hipblasComplex* const       y[],
                                                      int64_t                     incy,
                                                      int64_t                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgbmvBatched_64(hipblasHandle_t                   handle,
                                                      hipblasOperation_t                trans,
                                                      int64_t                           m,
                                                      int64_t                           n,
                                                      int64_t                           kl,
                                                      int64_t                           ku,
                                                      const hipblasDoubleComplex*       alpha,
                                                      const hipblasDoubleComplex* const AP[],
                                                      int64_t                           lda,
                                                      const hipblasDoubleComplex* const x[],
                                                      int64_t                           incx,
                                                      const hipblasDoubleComplex*       beta,
                                                      hipblasDoubleComplex* const       y[],
                                                      int64_t                           incy,
                                                      int64_t                           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgbmvBatched_v2_64(hipblasHandle_t         handle,
                                                         hipblasOperation_t      trans,
                                                         int64_t                 m,
                                                         int64_t                 n,
                                                         int64_t                 kl,
                                                         int64_t                 ku,
                                                         const hipComplex*       alpha,
                                                         const hipComplex* const AP[],
                                                         int64_t                 lda,
                                                         const hipComplex* const x[],
                                                         int64_t                 incx,
                                                         const hipComplex*       beta,
                                                         hipComplex* const       y[],
                                                         int64_t                 incy,
                                                         int64_t                 batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgbmvBatched_v2_64(hipblasHandle_t               handle,
                                                         hipblasOperation_t            trans,
                                                         int64_t                       m,
                                                         int64_t                       n,
                                                         int64_t                       kl,
                                                         int64_t                       ku,
                                                         const hipDoubleComplex*       alpha,
                                                         const hipDoubleComplex* const AP[],
                                                         int64_t                       lda,
                                                         const hipDoubleComplex* const x[],
                                                         int64_t                       incx,
                                                         const hipDoubleComplex*       beta,
                                                         hipDoubleComplex* const       y[],
                                                         int64_t                       incy,
                                                         int64_t                       batchCount);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    gbmvStridedBatched performs one of the matrix-vector operations

        y_i := alpha*A_i*x_i    + beta*y_i,   or
        y_i := alpha*A_i**T*x_i + beta*y_i,   or
        y_i := alpha*A_i**H*x_i + beta*y_i,

    where (A_i, x_i, y_i) is the i-th instance of the batch.
    alpha and beta are scalars, x_i and y_i are vectors and A_i is an
    m by n banded matrix with kl sub-diagonals and ku super-diagonals,
    for i = 1, ..., batchCount.

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    trans     [hipblasOperation_t]
              indicates whether matrix A is tranposed (conjugated) or not
    @param[in]
    m         [int]
              number of rows of matrix A
    @param[in]
    n         [int]
              number of columns of matrix A
    @param[in]
    kl        [int]
              number of sub-diagonals of A
    @param[in]
    ku        [int]
              number of super-diagonals of A
    @param[in]
    alpha     device pointer or host pointer to scalar alpha.
    @param[in]
        AP    device pointer to first banded matrix (A_1).
              Leading (kl + ku + 1) by n part of the matrix contains the coefficients
              of the banded matrix. The leading diagonal resides in row (ku + 1) with
              the first super-diagonal above on the RHS of row ku. The first sub-diagonal
              resides below on the LHS of row ku + 2. This propagates up and down across
              sub/super-diagonals.
                Ex: (m = n = 7; ku = 2, kl = 2)
                1 2 3 0 0 0 0             0 0 3 3 3 3 3
                4 1 2 3 0 0 0             0 2 2 2 2 2 2
                5 4 1 2 3 0 0    ---->    1 1 1 1 1 1 1
                0 5 4 1 2 3 0             4 4 4 4 4 4 0
                0 0 5 4 1 2 0             5 5 5 5 5 0 0
                0 0 0 5 4 1 2             0 0 0 0 0 0 0
                0 0 0 0 5 4 1             0 0 0 0 0 0 0
              Note that the empty elements which don't correspond to data will not
              be referenced.
    @param[in]
    lda       [int]
              specifies the leading dimension of A. Must be >= (kl + ku + 1)
    @param[in]
    strideA  [hipblasStride]
              stride from the start of one matrix (A_i) and the next one (A_i+1)
    @param[in]
    x         device pointer to first vector (x_1).
    @param[in]
    incx      [int]
              specifies the increment for the elements of x.
    @param[in]
    stridex  [hipblasStride]
              stride from the start of one vector (x_i) and the next one (x_i+1)
    @param[in]
    beta      device pointer or host pointer to scalar beta.
    @param[inout]
    y         device pointer to first vector (y_1).
    @param[in]
    incy      [int]
              specifies the increment for the elements of y.
    @param[in]
    stridey  [hipblasStride]
              stride from the start of one vector (y_i) and the next one (x_i+1)
    @param[in]
    batchCount [int]
                specifies the number of instances in the batch.

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSgbmvStridedBatched(hipblasHandle_t    handle,
                                                          hipblasOperation_t trans,
                                                          int                m,
                                                          int                n,
                                                          int                kl,
                                                          int                ku,
                                                          const float*       alpha,
                                                          const float*       AP,
                                                          int                lda,
                                                          hipblasStride      strideA,
                                                          const float*       x,
                                                          int                incx,
                                                          hipblasStride      stridex,
                                                          const float*       beta,
                                                          float*             y,
                                                          int                incy,
                                                          hipblasStride      stridey,
                                                          int                batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDgbmvStridedBatched(hipblasHandle_t    handle,
                                                          hipblasOperation_t trans,
                                                          int                m,
                                                          int                n,
                                                          int                kl,
                                                          int                ku,
                                                          const double*      alpha,
                                                          const double*      AP,
                                                          int                lda,
                                                          hipblasStride      strideA,
                                                          const double*      x,
                                                          int                incx,
                                                          hipblasStride      stridex,
                                                          const double*      beta,
                                                          double*            y,
                                                          int                incy,
                                                          hipblasStride      stridey,
                                                          int                batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgbmvStridedBatched(hipblasHandle_t       handle,
                                                          hipblasOperation_t    trans,
                                                          int                   m,
                                                          int                   n,
                                                          int                   kl,
                                                          int                   ku,
                                                          const hipblasComplex* alpha,
                                                          const hipblasComplex* AP,
                                                          int                   lda,
                                                          hipblasStride         strideA,
                                                          const hipblasComplex* x,
                                                          int                   incx,
                                                          hipblasStride         stridex,
                                                          const hipblasComplex* beta,
                                                          hipblasComplex*       y,
                                                          int                   incy,
                                                          hipblasStride         stridey,
                                                          int                   batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgbmvStridedBatched(hipblasHandle_t             handle,
                                                          hipblasOperation_t          trans,
                                                          int                         m,
                                                          int                         n,
                                                          int                         kl,
                                                          int                         ku,
                                                          const hipblasDoubleComplex* alpha,
                                                          const hipblasDoubleComplex* AP,
                                                          int                         lda,
                                                          hipblasStride               strideA,
                                                          const hipblasDoubleComplex* x,
                                                          int                         incx,
                                                          hipblasStride               stridex,
                                                          const hipblasDoubleComplex* beta,
                                                          hipblasDoubleComplex*       y,
                                                          int                         incy,
                                                          hipblasStride               stridey,
                                                          int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgbmvStridedBatched_v2(hipblasHandle_t    handle,
                                                             hipblasOperation_t trans,
                                                             int                m,
                                                             int                n,
                                                             int                kl,
                                                             int                ku,
                                                             const hipComplex*  alpha,
                                                             const hipComplex*  AP,
                                                             int                lda,
                                                             hipblasStride      strideA,
                                                             const hipComplex*  x,
                                                             int                incx,
                                                             hipblasStride      stridex,
                                                             const hipComplex*  beta,
                                                             hipComplex*        y,
                                                             int                incy,
                                                             hipblasStride      stridey,
                                                             int                batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgbmvStridedBatched_v2(hipblasHandle_t         handle,
                                                             hipblasOperation_t      trans,
                                                             int                     m,
                                                             int                     n,
                                                             int                     kl,
                                                             int                     ku,
                                                             const hipDoubleComplex* alpha,
                                                             const hipDoubleComplex* AP,
                                                             int                     lda,
                                                             hipblasStride           strideA,
                                                             const hipDoubleComplex* x,
                                                             int                     incx,
                                                             hipblasStride           stridex,
                                                             const hipDoubleComplex* beta,
                                                             hipDoubleComplex*       y,
                                                             int                     incy,
                                                             hipblasStride           stridey,
                                                             int                     batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSgbmvStridedBatched_64(hipblasHandle_t    handle,
                                                             hipblasOperation_t trans,
                                                             int64_t            m,
                                                             int64_t            n,
                                                             int64_t            kl,
                                                             int64_t            ku,
                                                             const float*       alpha,
                                                             const float*       AP,
                                                             int64_t            lda,
                                                             hipblasStride      strideA,
                                                             const float*       x,
                                                             int64_t            incx,
                                                             hipblasStride      stridex,
                                                             const float*       beta,
                                                             float*             y,
                                                             int64_t            incy,
                                                             hipblasStride      stridey,
                                                             int64_t            batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDgbmvStridedBatched_64(hipblasHandle_t    handle,
                                                             hipblasOperation_t trans,
                                                             int64_t            m,
                                                             int64_t            n,
                                                             int64_t            kl,
                                                             int64_t            ku,
                                                             const double*      alpha,
                                                             const double*      AP,
                                                             int64_t            lda,
                                                             hipblasStride      strideA,
                                                             const double*      x,
                                                             int64_t            incx,
                                                             hipblasStride      stridex,
                                                             const double*      beta,
                                                             double*            y,
                                                             int64_t            incy,
                                                             hipblasStride      stridey,
                                                             int64_t            batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgbmvStridedBatched_64(hipblasHandle_t       handle,
                                                             hipblasOperation_t    trans,
                                                             int64_t               m,
                                                             int64_t               n,
                                                             int64_t               kl,
                                                             int64_t               ku,
                                                             const hipblasComplex* alpha,
                                                             const hipblasComplex* AP,
                                                             int64_t               lda,
                                                             hipblasStride         strideA,
                                                             const hipblasComplex* x,
                                                             int64_t               incx,
                                                             hipblasStride         stridex,
                                                             const hipblasComplex* beta,
                                                             hipblasComplex*       y,
                                                             int64_t               incy,
                                                             hipblasStride         stridey,
                                                             int64_t               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgbmvStridedBatched_64(hipblasHandle_t             handle,
                                                             hipblasOperation_t          trans,
                                                             int64_t                     m,
                                                             int64_t                     n,
                                                             int64_t                     kl,
                                                             int64_t                     ku,
                                                             const hipblasDoubleComplex* alpha,
                                                             const hipblasDoubleComplex* AP,
                                                             int64_t                     lda,
                                                             hipblasStride               strideA,
                                                             const hipblasDoubleComplex* x,
                                                             int64_t                     incx,
                                                             hipblasStride               stridex,
                                                             const hipblasDoubleComplex* beta,
                                                             hipblasDoubleComplex*       y,
                                                             int64_t                     incy,
                                                             hipblasStride               stridey,
                                                             int64_t batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgbmvStridedBatched_v2_64(hipblasHandle_t    handle,
                                                                hipblasOperation_t trans,
                                                                int64_t            m,
                                                                int64_t            n,
                                                                int64_t            kl,
                                                                int64_t            ku,
                                                                const hipComplex*  alpha,
                                                                const hipComplex*  AP,
                                                                int64_t            lda,
                                                                hipblasStride      strideA,
                                                                const hipComplex*  x,
                                                                int64_t            incx,
                                                                hipblasStride      stridex,
                                                                const hipComplex*  beta,
                                                                hipComplex*        y,
                                                                int64_t            incy,
                                                                hipblasStride      stridey,
                                                                int64_t            batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgbmvStridedBatched_v2_64(hipblasHandle_t         handle,
                                                                hipblasOperation_t      trans,
                                                                int64_t                 m,
                                                                int64_t                 n,
                                                                int64_t                 kl,
                                                                int64_t                 ku,
                                                                const hipDoubleComplex* alpha,
                                                                const hipDoubleComplex* AP,
                                                                int64_t                 lda,
                                                                hipblasStride           strideA,
                                                                const hipDoubleComplex* x,
                                                                int64_t                 incx,
                                                                hipblasStride           stridex,
                                                                const hipDoubleComplex* beta,
                                                                hipDoubleComplex*       y,
                                                                int64_t                 incy,
                                                                hipblasStride           stridey,
                                                                int64_t                 batchCount);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    gemv performs one of the matrix-vector operations

        y := alpha*A*x    + beta*y,   or
        y := alpha*A**T*x + beta*y,   or
        y := alpha*A**H*x + beta*y,

    where alpha and beta are scalars, x and y are vectors and A is an
    m by n matrix.

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : s,d,c,z

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    trans     [hipblasOperation_t]
              indicates whether matrix A is tranposed (conjugated) or not
    @param[in]
    m         [int]
              number of rows of matrix A
    @param[in]
    n         [int]
              number of columns of matrix A
    @param[in]
    alpha     device pointer or host pointer to scalar alpha.
    @param[in]
    AP        device pointer storing matrix A.
    @param[in]
    lda       [int]
              specifies the leading dimension of A.
    @param[in]
    x         device pointer storing vector x.
    @param[in]
    incx      [int]
              specifies the increment for the elements of x.
    @param[in]
    beta      device pointer or host pointer to scalar beta.
    @param[inout]
    y         device pointer storing vector y.
    @param[in]
    incy      [int]
              specifies the increment for the elements of y.

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSgemv(hipblasHandle_t    handle,
                                            hipblasOperation_t trans,
                                            int                m,
                                            int                n,
                                            const float*       alpha,
                                            const float*       AP,
                                            int                lda,
                                            const float*       x,
                                            int                incx,
                                            const float*       beta,
                                            float*             y,
                                            int                incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasDgemv(hipblasHandle_t    handle,
                                            hipblasOperation_t trans,
                                            int                m,
                                            int                n,
                                            const double*      alpha,
                                            const double*      AP,
                                            int                lda,
                                            const double*      x,
                                            int                incx,
                                            const double*      beta,
                                            double*            y,
                                            int                incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgemv(hipblasHandle_t       handle,
                                            hipblasOperation_t    trans,
                                            int                   m,
                                            int                   n,
                                            const hipblasComplex* alpha,
                                            const hipblasComplex* AP,
                                            int                   lda,
                                            const hipblasComplex* x,
                                            int                   incx,
                                            const hipblasComplex* beta,
                                            hipblasComplex*       y,
                                            int                   incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgemv(hipblasHandle_t             handle,
                                            hipblasOperation_t          trans,
                                            int                         m,
                                            int                         n,
                                            const hipblasDoubleComplex* alpha,
                                            const hipblasDoubleComplex* AP,
                                            int                         lda,
                                            const hipblasDoubleComplex* x,
                                            int                         incx,
                                            const hipblasDoubleComplex* beta,
                                            hipblasDoubleComplex*       y,
                                            int                         incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgemv_v2(hipblasHandle_t    handle,
                                               hipblasOperation_t trans,
                                               int                m,
                                               int                n,
                                               const hipComplex*  alpha,
                                               const hipComplex*  AP,
                                               int                lda,
                                               const hipComplex*  x,
                                               int                incx,
                                               const hipComplex*  beta,
                                               hipComplex*        y,
                                               int                incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgemv_v2(hipblasHandle_t         handle,
                                               hipblasOperation_t      trans,
                                               int                     m,
                                               int                     n,
                                               const hipDoubleComplex* alpha,
                                               const hipDoubleComplex* AP,
                                               int                     lda,
                                               const hipDoubleComplex* x,
                                               int                     incx,
                                               const hipDoubleComplex* beta,
                                               hipDoubleComplex*       y,
                                               int                     incy);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSgemv_64(hipblasHandle_t    handle,
                                               hipblasOperation_t trans,
                                               int64_t            m,
                                               int64_t            n,
                                               const float*       alpha,
                                               const float*       AP,
                                               int64_t            lda,
                                               const float*       x,
                                               int64_t            incx,
                                               const float*       beta,
                                               float*             y,
                                               int64_t            incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasDgemv_64(hipblasHandle_t    handle,
                                               hipblasOperation_t trans,
                                               int64_t            m,
                                               int64_t            n,
                                               const double*      alpha,
                                               const double*      AP,
                                               int64_t            lda,
                                               const double*      x,
                                               int64_t            incx,
                                               const double*      beta,
                                               double*            y,
                                               int64_t            incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgemv_64(hipblasHandle_t       handle,
                                               hipblasOperation_t    trans,
                                               int64_t               m,
                                               int64_t               n,
                                               const hipblasComplex* alpha,
                                               const hipblasComplex* AP,
                                               int64_t               lda,
                                               const hipblasComplex* x,
                                               int64_t               incx,
                                               const hipblasComplex* beta,
                                               hipblasComplex*       y,
                                               int64_t               incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgemv_64(hipblasHandle_t             handle,
                                               hipblasOperation_t          trans,
                                               int64_t                     m,
                                               int64_t                     n,
                                               const hipblasDoubleComplex* alpha,
                                               const hipblasDoubleComplex* AP,
                                               int64_t                     lda,
                                               const hipblasDoubleComplex* x,
                                               int64_t                     incx,
                                               const hipblasDoubleComplex* beta,
                                               hipblasDoubleComplex*       y,
                                               int64_t                     incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgemv_v2_64(hipblasHandle_t    handle,
                                                  hipblasOperation_t trans,
                                                  int64_t            m,
                                                  int64_t            n,
                                                  const hipComplex*  alpha,
                                                  const hipComplex*  AP,
                                                  int64_t            lda,
                                                  const hipComplex*  x,
                                                  int64_t            incx,
                                                  const hipComplex*  beta,
                                                  hipComplex*        y,
                                                  int64_t            incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgemv_v2_64(hipblasHandle_t         handle,
                                                  hipblasOperation_t      trans,
                                                  int64_t                 m,
                                                  int64_t                 n,
                                                  const hipDoubleComplex* alpha,
                                                  const hipDoubleComplex* AP,
                                                  int64_t                 lda,
                                                  const hipDoubleComplex* x,
                                                  int64_t                 incx,
                                                  const hipDoubleComplex* beta,
                                                  hipDoubleComplex*       y,
                                                  int64_t                 incy);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    gemvBatched performs a batch of matrix-vector operations

        y_i := alpha*A_i*x_i    + beta*y_i,   or
        y_i := alpha*A_i**T*x_i + beta*y_i,   or
        y_i := alpha*A_i**H*x_i + beta*y_i,

    where (A_i, x_i, y_i) is the i-th instance of the batch.
    alpha and beta are scalars, x_i and y_i are vectors and A_i is an
    m by n matrix, for i = 1, ..., batchCount.

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle      [hipblasHandle_t]
                handle to the hipblas library context queue.
    @param[in]
    trans       [hipblasOperation_t]
                indicates whether matrices A_i are tranposed (conjugated) or not
    @param[in]
    m           [int]
                number of rows of each matrix A_i
    @param[in]
    n           [int]
                number of columns of each matrix A_i
    @param[in]
    alpha       device pointer or host pointer to scalar alpha.
    @param[in]
    AP         device array of device pointers storing each matrix A_i.
    @param[in]
    lda         [int]
                specifies the leading dimension of each matrix A_i.
    @param[in]
    x           device array of device pointers storing each vector x_i.
    @param[in]
    incx        [int]
                specifies the increment for the elements of each vector x_i.
    @param[in]
    beta        device pointer or host pointer to scalar beta.
    @param[inout]
    y           device array of device pointers storing each vector y_i.
    @param[in]
    incy        [int]
                specifies the increment for the elements of each vector y_i.
    @param[in]
    batchCount [int]
                number of instances in the batch

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSgemvBatched(hipblasHandle_t    handle,
                                                   hipblasOperation_t trans,
                                                   int                m,
                                                   int                n,
                                                   const float*       alpha,
                                                   const float* const AP[],
                                                   int                lda,
                                                   const float* const x[],
                                                   int                incx,
                                                   const float*       beta,
                                                   float* const       y[],
                                                   int                incy,
                                                   int                batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDgemvBatched(hipblasHandle_t     handle,
                                                   hipblasOperation_t  trans,
                                                   int                 m,
                                                   int                 n,
                                                   const double*       alpha,
                                                   const double* const AP[],
                                                   int                 lda,
                                                   const double* const x[],
                                                   int                 incx,
                                                   const double*       beta,
                                                   double* const       y[],
                                                   int                 incy,
                                                   int                 batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgemvBatched(hipblasHandle_t             handle,
                                                   hipblasOperation_t          trans,
                                                   int                         m,
                                                   int                         n,
                                                   const hipblasComplex*       alpha,
                                                   const hipblasComplex* const AP[],
                                                   int                         lda,
                                                   const hipblasComplex* const x[],
                                                   int                         incx,
                                                   const hipblasComplex*       beta,
                                                   hipblasComplex* const       y[],
                                                   int                         incy,
                                                   int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgemvBatched(hipblasHandle_t                   handle,
                                                   hipblasOperation_t                trans,
                                                   int                               m,
                                                   int                               n,
                                                   const hipblasDoubleComplex*       alpha,
                                                   const hipblasDoubleComplex* const AP[],
                                                   int                               lda,
                                                   const hipblasDoubleComplex* const x[],
                                                   int                               incx,
                                                   const hipblasDoubleComplex*       beta,
                                                   hipblasDoubleComplex* const       y[],
                                                   int                               incy,
                                                   int                               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgemvBatched_v2(hipblasHandle_t         handle,
                                                      hipblasOperation_t      trans,
                                                      int                     m,
                                                      int                     n,
                                                      const hipComplex*       alpha,
                                                      const hipComplex* const AP[],
                                                      int                     lda,
                                                      const hipComplex* const x[],
                                                      int                     incx,
                                                      const hipComplex*       beta,
                                                      hipComplex* const       y[],
                                                      int                     incy,
                                                      int                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgemvBatched_v2(hipblasHandle_t               handle,
                                                      hipblasOperation_t            trans,
                                                      int                           m,
                                                      int                           n,
                                                      const hipDoubleComplex*       alpha,
                                                      const hipDoubleComplex* const AP[],
                                                      int                           lda,
                                                      const hipDoubleComplex* const x[],
                                                      int                           incx,
                                                      const hipDoubleComplex*       beta,
                                                      hipDoubleComplex* const       y[],
                                                      int                           incy,
                                                      int                           batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSgemvBatched_64(hipblasHandle_t    handle,
                                                      hipblasOperation_t trans,
                                                      int64_t            m,
                                                      int64_t            n,
                                                      const float*       alpha,
                                                      const float* const AP[],
                                                      int64_t            lda,
                                                      const float* const x[],
                                                      int64_t            incx,
                                                      const float*       beta,
                                                      float* const       y[],
                                                      int64_t            incy,
                                                      int64_t            batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDgemvBatched_64(hipblasHandle_t     handle,
                                                      hipblasOperation_t  trans,
                                                      int64_t             m,
                                                      int64_t             n,
                                                      const double*       alpha,
                                                      const double* const AP[],
                                                      int64_t             lda,
                                                      const double* const x[],
                                                      int64_t             incx,
                                                      const double*       beta,
                                                      double* const       y[],
                                                      int64_t             incy,
                                                      int64_t             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgemvBatched_64(hipblasHandle_t             handle,
                                                      hipblasOperation_t          trans,
                                                      int64_t                     m,
                                                      int64_t                     n,
                                                      const hipblasComplex*       alpha,
                                                      const hipblasComplex* const AP[],
                                                      int64_t                     lda,
                                                      const hipblasComplex* const x[],
                                                      int64_t                     incx,
                                                      const hipblasComplex*       beta,
                                                      hipblasComplex* const       y[],
                                                      int64_t                     incy,
                                                      int64_t                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgemvBatched_64(hipblasHandle_t                   handle,
                                                      hipblasOperation_t                trans,
                                                      int64_t                           m,
                                                      int64_t                           n,
                                                      const hipblasDoubleComplex*       alpha,
                                                      const hipblasDoubleComplex* const AP[],
                                                      int64_t                           lda,
                                                      const hipblasDoubleComplex* const x[],
                                                      int64_t                           incx,
                                                      const hipblasDoubleComplex*       beta,
                                                      hipblasDoubleComplex* const       y[],
                                                      int64_t                           incy,
                                                      int64_t                           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgemvBatched_v2_64(hipblasHandle_t         handle,
                                                         hipblasOperation_t      trans,
                                                         int64_t                 m,
                                                         int64_t                 n,
                                                         const hipComplex*       alpha,
                                                         const hipComplex* const AP[],
                                                         int64_t                 lda,
                                                         const hipComplex* const x[],
                                                         int64_t                 incx,
                                                         const hipComplex*       beta,
                                                         hipComplex* const       y[],
                                                         int64_t                 incy,
                                                         int64_t                 batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgemvBatched_v2_64(hipblasHandle_t               handle,
                                                         hipblasOperation_t            trans,
                                                         int64_t                       m,
                                                         int64_t                       n,
                                                         const hipDoubleComplex*       alpha,
                                                         const hipDoubleComplex* const AP[],
                                                         int64_t                       lda,
                                                         const hipDoubleComplex* const x[],
                                                         int64_t                       incx,
                                                         const hipDoubleComplex*       beta,
                                                         hipDoubleComplex* const       y[],
                                                         int64_t                       incy,
                                                         int64_t                       batchCount);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    gemvStridedBatched performs a batch of matrix-vector operations

        y_i := alpha*A_i*x_i    + beta*y_i,   or
        y_i := alpha*A_i**T*x_i + beta*y_i,   or
        y_i := alpha*A_i**H*x_i + beta*y_i,

    where (A_i, x_i, y_i) is the i-th instance of the batch.
    alpha and beta are scalars, x_i and y_i are vectors and A_i is an
    m by n matrix, for i = 1, ..., batchCount.

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle      [hipblasHandle_t]
                handle to the hipblas library context queue.
    @param[in]
    transA      [hipblasOperation_t]
                indicates whether matrices A_i are tranposed (conjugated) or not
    @param[in]
    m           [int]
                number of rows of matrices A_i
    @param[in]
    n           [int]
                number of columns of matrices A_i
    @param[in]
    alpha       device pointer or host pointer to scalar alpha.
    @param[in]
    AP          device pointer to the first matrix (A_1) in the batch.
    @param[in]
    lda         [int]
                specifies the leading dimension of matrices A_i.
    @param[in]
    strideA     [hipblasStride]
                stride from the start of one matrix (A_i) and the next one (A_i+1)
    @param[in]
    x           device pointer to the first vector (x_1) in the batch.
    @param[in]
    incx        [int]
                specifies the increment for the elements of vectors x_i.
    @param[in]
    stridex     [hipblasStride]
                stride from the start of one vector (x_i) and the next one (x_i+1).
                There are no restrictions placed on stridex, however the user should
                take care to ensure that stridex is of appropriate size. When trans equals HIPBLAS_OP_N
                this typically means stridex >= n * incx, otherwise stridex >= m * incx.
    @param[in]
    beta        device pointer or host pointer to scalar beta.
    @param[inout]
    y           device pointer to the first vector (y_1) in the batch.
    @param[in]
    incy        [int]
                specifies the increment for the elements of vectors y_i.
    @param[in]
    stridey     [hipblasStride]
                stride from the start of one vector (y_i) and the next one (y_i+1).
                There are no restrictions placed on stridey, however the user should
                take care to ensure that stridey is of appropriate size. When trans equals HIPBLAS_OP_N
                this typically means stridey >= m * incy, otherwise stridey >= n * incy. stridey should be non zero.
    @param[in]
    batchCount [int]
                number of instances in the batch

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSgemvStridedBatched(hipblasHandle_t    handle,
                                                          hipblasOperation_t transA,
                                                          int                m,
                                                          int                n,
                                                          const float*       alpha,
                                                          const float*       AP,
                                                          int                lda,
                                                          hipblasStride      strideA,
                                                          const float*       x,
                                                          int                incx,
                                                          hipblasStride      stridex,
                                                          const float*       beta,
                                                          float*             y,
                                                          int                incy,
                                                          hipblasStride      stridey,
                                                          int                batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDgemvStridedBatched(hipblasHandle_t    handle,
                                                          hipblasOperation_t transA,
                                                          int                m,
                                                          int                n,
                                                          const double*      alpha,
                                                          const double*      AP,
                                                          int                lda,
                                                          hipblasStride      strideA,
                                                          const double*      x,
                                                          int                incx,
                                                          hipblasStride      stridex,
                                                          const double*      beta,
                                                          double*            y,
                                                          int                incy,
                                                          hipblasStride      stridey,
                                                          int                batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgemvStridedBatched(hipblasHandle_t       handle,
                                                          hipblasOperation_t    transA,
                                                          int                   m,
                                                          int                   n,
                                                          const hipblasComplex* alpha,
                                                          const hipblasComplex* AP,
                                                          int                   lda,
                                                          hipblasStride         strideA,
                                                          const hipblasComplex* x,
                                                          int                   incx,
                                                          hipblasStride         stridex,
                                                          const hipblasComplex* beta,
                                                          hipblasComplex*       y,
                                                          int                   incy,
                                                          hipblasStride         stridey,
                                                          int                   batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgemvStridedBatched(hipblasHandle_t             handle,
                                                          hipblasOperation_t          transA,
                                                          int                         m,
                                                          int                         n,
                                                          const hipblasDoubleComplex* alpha,
                                                          const hipblasDoubleComplex* AP,
                                                          int                         lda,
                                                          hipblasStride               strideA,
                                                          const hipblasDoubleComplex* x,
                                                          int                         incx,
                                                          hipblasStride               stridex,
                                                          const hipblasDoubleComplex* beta,
                                                          hipblasDoubleComplex*       y,
                                                          int                         incy,
                                                          hipblasStride               stridey,
                                                          int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgemvStridedBatched_v2(hipblasHandle_t    handle,
                                                             hipblasOperation_t transA,
                                                             int                m,
                                                             int                n,
                                                             const hipComplex*  alpha,
                                                             const hipComplex*  AP,
                                                             int                lda,
                                                             hipblasStride      strideA,
                                                             const hipComplex*  x,
                                                             int                incx,
                                                             hipblasStride      stridex,
                                                             const hipComplex*  beta,
                                                             hipComplex*        y,
                                                             int                incy,
                                                             hipblasStride      stridey,
                                                             int                batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgemvStridedBatched_v2(hipblasHandle_t         handle,
                                                             hipblasOperation_t      transA,
                                                             int                     m,
                                                             int                     n,
                                                             const hipDoubleComplex* alpha,
                                                             const hipDoubleComplex* AP,
                                                             int                     lda,
                                                             hipblasStride           strideA,
                                                             const hipDoubleComplex* x,
                                                             int                     incx,
                                                             hipblasStride           stridex,
                                                             const hipDoubleComplex* beta,
                                                             hipDoubleComplex*       y,
                                                             int                     incy,
                                                             hipblasStride           stridey,
                                                             int                     batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSgemvStridedBatched_64(hipblasHandle_t    handle,
                                                             hipblasOperation_t transA,
                                                             int64_t            m,
                                                             int64_t            n,
                                                             const float*       alpha,
                                                             const float*       AP,
                                                             int64_t            lda,
                                                             hipblasStride      strideA,
                                                             const float*       x,
                                                             int64_t            incx,
                                                             hipblasStride      stridex,
                                                             const float*       beta,
                                                             float*             y,
                                                             int64_t            incy,
                                                             hipblasStride      stridey,
                                                             int64_t            batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDgemvStridedBatched_64(hipblasHandle_t    handle,
                                                             hipblasOperation_t transA,
                                                             int64_t            m,
                                                             int64_t            n,
                                                             const double*      alpha,
                                                             const double*      AP,
                                                             int64_t            lda,
                                                             hipblasStride      strideA,
                                                             const double*      x,
                                                             int64_t            incx,
                                                             hipblasStride      stridex,
                                                             const double*      beta,
                                                             double*            y,
                                                             int64_t            incy,
                                                             hipblasStride      stridey,
                                                             int64_t            batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgemvStridedBatched_64(hipblasHandle_t       handle,
                                                             hipblasOperation_t    transA,
                                                             int64_t               m,
                                                             int64_t               n,
                                                             const hipblasComplex* alpha,
                                                             const hipblasComplex* AP,
                                                             int64_t               lda,
                                                             hipblasStride         strideA,
                                                             const hipblasComplex* x,
                                                             int64_t               incx,
                                                             hipblasStride         stridex,
                                                             const hipblasComplex* beta,
                                                             hipblasComplex*       y,
                                                             int64_t               incy,
                                                             hipblasStride         stridey,
                                                             int64_t               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgemvStridedBatched_64(hipblasHandle_t             handle,
                                                             hipblasOperation_t          transA,
                                                             int64_t                     m,
                                                             int64_t                     n,
                                                             const hipblasDoubleComplex* alpha,
                                                             const hipblasDoubleComplex* AP,
                                                             int64_t                     lda,
                                                             hipblasStride               strideA,
                                                             const hipblasDoubleComplex* x,
                                                             int64_t                     incx,
                                                             hipblasStride               stridex,
                                                             const hipblasDoubleComplex* beta,
                                                             hipblasDoubleComplex*       y,
                                                             int64_t                     incy,
                                                             hipblasStride               stridey,
                                                             int64_t batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgemvStridedBatched_v2_64(hipblasHandle_t    handle,
                                                                hipblasOperation_t transA,
                                                                int64_t            m,
                                                                int64_t            n,
                                                                const hipComplex*  alpha,
                                                                const hipComplex*  AP,
                                                                int64_t            lda,
                                                                hipblasStride      strideA,
                                                                const hipComplex*  x,
                                                                int64_t            incx,
                                                                hipblasStride      stridex,
                                                                const hipComplex*  beta,
                                                                hipComplex*        y,
                                                                int64_t            incy,
                                                                hipblasStride      stridey,
                                                                int64_t            batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgemvStridedBatched_v2_64(hipblasHandle_t         handle,
                                                                hipblasOperation_t      transA,
                                                                int64_t                 m,
                                                                int64_t                 n,
                                                                const hipDoubleComplex* alpha,
                                                                const hipDoubleComplex* AP,
                                                                int64_t                 lda,
                                                                hipblasStride           strideA,
                                                                const hipDoubleComplex* x,
                                                                int64_t                 incx,
                                                                hipblasStride           stridex,
                                                                const hipDoubleComplex* beta,
                                                                hipDoubleComplex*       y,
                                                                int64_t                 incy,
                                                                hipblasStride           stridey,
                                                                int64_t                 batchCount);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    ger,geru,gerc performs the matrix-vector operations

        A := A + alpha*x*y**T , OR
        A := A + alpha*x*y**H for gerc

    where alpha is a scalar, x and y are vectors, and A is an
    m by n matrix.

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : s,d,c,z

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    m         [int]
              the number of rows of the matrix A.
    @param[in]
    n         [int]
              the number of columns of the matrix A.
    @param[in]
    alpha
              device pointer or host pointer to scalar alpha.
    @param[in]
    x         device pointer storing vector x.
    @param[in]
    incx      [int]
              specifies the increment for the elements of x.
    @param[in]
    y         device pointer storing vector y.
    @param[in]
    incy      [int]
              specifies the increment for the elements of y.
    @param[inout]
    AP         device pointer storing matrix A.
    @param[in]
    lda       [int]
              specifies the leading dimension of A.

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSger(hipblasHandle_t handle,
                                           int             m,
                                           int             n,
                                           const float*    alpha,
                                           const float*    x,
                                           int             incx,
                                           const float*    y,
                                           int             incy,
                                           float*          AP,
                                           int             lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasDger(hipblasHandle_t handle,
                                           int             m,
                                           int             n,
                                           const double*   alpha,
                                           const double*   x,
                                           int             incx,
                                           const double*   y,
                                           int             incy,
                                           double*         AP,
                                           int             lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgeru(hipblasHandle_t       handle,
                                            int                   m,
                                            int                   n,
                                            const hipblasComplex* alpha,
                                            const hipblasComplex* x,
                                            int                   incx,
                                            const hipblasComplex* y,
                                            int                   incy,
                                            hipblasComplex*       AP,
                                            int                   lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgerc(hipblasHandle_t       handle,
                                            int                   m,
                                            int                   n,
                                            const hipblasComplex* alpha,
                                            const hipblasComplex* x,
                                            int                   incx,
                                            const hipblasComplex* y,
                                            int                   incy,
                                            hipblasComplex*       AP,
                                            int                   lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgeru(hipblasHandle_t             handle,
                                            int                         m,
                                            int                         n,
                                            const hipblasDoubleComplex* alpha,
                                            const hipblasDoubleComplex* x,
                                            int                         incx,
                                            const hipblasDoubleComplex* y,
                                            int                         incy,
                                            hipblasDoubleComplex*       AP,
                                            int                         lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgerc(hipblasHandle_t             handle,
                                            int                         m,
                                            int                         n,
                                            const hipblasDoubleComplex* alpha,
                                            const hipblasDoubleComplex* x,
                                            int                         incx,
                                            const hipblasDoubleComplex* y,
                                            int                         incy,
                                            hipblasDoubleComplex*       AP,
                                            int                         lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgeru_v2(hipblasHandle_t   handle,
                                               int               m,
                                               int               n,
                                               const hipComplex* alpha,
                                               const hipComplex* x,
                                               int               incx,
                                               const hipComplex* y,
                                               int               incy,
                                               hipComplex*       AP,
                                               int               lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgerc_v2(hipblasHandle_t   handle,
                                               int               m,
                                               int               n,
                                               const hipComplex* alpha,
                                               const hipComplex* x,
                                               int               incx,
                                               const hipComplex* y,
                                               int               incy,
                                               hipComplex*       AP,
                                               int               lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgeru_v2(hipblasHandle_t         handle,
                                               int                     m,
                                               int                     n,
                                               const hipDoubleComplex* alpha,
                                               const hipDoubleComplex* x,
                                               int                     incx,
                                               const hipDoubleComplex* y,
                                               int                     incy,
                                               hipDoubleComplex*       AP,
                                               int                     lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgerc_v2(hipblasHandle_t         handle,
                                               int                     m,
                                               int                     n,
                                               const hipDoubleComplex* alpha,
                                               const hipDoubleComplex* x,
                                               int                     incx,
                                               const hipDoubleComplex* y,
                                               int                     incy,
                                               hipDoubleComplex*       AP,
                                               int                     lda);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSger_64(hipblasHandle_t handle,
                                              int64_t         m,
                                              int64_t         n,
                                              const float*    alpha,
                                              const float*    x,
                                              int64_t         incx,
                                              const float*    y,
                                              int64_t         incy,
                                              float*          AP,
                                              int64_t         lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasDger_64(hipblasHandle_t handle,
                                              int64_t         m,
                                              int64_t         n,
                                              const double*   alpha,
                                              const double*   x,
                                              int64_t         incx,
                                              const double*   y,
                                              int64_t         incy,
                                              double*         AP,
                                              int64_t         lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgeru_64(hipblasHandle_t       handle,
                                               int64_t               m,
                                               int64_t               n,
                                               const hipblasComplex* alpha,
                                               const hipblasComplex* x,
                                               int64_t               incx,
                                               const hipblasComplex* y,
                                               int64_t               incy,
                                               hipblasComplex*       AP,
                                               int64_t               lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgerc_64(hipblasHandle_t       handle,
                                               int64_t               m,
                                               int64_t               n,
                                               const hipblasComplex* alpha,
                                               const hipblasComplex* x,
                                               int64_t               incx,
                                               const hipblasComplex* y,
                                               int64_t               incy,
                                               hipblasComplex*       AP,
                                               int64_t               lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgeru_64(hipblasHandle_t             handle,
                                               int64_t                     m,
                                               int64_t                     n,
                                               const hipblasDoubleComplex* alpha,
                                               const hipblasDoubleComplex* x,
                                               int64_t                     incx,
                                               const hipblasDoubleComplex* y,
                                               int64_t                     incy,
                                               hipblasDoubleComplex*       AP,
                                               int64_t                     lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgerc_64(hipblasHandle_t             handle,
                                               int64_t                     m,
                                               int64_t                     n,
                                               const hipblasDoubleComplex* alpha,
                                               const hipblasDoubleComplex* x,
                                               int64_t                     incx,
                                               const hipblasDoubleComplex* y,
                                               int64_t                     incy,
                                               hipblasDoubleComplex*       AP,
                                               int64_t                     lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgeru_v2_64(hipblasHandle_t   handle,
                                                  int64_t           m,
                                                  int64_t           n,
                                                  const hipComplex* alpha,
                                                  const hipComplex* x,
                                                  int64_t           incx,
                                                  const hipComplex* y,
                                                  int64_t           incy,
                                                  hipComplex*       AP,
                                                  int64_t           lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgerc_v2_64(hipblasHandle_t   handle,
                                                  int64_t           m,
                                                  int64_t           n,
                                                  const hipComplex* alpha,
                                                  const hipComplex* x,
                                                  int64_t           incx,
                                                  const hipComplex* y,
                                                  int64_t           incy,
                                                  hipComplex*       AP,
                                                  int64_t           lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgeru_v2_64(hipblasHandle_t         handle,
                                                  int64_t                 m,
                                                  int64_t                 n,
                                                  const hipDoubleComplex* alpha,
                                                  const hipDoubleComplex* x,
                                                  int64_t                 incx,
                                                  const hipDoubleComplex* y,
                                                  int64_t                 incy,
                                                  hipDoubleComplex*       AP,
                                                  int64_t                 lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgerc_v2_64(hipblasHandle_t         handle,
                                                  int64_t                 m,
                                                  int64_t                 n,
                                                  const hipDoubleComplex* alpha,
                                                  const hipDoubleComplex* x,
                                                  int64_t                 incx,
                                                  const hipDoubleComplex* y,
                                                  int64_t                 incy,
                                                  hipDoubleComplex*       AP,
                                                  int64_t                 lda);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    gerBatched,geruBatched,gercBatched performs a batch of the matrix-vector operations

        A := A + alpha*x*y**T , OR
        A := A + alpha*x*y**H for gerc

    where (A_i, x_i, y_i) is the i-th instance of the batch.
    alpha is a scalar, x_i and y_i are vectors and A_i is an
    m by n matrix, for i = 1, ..., batchCount.

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    m         [int]
              the number of rows of each matrix A_i.
    @param[in]
    n         [int]
              the number of columns of eaceh matrix A_i.
    @param[in]
    alpha
              device pointer or host pointer to scalar alpha.
    @param[in]
    x         device array of device pointers storing each vector x_i.
    @param[in]
    incx      [int]
              specifies the increment for the elements of each vector x_i.
    @param[in]
    y         device array of device pointers storing each vector y_i.
    @param[in]
    incy      [int]
              specifies the increment for the elements of each vector y_i.
    @param[inout]
    AP        device array of device pointers storing each matrix A_i.
    @param[in]
    lda       [int]
              specifies the leading dimension of each A_i.
    @param[in]
    batchCount [int]
                number of instances in the batch

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSgerBatched(hipblasHandle_t    handle,
                                                  int                m,
                                                  int                n,
                                                  const float*       alpha,
                                                  const float* const x[],
                                                  int                incx,
                                                  const float* const y[],
                                                  int                incy,
                                                  float* const       AP[],
                                                  int                lda,
                                                  int                batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDgerBatched(hipblasHandle_t     handle,
                                                  int                 m,
                                                  int                 n,
                                                  const double*       alpha,
                                                  const double* const x[],
                                                  int                 incx,
                                                  const double* const y[],
                                                  int                 incy,
                                                  double* const       AP[],
                                                  int                 lda,
                                                  int                 batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgeruBatched(hipblasHandle_t             handle,
                                                   int                         m,
                                                   int                         n,
                                                   const hipblasComplex*       alpha,
                                                   const hipblasComplex* const x[],
                                                   int                         incx,
                                                   const hipblasComplex* const y[],
                                                   int                         incy,
                                                   hipblasComplex* const       AP[],
                                                   int                         lda,
                                                   int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgercBatched(hipblasHandle_t             handle,
                                                   int                         m,
                                                   int                         n,
                                                   const hipblasComplex*       alpha,
                                                   const hipblasComplex* const x[],
                                                   int                         incx,
                                                   const hipblasComplex* const y[],
                                                   int                         incy,
                                                   hipblasComplex* const       AP[],
                                                   int                         lda,
                                                   int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgeruBatched(hipblasHandle_t                   handle,
                                                   int                               m,
                                                   int                               n,
                                                   const hipblasDoubleComplex*       alpha,
                                                   const hipblasDoubleComplex* const x[],
                                                   int                               incx,
                                                   const hipblasDoubleComplex* const y[],
                                                   int                               incy,
                                                   hipblasDoubleComplex* const       AP[],
                                                   int                               lda,
                                                   int                               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgercBatched(hipblasHandle_t                   handle,
                                                   int                               m,
                                                   int                               n,
                                                   const hipblasDoubleComplex*       alpha,
                                                   const hipblasDoubleComplex* const x[],
                                                   int                               incx,
                                                   const hipblasDoubleComplex* const y[],
                                                   int                               incy,
                                                   hipblasDoubleComplex* const       AP[],
                                                   int                               lda,
                                                   int                               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgeruBatched_v2(hipblasHandle_t         handle,
                                                      int                     m,
                                                      int                     n,
                                                      const hipComplex*       alpha,
                                                      const hipComplex* const x[],
                                                      int                     incx,
                                                      const hipComplex* const y[],
                                                      int                     incy,
                                                      hipComplex* const       AP[],
                                                      int                     lda,
                                                      int                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgercBatched_v2(hipblasHandle_t         handle,
                                                      int                     m,
                                                      int                     n,
                                                      const hipComplex*       alpha,
                                                      const hipComplex* const x[],
                                                      int                     incx,
                                                      const hipComplex* const y[],
                                                      int                     incy,
                                                      hipComplex* const       AP[],
                                                      int                     lda,
                                                      int                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgeruBatched_v2(hipblasHandle_t               handle,
                                                      int                           m,
                                                      int                           n,
                                                      const hipDoubleComplex*       alpha,
                                                      const hipDoubleComplex* const x[],
                                                      int                           incx,
                                                      const hipDoubleComplex* const y[],
                                                      int                           incy,
                                                      hipDoubleComplex* const       AP[],
                                                      int                           lda,
                                                      int                           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgercBatched_v2(hipblasHandle_t               handle,
                                                      int                           m,
                                                      int                           n,
                                                      const hipDoubleComplex*       alpha,
                                                      const hipDoubleComplex* const x[],
                                                      int                           incx,
                                                      const hipDoubleComplex* const y[],
                                                      int                           incy,
                                                      hipDoubleComplex* const       AP[],
                                                      int                           lda,
                                                      int                           batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSgerBatched_64(hipblasHandle_t    handle,
                                                     int64_t            m,
                                                     int64_t            n,
                                                     const float*       alpha,
                                                     const float* const x[],
                                                     int64_t            incx,
                                                     const float* const y[],
                                                     int64_t            incy,
                                                     float* const       AP[],
                                                     int64_t            lda,
                                                     int64_t            batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDgerBatched_64(hipblasHandle_t     handle,
                                                     int64_t             m,
                                                     int64_t             n,
                                                     const double*       alpha,
                                                     const double* const x[],
                                                     int64_t             incx,
                                                     const double* const y[],
                                                     int64_t             incy,
                                                     double* const       AP[],
                                                     int64_t             lda,
                                                     int64_t             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgeruBatched_64(hipblasHandle_t             handle,
                                                      int64_t                     m,
                                                      int64_t                     n,
                                                      const hipblasComplex*       alpha,
                                                      const hipblasComplex* const x[],
                                                      int64_t                     incx,
                                                      const hipblasComplex* const y[],
                                                      int64_t                     incy,
                                                      hipblasComplex* const       AP[],
                                                      int64_t                     lda,
                                                      int64_t                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgercBatched_64(hipblasHandle_t             handle,
                                                      int64_t                     m,
                                                      int64_t                     n,
                                                      const hipblasComplex*       alpha,
                                                      const hipblasComplex* const x[],
                                                      int64_t                     incx,
                                                      const hipblasComplex* const y[],
                                                      int64_t                     incy,
                                                      hipblasComplex* const       AP[],
                                                      int64_t                     lda,
                                                      int64_t                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgeruBatched_64(hipblasHandle_t                   handle,
                                                      int64_t                           m,
                                                      int64_t                           n,
                                                      const hipblasDoubleComplex*       alpha,
                                                      const hipblasDoubleComplex* const x[],
                                                      int64_t                           incx,
                                                      const hipblasDoubleComplex* const y[],
                                                      int64_t                           incy,
                                                      hipblasDoubleComplex* const       AP[],
                                                      int64_t                           lda,
                                                      int64_t                           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgercBatched_64(hipblasHandle_t                   handle,
                                                      int64_t                           m,
                                                      int64_t                           n,
                                                      const hipblasDoubleComplex*       alpha,
                                                      const hipblasDoubleComplex* const x[],
                                                      int64_t                           incx,
                                                      const hipblasDoubleComplex* const y[],
                                                      int64_t                           incy,
                                                      hipblasDoubleComplex* const       AP[],
                                                      int64_t                           lda,
                                                      int64_t                           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgeruBatched_v2_64(hipblasHandle_t         handle,
                                                         int64_t                 m,
                                                         int64_t                 n,
                                                         const hipComplex*       alpha,
                                                         const hipComplex* const x[],
                                                         int64_t                 incx,
                                                         const hipComplex* const y[],
                                                         int64_t                 incy,
                                                         hipComplex* const       AP[],
                                                         int64_t                 lda,
                                                         int64_t                 batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgercBatched_v2_64(hipblasHandle_t         handle,
                                                         int64_t                 m,
                                                         int64_t                 n,
                                                         const hipComplex*       alpha,
                                                         const hipComplex* const x[],
                                                         int64_t                 incx,
                                                         const hipComplex* const y[],
                                                         int64_t                 incy,
                                                         hipComplex* const       AP[],
                                                         int64_t                 lda,
                                                         int64_t                 batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgeruBatched_v2_64(hipblasHandle_t               handle,
                                                         int64_t                       m,
                                                         int64_t                       n,
                                                         const hipDoubleComplex*       alpha,
                                                         const hipDoubleComplex* const x[],
                                                         int64_t                       incx,
                                                         const hipDoubleComplex* const y[],
                                                         int64_t                       incy,
                                                         hipDoubleComplex* const       AP[],
                                                         int64_t                       lda,
                                                         int64_t                       batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgercBatched_v2_64(hipblasHandle_t               handle,
                                                         int64_t                       m,
                                                         int64_t                       n,
                                                         const hipDoubleComplex*       alpha,
                                                         const hipDoubleComplex* const x[],
                                                         int64_t                       incx,
                                                         const hipDoubleComplex* const y[],
                                                         int64_t                       incy,
                                                         hipDoubleComplex* const       AP[],
                                                         int64_t                       lda,
                                                         int64_t                       batchCount);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    gerStridedBatched,geruStridedBatched,gercStridedBatched performs the matrix-vector operations

        A_i := A_i + alpha*x_i*y_i**T, OR
        A_i := A_i + alpha*x_i*y_i**H  for gerc

    where (A_i, x_i, y_i) is the i-th instance of the batch.
    alpha is a scalar, x_i and y_i are vectors and A_i is an
    m by n matrix, for i = 1, ..., batchCount.

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    m         [int]
              the number of rows of each matrix A_i.
    @param[in]
    n         [int]
              the number of columns of each matrix A_i.
    @param[in]
    alpha
              device pointer or host pointer to scalar alpha.
    @param[in]
    x         device pointer to the first vector (x_1) in the batch.
    @param[in]
    incx      [int]
              specifies the increments for the elements of each vector x_i.
    @param[in]
    stridex   [hipblasStride]
              stride from the start of one vector (x_i) and the next one (x_i+1).
              There are no restrictions placed on stridex, however the user should
              take care to ensure that stridex is of appropriate size, for a typical
              case this means stridex >= m * incx.
    @param[inout]
    y         device pointer to the first vector (y_1) in the batch.
    @param[in]
    incy      [int]
              specifies the increment for the elements of each vector y_i.
    @param[in]
    stridey   [hipblasStride]
              stride from the start of one vector (y_i) and the next one (y_i+1).
              There are no restrictions placed on stridey, however the user should
              take care to ensure that stridey is of appropriate size, for a typical
              case this means stridey >= n * incy.
    @param[inout]
    AP        device pointer to the first matrix (A_1) in the batch.
    @param[in]
    lda       [int]
              specifies the leading dimension of each A_i.
    @param[in]
    strideA     [hipblasStride]
                stride from the start of one matrix (A_i) and the next one (A_i+1)
    @param[in]
    batchCount [int]
                number of instances in the batch

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSgerStridedBatched(hipblasHandle_t handle,
                                                         int             m,
                                                         int             n,
                                                         const float*    alpha,
                                                         const float*    x,
                                                         int             incx,
                                                         hipblasStride   stridex,
                                                         const float*    y,
                                                         int             incy,
                                                         hipblasStride   stridey,
                                                         float*          AP,
                                                         int             lda,
                                                         hipblasStride   strideA,
                                                         int             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDgerStridedBatched(hipblasHandle_t handle,
                                                         int             m,
                                                         int             n,
                                                         const double*   alpha,
                                                         const double*   x,
                                                         int             incx,
                                                         hipblasStride   stridex,
                                                         const double*   y,
                                                         int             incy,
                                                         hipblasStride   stridey,
                                                         double*         AP,
                                                         int             lda,
                                                         hipblasStride   strideA,
                                                         int             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgeruStridedBatched(hipblasHandle_t       handle,
                                                          int                   m,
                                                          int                   n,
                                                          const hipblasComplex* alpha,
                                                          const hipblasComplex* x,
                                                          int                   incx,
                                                          hipblasStride         stridex,
                                                          const hipblasComplex* y,
                                                          int                   incy,
                                                          hipblasStride         stridey,
                                                          hipblasComplex*       AP,
                                                          int                   lda,
                                                          hipblasStride         strideA,
                                                          int                   batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgercStridedBatched(hipblasHandle_t       handle,
                                                          int                   m,
                                                          int                   n,
                                                          const hipblasComplex* alpha,
                                                          const hipblasComplex* x,
                                                          int                   incx,
                                                          hipblasStride         stridex,
                                                          const hipblasComplex* y,
                                                          int                   incy,
                                                          hipblasStride         stridey,
                                                          hipblasComplex*       AP,
                                                          int                   lda,
                                                          hipblasStride         strideA,
                                                          int                   batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgeruStridedBatched(hipblasHandle_t             handle,
                                                          int                         m,
                                                          int                         n,
                                                          const hipblasDoubleComplex* alpha,
                                                          const hipblasDoubleComplex* x,
                                                          int                         incx,
                                                          hipblasStride               stridex,
                                                          const hipblasDoubleComplex* y,
                                                          int                         incy,
                                                          hipblasStride               stridey,
                                                          hipblasDoubleComplex*       AP,
                                                          int                         lda,
                                                          hipblasStride               strideA,
                                                          int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgercStridedBatched(hipblasHandle_t             handle,
                                                          int                         m,
                                                          int                         n,
                                                          const hipblasDoubleComplex* alpha,
                                                          const hipblasDoubleComplex* x,
                                                          int                         incx,
                                                          hipblasStride               stridex,
                                                          const hipblasDoubleComplex* y,
                                                          int                         incy,
                                                          hipblasStride               stridey,
                                                          hipblasDoubleComplex*       AP,
                                                          int                         lda,
                                                          hipblasStride               strideA,
                                                          int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgeruStridedBatched_v2(hipblasHandle_t   handle,
                                                             int               m,
                                                             int               n,
                                                             const hipComplex* alpha,
                                                             const hipComplex* x,
                                                             int               incx,
                                                             hipblasStride     stridex,
                                                             const hipComplex* y,
                                                             int               incy,
                                                             hipblasStride     stridey,
                                                             hipComplex*       AP,
                                                             int               lda,
                                                             hipblasStride     strideA,
                                                             int               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgercStridedBatched_v2(hipblasHandle_t   handle,
                                                             int               m,
                                                             int               n,
                                                             const hipComplex* alpha,
                                                             const hipComplex* x,
                                                             int               incx,
                                                             hipblasStride     stridex,
                                                             const hipComplex* y,
                                                             int               incy,
                                                             hipblasStride     stridey,
                                                             hipComplex*       AP,
                                                             int               lda,
                                                             hipblasStride     strideA,
                                                             int               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgeruStridedBatched_v2(hipblasHandle_t         handle,
                                                             int                     m,
                                                             int                     n,
                                                             const hipDoubleComplex* alpha,
                                                             const hipDoubleComplex* x,
                                                             int                     incx,
                                                             hipblasStride           stridex,
                                                             const hipDoubleComplex* y,
                                                             int                     incy,
                                                             hipblasStride           stridey,
                                                             hipDoubleComplex*       AP,
                                                             int                     lda,
                                                             hipblasStride           strideA,
                                                             int                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgercStridedBatched_v2(hipblasHandle_t         handle,
                                                             int                     m,
                                                             int                     n,
                                                             const hipDoubleComplex* alpha,
                                                             const hipDoubleComplex* x,
                                                             int                     incx,
                                                             hipblasStride           stridex,
                                                             const hipDoubleComplex* y,
                                                             int                     incy,
                                                             hipblasStride           stridey,
                                                             hipDoubleComplex*       AP,
                                                             int                     lda,
                                                             hipblasStride           strideA,
                                                             int                     batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSgerStridedBatched_64(hipblasHandle_t handle,
                                                            int64_t         m,
                                                            int64_t         n,
                                                            const float*    alpha,
                                                            const float*    x,
                                                            int64_t         incx,
                                                            hipblasStride   stridex,
                                                            const float*    y,
                                                            int64_t         incy,
                                                            hipblasStride   stridey,
                                                            float*          AP,
                                                            int64_t         lda,
                                                            hipblasStride   strideA,
                                                            int64_t         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDgerStridedBatched_64(hipblasHandle_t handle,
                                                            int64_t         m,
                                                            int64_t         n,
                                                            const double*   alpha,
                                                            const double*   x,
                                                            int64_t         incx,
                                                            hipblasStride   stridex,
                                                            const double*   y,
                                                            int64_t         incy,
                                                            hipblasStride   stridey,
                                                            double*         AP,
                                                            int64_t         lda,
                                                            hipblasStride   strideA,
                                                            int64_t         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgeruStridedBatched_64(hipblasHandle_t       handle,
                                                             int64_t               m,
                                                             int64_t               n,
                                                             const hipblasComplex* alpha,
                                                             const hipblasComplex* x,
                                                             int64_t               incx,
                                                             hipblasStride         stridex,
                                                             const hipblasComplex* y,
                                                             int64_t               incy,
                                                             hipblasStride         stridey,
                                                             hipblasComplex*       AP,
                                                             int64_t               lda,
                                                             hipblasStride         strideA,
                                                             int64_t               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgercStridedBatched_64(hipblasHandle_t       handle,
                                                             int64_t               m,
                                                             int64_t               n,
                                                             const hipblasComplex* alpha,
                                                             const hipblasComplex* x,
                                                             int64_t               incx,
                                                             hipblasStride         stridex,
                                                             const hipblasComplex* y,
                                                             int64_t               incy,
                                                             hipblasStride         stridey,
                                                             hipblasComplex*       AP,
                                                             int64_t               lda,
                                                             hipblasStride         strideA,
                                                             int64_t               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgeruStridedBatched_64(hipblasHandle_t             handle,
                                                             int64_t                     m,
                                                             int64_t                     n,
                                                             const hipblasDoubleComplex* alpha,
                                                             const hipblasDoubleComplex* x,
                                                             int64_t                     incx,
                                                             hipblasStride               stridex,
                                                             const hipblasDoubleComplex* y,
                                                             int64_t                     incy,
                                                             hipblasStride               stridey,
                                                             hipblasDoubleComplex*       AP,
                                                             int64_t                     lda,
                                                             hipblasStride               strideA,
                                                             int64_t batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgercStridedBatched_64(hipblasHandle_t             handle,
                                                             int64_t                     m,
                                                             int64_t                     n,
                                                             const hipblasDoubleComplex* alpha,
                                                             const hipblasDoubleComplex* x,
                                                             int64_t                     incx,
                                                             hipblasStride               stridex,
                                                             const hipblasDoubleComplex* y,
                                                             int64_t                     incy,
                                                             hipblasStride               stridey,
                                                             hipblasDoubleComplex*       AP,
                                                             int64_t                     lda,
                                                             hipblasStride               strideA,
                                                             int64_t batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgeruStridedBatched_v2_64(hipblasHandle_t   handle,
                                                                int64_t           m,
                                                                int64_t           n,
                                                                const hipComplex* alpha,
                                                                const hipComplex* x,
                                                                int64_t           incx,
                                                                hipblasStride     stridex,
                                                                const hipComplex* y,
                                                                int64_t           incy,
                                                                hipblasStride     stridey,
                                                                hipComplex*       AP,
                                                                int64_t           lda,
                                                                hipblasStride     strideA,
                                                                int64_t           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCgercStridedBatched_v2_64(hipblasHandle_t   handle,
                                                                int64_t           m,
                                                                int64_t           n,
                                                                const hipComplex* alpha,
                                                                const hipComplex* x,
                                                                int64_t           incx,
                                                                hipblasStride     stridex,
                                                                const hipComplex* y,
                                                                int64_t           incy,
                                                                hipblasStride     stridey,
                                                                hipComplex*       AP,
                                                                int64_t           lda,
                                                                hipblasStride     strideA,
                                                                int64_t           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgeruStridedBatched_v2_64(hipblasHandle_t         handle,
                                                                int64_t                 m,
                                                                int64_t                 n,
                                                                const hipDoubleComplex* alpha,
                                                                const hipDoubleComplex* x,
                                                                int64_t                 incx,
                                                                hipblasStride           stridex,
                                                                const hipDoubleComplex* y,
                                                                int64_t                 incy,
                                                                hipblasStride           stridey,
                                                                hipDoubleComplex*       AP,
                                                                int64_t                 lda,
                                                                hipblasStride           strideA,
                                                                int64_t                 batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZgercStridedBatched_v2_64(hipblasHandle_t         handle,
                                                                int64_t                 m,
                                                                int64_t                 n,
                                                                const hipDoubleComplex* alpha,
                                                                const hipDoubleComplex* x,
                                                                int64_t                 incx,
                                                                hipblasStride           stridex,
                                                                const hipDoubleComplex* y,
                                                                int64_t                 incy,
                                                                hipblasStride           stridey,
                                                                hipDoubleComplex*       AP,
                                                                int64_t                 lda,
                                                                hipblasStride           strideA,
                                                                int64_t                 batchCount);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    hbmv performs the matrix-vector operations

        y := alpha*A*x + beta*y

    where alpha and beta are scalars, x and y are n element vectors and A is an
    n by n Hermitian band matrix, with k super-diagonals.

    - Supported precisions in rocBLAS : c,z
    - Supported precisions in cuBLAS  : c,z

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              HIPBLAS_FILL_MODE_UPPER: The upper triangular part of A is being supplied.
              HIPBLAS_FILL_MODE_LOWER: The lower triangular part of A is being supplied.
    @param[in]
    n         [int]
              the order of the matrix A.
    @param[in]
    k         [int]
              the number of super-diagonals of the matrix A. Must be >= 0.
    @param[in]
    alpha     device pointer or host pointer to scalar alpha.
    @param[in]
    AP        device pointer storing matrix A. Of dimension (lda, n).
              if uplo == HIPBLAS_FILL_MODE_UPPER:
                The leading (k + 1) by n part of A must contain the upper
                triangular band part of the Hermitian matrix, with the leading
                diagonal in row (k + 1), the first super-diagonal on the RHS
                of row k, etc.
                The top left k by x triangle of A will not be referenced.
                    Ex (upper, lda = n = 4, k = 1):
                    A                             Represented matrix
                    (0,0) (5,9) (6,8) (7,7)       (1, 0) (5, 9) (0, 0) (0, 0)
                    (1,0) (2,0) (3,0) (4,0)       (5,-9) (2, 0) (6, 8) (0, 0)
                    (0,0) (0,0) (0,0) (0,0)       (0, 0) (6,-8) (3, 0) (7, 7)
                    (0,0) (0,0) (0,0) (0,0)       (0, 0) (0, 0) (7,-7) (4, 0)

              if uplo == HIPBLAS_FILL_MODE_LOWER:
                The leading (k + 1) by n part of A must contain the lower
                triangular band part of the Hermitian matrix, with the leading
                diagonal in row (1), the first sub-diagonal on the LHS of
                row 2, etc.
                The bottom right k by k triangle of A will not be referenced.
                    Ex (lower, lda = 2, n = 4, k = 1):
                    A                               Represented matrix
                    (1,0) (2,0) (3,0) (4,0)         (1, 0) (5,-9) (0, 0) (0, 0)
                    (5,9) (6,8) (7,7) (0,0)         (5, 9) (2, 0) (6,-8) (0, 0)
                                                    (0, 0) (6, 8) (3, 0) (7,-7)
                                                    (0, 0) (0, 0) (7, 7) (4, 0)

              As a Hermitian matrix, the imaginary part of the main diagonal
              of A will not be referenced and is assumed to be == 0.
    @param[in]
    lda       [int]
              specifies the leading dimension of A. must be >= k + 1
    @param[in]
    x         device pointer storing vector x.
    @param[in]
    incx      [int]
              specifies the increment for the elements of x.
    @param[in]
    beta      device pointer or host pointer to scalar beta.
    @param[inout]
    y         device pointer storing vector y.
    @param[in]
    incy      [int]
              specifies the increment for the elements of y.

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasChbmv(hipblasHandle_t       handle,
                                            hipblasFillMode_t     uplo,
                                            int                   n,
                                            int                   k,
                                            const hipblasComplex* alpha,
                                            const hipblasComplex* AP,
                                            int                   lda,
                                            const hipblasComplex* x,
                                            int                   incx,
                                            const hipblasComplex* beta,
                                            hipblasComplex*       y,
                                            int                   incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhbmv(hipblasHandle_t             handle,
                                            hipblasFillMode_t           uplo,
                                            int                         n,
                                            int                         k,
                                            const hipblasDoubleComplex* alpha,
                                            const hipblasDoubleComplex* AP,
                                            int                         lda,
                                            const hipblasDoubleComplex* x,
                                            int                         incx,
                                            const hipblasDoubleComplex* beta,
                                            hipblasDoubleComplex*       y,
                                            int                         incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasChbmv_v2(hipblasHandle_t   handle,
                                               hipblasFillMode_t uplo,
                                               int               n,
                                               int               k,
                                               const hipComplex* alpha,
                                               const hipComplex* AP,
                                               int               lda,
                                               const hipComplex* x,
                                               int               incx,
                                               const hipComplex* beta,
                                               hipComplex*       y,
                                               int               incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhbmv_v2(hipblasHandle_t         handle,
                                               hipblasFillMode_t       uplo,
                                               int                     n,
                                               int                     k,
                                               const hipDoubleComplex* alpha,
                                               const hipDoubleComplex* AP,
                                               int                     lda,
                                               const hipDoubleComplex* x,
                                               int                     incx,
                                               const hipDoubleComplex* beta,
                                               hipDoubleComplex*       y,
                                               int                     incy);

//64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasChbmv_64(hipblasHandle_t       handle,
                                               hipblasFillMode_t     uplo,
                                               int64_t               n,
                                               int64_t               k,
                                               const hipblasComplex* alpha,
                                               const hipblasComplex* AP,
                                               int64_t               lda,
                                               const hipblasComplex* x,
                                               int64_t               incx,
                                               const hipblasComplex* beta,
                                               hipblasComplex*       y,
                                               int64_t               incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhbmv_64(hipblasHandle_t             handle,
                                               hipblasFillMode_t           uplo,
                                               int64_t                     n,
                                               int64_t                     k,
                                               const hipblasDoubleComplex* alpha,
                                               const hipblasDoubleComplex* AP,
                                               int64_t                     lda,
                                               const hipblasDoubleComplex* x,
                                               int64_t                     incx,
                                               const hipblasDoubleComplex* beta,
                                               hipblasDoubleComplex*       y,
                                               int64_t                     incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasChbmv_v2_64(hipblasHandle_t   handle,
                                                  hipblasFillMode_t uplo,
                                                  int64_t           n,
                                                  int64_t           k,
                                                  const hipComplex* alpha,
                                                  const hipComplex* AP,
                                                  int64_t           lda,
                                                  const hipComplex* x,
                                                  int64_t           incx,
                                                  const hipComplex* beta,
                                                  hipComplex*       y,
                                                  int64_t           incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhbmv_v2_64(hipblasHandle_t         handle,
                                                  hipblasFillMode_t       uplo,
                                                  int64_t                 n,
                                                  int64_t                 k,
                                                  const hipDoubleComplex* alpha,
                                                  const hipDoubleComplex* AP,
                                                  int64_t                 lda,
                                                  const hipDoubleComplex* x,
                                                  int64_t                 incx,
                                                  const hipDoubleComplex* beta,
                                                  hipDoubleComplex*       y,
                                                  int64_t                 incy);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    hbmvBatched performs one of the matrix-vector operations

        y_i := alpha*A_i*x_i + beta*y_i

    where alpha and beta are scalars, x_i and y_i are n element vectors and A_i is an
    n by n Hermitian band matrix with k super-diagonals, for each batch in i = [1, batchCount].

    - Supported precisions in rocBLAS : c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              HIPBLAS_FILL_MODE_UPPER: The upper triangular part of each A_i is being supplied.
              HIPBLAS_FILL_MODE_LOWER: The lower triangular part of each A_i is being supplied.
    @param[in]
    n         [int]
              the order of each matrix A_i.
    @param[in]
    k         [int]
              the number of super-diagonals of each matrix A_i. Must be >= 0.
    @param[in]
    alpha     device pointer or host pointer to scalar alpha.
    @param[in]
    AP        device array of device pointers storing each matrix_i A of dimension (lda, n).
              if uplo == HIPBLAS_FILL_MODE_UPPER:
                The leading (k + 1) by n part of each A_i must contain the upper
                triangular band part of the Hermitian matrix, with the leading
                diagonal in row (k + 1), the first super-diagonal on the RHS
                of row k, etc.
                The top left k by x triangle of each A_i will not be referenced.
                    Ex (upper, lda = n = 4, k = 1):
                    A                             Represented matrix
                    (0,0) (5,9) (6,8) (7,7)       (1, 0) (5, 9) (0, 0) (0, 0)
                    (1,0) (2,0) (3,0) (4,0)       (5,-9) (2, 0) (6, 8) (0, 0)
                    (0,0) (0,0) (0,0) (0,0)       (0, 0) (6,-8) (3, 0) (7, 7)
                    (0,0) (0,0) (0,0) (0,0)       (0, 0) (0, 0) (7,-7) (4, 0)

              if uplo == HIPBLAS_FILL_MODE_LOWER:
                The leading (k + 1) by n part of each A_i must contain the lower
                triangular band part of the Hermitian matrix, with the leading
                diagonal in row (1), the first sub-diagonal on the LHS of
                row 2, etc.
                The bottom right k by k triangle of each A_i will not be referenced.
                    Ex (lower, lda = 2, n = 4, k = 1):
                    A                               Represented matrix
                    (1,0) (2,0) (3,0) (4,0)         (1, 0) (5,-9) (0, 0) (0, 0)
                    (5,9) (6,8) (7,7) (0,0)         (5, 9) (2, 0) (6,-8) (0, 0)
                                                    (0, 0) (6, 8) (3, 0) (7,-7)
                                                    (0, 0) (0, 0) (7, 7) (4, 0)

              As a Hermitian matrix, the imaginary part of the main diagonal
              of each A_i will not be referenced and is assumed to be == 0.
    @param[in]
    lda       [int]
              specifies the leading dimension of each A_i. must be >= max(1, n)
    @param[in]
    x         device array of device pointers storing each vector x_i.
    @param[in]
    incx      [int]
              specifies the increment for the elements of each x_i.
    @param[in]
    beta      device pointer or host pointer to scalar beta.
    @param[inout]
    y         device array of device pointers storing each vector y_i.
    @param[in]
    incy      [int]
              specifies the increment for the elements of y.
    @param[in]
    batchCount [int]
                number of instances in the batch.

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasChbmvBatched(hipblasHandle_t             handle,
                                                   hipblasFillMode_t           uplo,
                                                   int                         n,
                                                   int                         k,
                                                   const hipblasComplex*       alpha,
                                                   const hipblasComplex* const AP[],
                                                   int                         lda,
                                                   const hipblasComplex* const x[],
                                                   int                         incx,
                                                   const hipblasComplex*       beta,
                                                   hipblasComplex* const       y[],
                                                   int                         incy,
                                                   int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhbmvBatched(hipblasHandle_t                   handle,
                                                   hipblasFillMode_t                 uplo,
                                                   int                               n,
                                                   int                               k,
                                                   const hipblasDoubleComplex*       alpha,
                                                   const hipblasDoubleComplex* const AP[],
                                                   int                               lda,
                                                   const hipblasDoubleComplex* const x[],
                                                   int                               incx,
                                                   const hipblasDoubleComplex*       beta,
                                                   hipblasDoubleComplex* const       y[],
                                                   int                               incy,
                                                   int                               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasChbmvBatched_v2(hipblasHandle_t         handle,
                                                      hipblasFillMode_t       uplo,
                                                      int                     n,
                                                      int                     k,
                                                      const hipComplex*       alpha,
                                                      const hipComplex* const AP[],
                                                      int                     lda,
                                                      const hipComplex* const x[],
                                                      int                     incx,
                                                      const hipComplex*       beta,
                                                      hipComplex* const       y[],
                                                      int                     incy,
                                                      int                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhbmvBatched_v2(hipblasHandle_t               handle,
                                                      hipblasFillMode_t             uplo,
                                                      int                           n,
                                                      int                           k,
                                                      const hipDoubleComplex*       alpha,
                                                      const hipDoubleComplex* const AP[],
                                                      int                           lda,
                                                      const hipDoubleComplex* const x[],
                                                      int                           incx,
                                                      const hipDoubleComplex*       beta,
                                                      hipDoubleComplex* const       y[],
                                                      int                           incy,
                                                      int                           batchCount);

//64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasChbmvBatched_64(hipblasHandle_t             handle,
                                                      hipblasFillMode_t           uplo,
                                                      int64_t                     n,
                                                      int64_t                     k,
                                                      const hipblasComplex*       alpha,
                                                      const hipblasComplex* const AP[],
                                                      int64_t                     lda,
                                                      const hipblasComplex* const x[],
                                                      int64_t                     incx,
                                                      const hipblasComplex*       beta,
                                                      hipblasComplex* const       y[],
                                                      int64_t                     incy,
                                                      int64_t                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhbmvBatched_64(hipblasHandle_t                   handle,
                                                      hipblasFillMode_t                 uplo,
                                                      int64_t                           n,
                                                      int64_t                           k,
                                                      const hipblasDoubleComplex*       alpha,
                                                      const hipblasDoubleComplex* const AP[],
                                                      int64_t                           lda,
                                                      const hipblasDoubleComplex* const x[],
                                                      int64_t                           incx,
                                                      const hipblasDoubleComplex*       beta,
                                                      hipblasDoubleComplex* const       y[],
                                                      int64_t                           incy,
                                                      int64_t                           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasChbmvBatched_v2_64(hipblasHandle_t         handle,
                                                         hipblasFillMode_t       uplo,
                                                         int64_t                 n,
                                                         int64_t                 k,
                                                         const hipComplex*       alpha,
                                                         const hipComplex* const AP[],
                                                         int64_t                 lda,
                                                         const hipComplex* const x[],
                                                         int64_t                 incx,
                                                         const hipComplex*       beta,
                                                         hipComplex* const       y[],
                                                         int64_t                 incy,
                                                         int64_t                 batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhbmvBatched_v2_64(hipblasHandle_t               handle,
                                                         hipblasFillMode_t             uplo,
                                                         int64_t                       n,
                                                         int64_t                       k,
                                                         const hipDoubleComplex*       alpha,
                                                         const hipDoubleComplex* const AP[],
                                                         int64_t                       lda,
                                                         const hipDoubleComplex* const x[],
                                                         int64_t                       incx,
                                                         const hipDoubleComplex*       beta,
                                                         hipDoubleComplex* const       y[],
                                                         int64_t                       incy,
                                                         int64_t                       batchCount);

//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    hbmvStridedBatched performs one of the matrix-vector operations

        y_i := alpha*A_i*x_i + beta*y_i

    where alpha and beta are scalars, x_i and y_i are n element vectors and A_i is an
    n by n Hermitian band matrix with k super-diagonals, for each batch in i = [1, batchCount].

    - Supported precisions in rocBLAS : c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              HIPBLAS_FILL_MODE_UPPER: The upper triangular part of each A_i is being supplied.
              HIPBLAS_FILL_MODE_LOWER: The lower triangular part of each A_i is being supplied.
    @param[in]
    n         [int]
              the order of each matrix A_i.
    @param[in]
    k         [int]
              the number of super-diagonals of each matrix A_i. Must be >= 0.
    @param[in]
    alpha     device pointer or host pointer to scalar alpha.
    @param[in]
    AP        device array pointing to the first matrix A_1. Each A_i is of dimension (lda, n).
              if uplo == HIPBLAS_FILL_MODE_UPPER:
                The leading (k + 1) by n part of each A_i must contain the upper
                triangular band part of the Hermitian matrix, with the leading
                diagonal in row (k + 1), the first super-diagonal on the RHS
                of row k, etc.
                The top left k by x triangle of each A_i will not be referenced.
                    Ex (upper, lda = n = 4, k = 1):
                    A                             Represented matrix
                    (0,0) (5,9) (6,8) (7,7)       (1, 0) (5, 9) (0, 0) (0, 0)
                    (1,0) (2,0) (3,0) (4,0)       (5,-9) (2, 0) (6, 8) (0, 0)
                    (0,0) (0,0) (0,0) (0,0)       (0, 0) (6,-8) (3, 0) (7, 7)
                    (0,0) (0,0) (0,0) (0,0)       (0, 0) (0, 0) (7,-7) (4, 0)

              if uplo == HIPBLAS_FILL_MODE_LOWER:
                The leading (k + 1) by n part of each A_i must contain the lower
                triangular band part of the Hermitian matrix, with the leading
                diagonal in row (1), the first sub-diagonal on the LHS of
                row 2, etc.
                The bottom right k by k triangle of each A_i will not be referenced.
                    Ex (lower, lda = 2, n = 4, k = 1):
                    A                               Represented matrix
                    (1,0) (2,0) (3,0) (4,0)         (1, 0) (5,-9) (0, 0) (0, 0)
                    (5,9) (6,8) (7,7) (0,0)         (5, 9) (2, 0) (6,-8) (0, 0)
                                                    (0, 0) (6, 8) (3, 0) (7,-7)
                                                    (0, 0) (0, 0) (7, 7) (4, 0)

              As a Hermitian matrix, the imaginary part of the main diagonal
              of each A_i will not be referenced and is assumed to be == 0.
    @param[in]
    lda       [int]
              specifies the leading dimension of each A_i. must be >= max(1, n)
    @param[in]
    strideA  [hipblasStride]
              stride from the start of one matrix (A_i) and the next one (A_i+1)
    @param[in]
    x         device array pointing to the first vector y_1.
    @param[in]
    incx      [int]
              specifies the increment for the elements of each x_i.
    @param[in]
    stridex  [hipblasStride]
              stride from the start of one vector (x_i) and the next one (x_i+1)
    @param[in]
    beta      device pointer or host pointer to scalar beta.
    @param[inout]
    y         device array pointing to the first vector y_1.
    @param[in]
    incy      [int]
              specifies the increment for the elements of y.
    @param[in]
    stridey  [hipblasStride]
              stride from the start of one vector (y_i) and the next one (y_i+1)
    @param[in]
    batchCount [int]
                number of instances in the batch.

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasChbmvStridedBatched(hipblasHandle_t       handle,
                                                          hipblasFillMode_t     uplo,
                                                          int                   n,
                                                          int                   k,
                                                          const hipblasComplex* alpha,
                                                          const hipblasComplex* AP,
                                                          int                   lda,
                                                          hipblasStride         strideA,
                                                          const hipblasComplex* x,
                                                          int                   incx,
                                                          hipblasStride         stridex,
                                                          const hipblasComplex* beta,
                                                          hipblasComplex*       y,
                                                          int                   incy,
                                                          hipblasStride         stridey,
                                                          int                   batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhbmvStridedBatched(hipblasHandle_t             handle,
                                                          hipblasFillMode_t           uplo,
                                                          int                         n,
                                                          int                         k,
                                                          const hipblasDoubleComplex* alpha,
                                                          const hipblasDoubleComplex* AP,
                                                          int                         lda,
                                                          hipblasStride               strideA,
                                                          const hipblasDoubleComplex* x,
                                                          int                         incx,
                                                          hipblasStride               stridex,
                                                          const hipblasDoubleComplex* beta,
                                                          hipblasDoubleComplex*       y,
                                                          int                         incy,
                                                          hipblasStride               stridey,
                                                          int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasChbmvStridedBatched_v2(hipblasHandle_t   handle,
                                                             hipblasFillMode_t uplo,
                                                             int               n,
                                                             int               k,
                                                             const hipComplex* alpha,
                                                             const hipComplex* AP,
                                                             int               lda,
                                                             hipblasStride     strideA,
                                                             const hipComplex* x,
                                                             int               incx,
                                                             hipblasStride     stridex,
                                                             const hipComplex* beta,
                                                             hipComplex*       y,
                                                             int               incy,
                                                             hipblasStride     stridey,
                                                             int               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhbmvStridedBatched_v2(hipblasHandle_t         handle,
                                                             hipblasFillMode_t       uplo,
                                                             int                     n,
                                                             int                     k,
                                                             const hipDoubleComplex* alpha,
                                                             const hipDoubleComplex* AP,
                                                             int                     lda,
                                                             hipblasStride           strideA,
                                                             const hipDoubleComplex* x,
                                                             int                     incx,
                                                             hipblasStride           stridex,
                                                             const hipDoubleComplex* beta,
                                                             hipDoubleComplex*       y,
                                                             int                     incy,
                                                             hipblasStride           stridey,
                                                             int                     batchCount);

//64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasChbmvStridedBatched_64(hipblasHandle_t       handle,
                                                             hipblasFillMode_t     uplo,
                                                             int64_t               n,
                                                             int64_t               k,
                                                             const hipblasComplex* alpha,
                                                             const hipblasComplex* AP,
                                                             int64_t               lda,
                                                             hipblasStride         strideA,
                                                             const hipblasComplex* x,
                                                             int64_t               incx,
                                                             hipblasStride         stridex,
                                                             const hipblasComplex* beta,
                                                             hipblasComplex*       y,
                                                             int64_t               incy,
                                                             hipblasStride         stridey,
                                                             int64_t               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhbmvStridedBatched_64(hipblasHandle_t             handle,
                                                             hipblasFillMode_t           uplo,
                                                             int64_t                     n,
                                                             int64_t                     k,
                                                             const hipblasDoubleComplex* alpha,
                                                             const hipblasDoubleComplex* AP,
                                                             int64_t                     lda,
                                                             hipblasStride               strideA,
                                                             const hipblasDoubleComplex* x,
                                                             int64_t                     incx,
                                                             hipblasStride               stridex,
                                                             const hipblasDoubleComplex* beta,
                                                             hipblasDoubleComplex*       y,
                                                             int64_t                     incy,
                                                             hipblasStride               stridey,
                                                             int64_t batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasChbmvStridedBatched_v2_64(hipblasHandle_t   handle,
                                                                hipblasFillMode_t uplo,
                                                                int64_t           n,
                                                                int64_t           k,
                                                                const hipComplex* alpha,
                                                                const hipComplex* AP,
                                                                int64_t           lda,
                                                                hipblasStride     strideA,
                                                                const hipComplex* x,
                                                                int64_t           incx,
                                                                hipblasStride     stridex,
                                                                const hipComplex* beta,
                                                                hipComplex*       y,
                                                                int64_t           incy,
                                                                hipblasStride     stridey,
                                                                int64_t           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhbmvStridedBatched_v2_64(hipblasHandle_t         handle,
                                                                hipblasFillMode_t       uplo,
                                                                int64_t                 n,
                                                                int64_t                 k,
                                                                const hipDoubleComplex* alpha,
                                                                const hipDoubleComplex* AP,
                                                                int64_t                 lda,
                                                                hipblasStride           strideA,
                                                                const hipDoubleComplex* x,
                                                                int64_t                 incx,
                                                                hipblasStride           stridex,
                                                                const hipDoubleComplex* beta,
                                                                hipDoubleComplex*       y,
                                                                int64_t                 incy,
                                                                hipblasStride           stridey,
                                                                int64_t                 batchCount);

//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    hemv performs one of the matrix-vector operations

        y := alpha*A*x + beta*y

    where alpha and beta are scalars, x and y are n element vectors and A is an
    n by n Hermitian matrix.

    - Supported precisions in rocBLAS : c,z
    - Supported precisions in cuBLAS  : c,z

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              HIPBLAS_FILL_MODE_UPPER: the upper triangular part of the Hermitian matrix A is supplied.
              HIPBLAS_FILL_MODE_LOWER: the lower triangular part of the Hermitian matrix A is supplied.
    @param[in]
    n         [int]
              the order of the matrix A.
    @param[in]
    alpha     device pointer or host pointer to scalar alpha.
    @param[in]
    AP        device pointer storing matrix A. Of dimension (lda, n).
              if uplo == HIPBLAS_FILL_MODE_UPPER:
                The upper triangular part of A must contain
                the upper triangular part of a Hermitian matrix. The lower
                triangular part of A will not be referenced.
              if uplo == HIPBLAS_FILL_MODE_LOWER:
                The lower triangular part of A must contain
                the lower triangular part of a Hermitian matrix. The upper
                triangular part of A will not be referenced.
              As a Hermitian matrix, the imaginary part of the main diagonal
              of A will not be referenced and is assumed to be == 0.
    @param[in]
    lda       [int]
              specifies the leading dimension of A. must be >= max(1, n)
    @param[in]
    x         device pointer storing vector x.
    @param[in]
    incx      [int]
              specifies the increment for the elements of x.
    @param[in]
    beta      device pointer or host pointer to scalar beta.
    @param[inout]
    y         device pointer storing vector y.
    @param[in]
    incy      [int]
              specifies the increment for the elements of y.

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasChemv(hipblasHandle_t       handle,
                                            hipblasFillMode_t     uplo,
                                            int                   n,
                                            const hipblasComplex* alpha,
                                            const hipblasComplex* AP,
                                            int                   lda,
                                            const hipblasComplex* x,
                                            int                   incx,
                                            const hipblasComplex* beta,
                                            hipblasComplex*       y,
                                            int                   incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhemv(hipblasHandle_t             handle,
                                            hipblasFillMode_t           uplo,
                                            int                         n,
                                            const hipblasDoubleComplex* alpha,
                                            const hipblasDoubleComplex* AP,
                                            int                         lda,
                                            const hipblasDoubleComplex* x,
                                            int                         incx,
                                            const hipblasDoubleComplex* beta,
                                            hipblasDoubleComplex*       y,
                                            int                         incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasChemv_v2(hipblasHandle_t   handle,
                                               hipblasFillMode_t uplo,
                                               int               n,
                                               const hipComplex* alpha,
                                               const hipComplex* AP,
                                               int               lda,
                                               const hipComplex* x,
                                               int               incx,
                                               const hipComplex* beta,
                                               hipComplex*       y,
                                               int               incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhemv_v2(hipblasHandle_t         handle,
                                               hipblasFillMode_t       uplo,
                                               int                     n,
                                               const hipDoubleComplex* alpha,
                                               const hipDoubleComplex* AP,
                                               int                     lda,
                                               const hipDoubleComplex* x,
                                               int                     incx,
                                               const hipDoubleComplex* beta,
                                               hipDoubleComplex*       y,
                                               int                     incy);

//64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasChemv_64(hipblasHandle_t       handle,
                                               hipblasFillMode_t     uplo,
                                               int64_t               n,
                                               const hipblasComplex* alpha,
                                               const hipblasComplex* AP,
                                               int64_t               lda,
                                               const hipblasComplex* x,
                                               int64_t               incx,
                                               const hipblasComplex* beta,
                                               hipblasComplex*       y,
                                               int64_t               incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhemv_64(hipblasHandle_t             handle,
                                               hipblasFillMode_t           uplo,
                                               int64_t                     n,
                                               const hipblasDoubleComplex* alpha,
                                               const hipblasDoubleComplex* AP,
                                               int64_t                     lda,
                                               const hipblasDoubleComplex* x,
                                               int64_t                     incx,
                                               const hipblasDoubleComplex* beta,
                                               hipblasDoubleComplex*       y,
                                               int64_t                     incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasChemv_v2_64(hipblasHandle_t   handle,
                                                  hipblasFillMode_t uplo,
                                                  int64_t           n,
                                                  const hipComplex* alpha,
                                                  const hipComplex* AP,
                                                  int64_t           lda,
                                                  const hipComplex* x,
                                                  int64_t           incx,
                                                  const hipComplex* beta,
                                                  hipComplex*       y,
                                                  int64_t           incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhemv_v2_64(hipblasHandle_t         handle,
                                                  hipblasFillMode_t       uplo,
                                                  int64_t                 n,
                                                  const hipDoubleComplex* alpha,
                                                  const hipDoubleComplex* AP,
                                                  int64_t                 lda,
                                                  const hipDoubleComplex* x,
                                                  int64_t                 incx,
                                                  const hipDoubleComplex* beta,
                                                  hipDoubleComplex*       y,
                                                  int64_t                 incy);

//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    hemvBatched performs one of the matrix-vector operations

        y_i := alpha*A_i*x_i + beta*y_i

    where alpha and beta are scalars, x_i and y_i are n element vectors and A_i is an
    n by n Hermitian matrix, for each batch in i = [1, batchCount].

    - Supported precisions in rocBLAS : c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              HIPBLAS_FILL_MODE_UPPER: the upper triangular part of the Hermitian matrix A is supplied.
              HIPBLAS_FILL_MODE_LOWER: the lower triangular part of the Hermitian matrix A is supplied.
    @param[in]
    n         [int]
              the order of each matrix A_i.
    @param[in]
    alpha     device pointer or host pointer to scalar alpha.
    @param[in]
    AP        device array of device pointers storing each matrix A_i of dimension (lda, n).
              if uplo == HIPBLAS_FILL_MODE_UPPER:
                The upper triangular part of each A_i must contain
                the upper triangular part of a Hermitian matrix. The lower
                triangular part of each A_i will not be referenced.
              if uplo == HIPBLAS_FILL_MODE_LOWER:
                The lower triangular part of each A_i must contain
                the lower triangular part of a Hermitian matrix. The upper
                triangular part of each A_i will not be referenced.
              As a Hermitian matrix, the imaginary part of the main diagonal
              of each A_i will not be referenced and is assumed to be == 0.
    @param[in]
    lda       [int]
              specifies the leading dimension of each A_i. must be >= max(1, n)
    @param[in]
    x         device array of device pointers storing each vector x_i.
    @param[in]
    incx      [int]
              specifies the increment for the elements of each x_i.
    @param[in]
    beta      device pointer or host pointer to scalar beta.
    @param[inout]
    y         device array of device pointers storing each vector y_i.
    @param[in]
    incy      [int]
              specifies the increment for the elements of y.
    @param[in]
    batchCount [int]
                number of instances in the batch.

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasChemvBatched(hipblasHandle_t             handle,
                                                   hipblasFillMode_t           uplo,
                                                   int                         n,
                                                   const hipblasComplex*       alpha,
                                                   const hipblasComplex* const AP[],
                                                   int                         lda,
                                                   const hipblasComplex* const x[],
                                                   int                         incx,
                                                   const hipblasComplex*       beta,
                                                   hipblasComplex* const       y[],
                                                   int                         incy,
                                                   int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhemvBatched(hipblasHandle_t                   handle,
                                                   hipblasFillMode_t                 uplo,
                                                   int                               n,
                                                   const hipblasDoubleComplex*       alpha,
                                                   const hipblasDoubleComplex* const AP[],
                                                   int                               lda,
                                                   const hipblasDoubleComplex* const x[],
                                                   int                               incx,
                                                   const hipblasDoubleComplex*       beta,
                                                   hipblasDoubleComplex* const       y[],
                                                   int                               incy,
                                                   int                               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasChemvBatched_v2(hipblasHandle_t         handle,
                                                      hipblasFillMode_t       uplo,
                                                      int                     n,
                                                      const hipComplex*       alpha,
                                                      const hipComplex* const AP[],
                                                      int                     lda,
                                                      const hipComplex* const x[],
                                                      int                     incx,
                                                      const hipComplex*       beta,
                                                      hipComplex* const       y[],
                                                      int                     incy,
                                                      int                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhemvBatched_v2(hipblasHandle_t               handle,
                                                      hipblasFillMode_t             uplo,
                                                      int                           n,
                                                      const hipDoubleComplex*       alpha,
                                                      const hipDoubleComplex* const AP[],
                                                      int                           lda,
                                                      const hipDoubleComplex* const x[],
                                                      int                           incx,
                                                      const hipDoubleComplex*       beta,
                                                      hipDoubleComplex* const       y[],
                                                      int                           incy,
                                                      int                           batchCount);

//64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasChemvBatched_64(hipblasHandle_t             handle,
                                                      hipblasFillMode_t           uplo,
                                                      int64_t                     n,
                                                      const hipblasComplex*       alpha,
                                                      const hipblasComplex* const AP[],
                                                      int64_t                     lda,
                                                      const hipblasComplex* const x[],
                                                      int64_t                     incx,
                                                      const hipblasComplex*       beta,
                                                      hipblasComplex* const       y[],
                                                      int64_t                     incy,
                                                      int64_t                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhemvBatched_64(hipblasHandle_t                   handle,
                                                      hipblasFillMode_t                 uplo,
                                                      int64_t                           n,
                                                      const hipblasDoubleComplex*       alpha,
                                                      const hipblasDoubleComplex* const AP[],
                                                      int64_t                           lda,
                                                      const hipblasDoubleComplex* const x[],
                                                      int64_t                           incx,
                                                      const hipblasDoubleComplex*       beta,
                                                      hipblasDoubleComplex* const       y[],
                                                      int64_t                           incy,
                                                      int64_t                           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasChemvBatched_v2_64(hipblasHandle_t         handle,
                                                         hipblasFillMode_t       uplo,
                                                         int64_t                 n,
                                                         const hipComplex*       alpha,
                                                         const hipComplex* const AP[],
                                                         int64_t                 lda,
                                                         const hipComplex* const x[],
                                                         int64_t                 incx,
                                                         const hipComplex*       beta,
                                                         hipComplex* const       y[],
                                                         int64_t                 incy,
                                                         int64_t                 batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhemvBatched_v2_64(hipblasHandle_t               handle,
                                                         hipblasFillMode_t             uplo,
                                                         int64_t                       n,
                                                         const hipDoubleComplex*       alpha,
                                                         const hipDoubleComplex* const AP[],
                                                         int64_t                       lda,
                                                         const hipDoubleComplex* const x[],
                                                         int64_t                       incx,
                                                         const hipDoubleComplex*       beta,
                                                         hipDoubleComplex* const       y[],
                                                         int64_t                       incy,
                                                         int64_t                       batchCount);

//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    hemvStridedBatched performs one of the matrix-vector operations

        y_i := alpha*A_i*x_i + beta*y_i

    where alpha and beta are scalars, x_i and y_i are n element vectors and A_i is an
    n by n Hermitian matrix, for each batch in i = [1, batchCount].

    - Supported precisions in rocBLAS : c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              HIPBLAS_FILL_MODE_UPPER: the upper triangular part of the Hermitian matrix A is supplied.
              HIPBLAS_FILL_MODE_LOWER: the lower triangular part of the Hermitian matrix A is supplied.
    @param[in]
    n         [int]
              the order of each matrix A_i.
    @param[in]
    alpha     device pointer or host pointer to scalar alpha.
    @param[in]
    AP        device array of device pointers storing each matrix A_i of dimension (lda, n).
              if uplo == HIPBLAS_FILL_MODE_UPPER:
                The upper triangular part of each A_i must contain
                the upper triangular part of a Hermitian matrix. The lower
                triangular part of each A_i will not be referenced.
              if uplo == HIPBLAS_FILL_MODE_LOWER:
                The lower triangular part of each A_i must contain
                the lower triangular part of a Hermitian matrix. The upper
                triangular part of each A_i will not be referenced.
              As a Hermitian matrix, the imaginary part of the main diagonal
              of each A_i will not be referenced and is assumed to be == 0.
    @param[in]
    lda       [int]
              specifies the leading dimension of each A_i. must be >= max(1, n)
    @param[in]
    strideA    [hipblasStride]
                stride from the start of one (A_i) to the next (A_i+1)

    @param[in]
    x         device array of device pointers storing each vector x_i.
    @param[in]
    incx      [int]
              specifies the increment for the elements of each x_i.
    @param[in]
    stridex  [hipblasStride]
              stride from the start of one vector (x_i) and the next one (x_i+1).
    @param[in]
    beta      device pointer or host pointer to scalar beta.
    @param[inout]
    y         device array of device pointers storing each vector y_i.
    @param[in]
    incy      [int]
              specifies the increment for the elements of y.
    @param[in]
    stridey  [hipblasStride]
              stride from the start of one vector (y_i) and the next one (y_i+1).
    @param[in]
    batchCount [int]
                number of instances in the batch.

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasChemvStridedBatched(hipblasHandle_t       handle,
                                                          hipblasFillMode_t     uplo,
                                                          int                   n,
                                                          const hipblasComplex* alpha,
                                                          const hipblasComplex* AP,
                                                          int                   lda,
                                                          hipblasStride         strideA,
                                                          const hipblasComplex* x,
                                                          int                   incx,
                                                          hipblasStride         stridex,
                                                          const hipblasComplex* beta,
                                                          hipblasComplex*       y,
                                                          int                   incy,
                                                          hipblasStride         stridey,
                                                          int                   batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhemvStridedBatched(hipblasHandle_t             handle,
                                                          hipblasFillMode_t           uplo,
                                                          int                         n,
                                                          const hipblasDoubleComplex* alpha,
                                                          const hipblasDoubleComplex* AP,
                                                          int                         lda,
                                                          hipblasStride               strideA,
                                                          const hipblasDoubleComplex* x,
                                                          int                         incx,
                                                          hipblasStride               stridex,
                                                          const hipblasDoubleComplex* beta,
                                                          hipblasDoubleComplex*       y,
                                                          int                         incy,
                                                          hipblasStride               stridey,
                                                          int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasChemvStridedBatched_v2(hipblasHandle_t   handle,
                                                             hipblasFillMode_t uplo,
                                                             int               n,
                                                             const hipComplex* alpha,
                                                             const hipComplex* AP,
                                                             int               lda,
                                                             hipblasStride     strideA,
                                                             const hipComplex* x,
                                                             int               incx,
                                                             hipblasStride     stridex,
                                                             const hipComplex* beta,
                                                             hipComplex*       y,
                                                             int               incy,
                                                             hipblasStride     stridey,
                                                             int               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhemvStridedBatched_v2(hipblasHandle_t         handle,
                                                             hipblasFillMode_t       uplo,
                                                             int                     n,
                                                             const hipDoubleComplex* alpha,
                                                             const hipDoubleComplex* AP,
                                                             int                     lda,
                                                             hipblasStride           strideA,
                                                             const hipDoubleComplex* x,
                                                             int                     incx,
                                                             hipblasStride           stridex,
                                                             const hipDoubleComplex* beta,
                                                             hipDoubleComplex*       y,
                                                             int                     incy,
                                                             hipblasStride           stridey,
                                                             int                     batchCount);

//64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasChemvStridedBatched_64(hipblasHandle_t       handle,
                                                             hipblasFillMode_t     uplo,
                                                             int64_t               n,
                                                             const hipblasComplex* alpha,
                                                             const hipblasComplex* AP,
                                                             int64_t               lda,
                                                             hipblasStride         strideA,
                                                             const hipblasComplex* x,
                                                             int64_t               incx,
                                                             hipblasStride         stridex,
                                                             const hipblasComplex* beta,
                                                             hipblasComplex*       y,
                                                             int64_t               incy,
                                                             hipblasStride         stridey,
                                                             int64_t               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhemvStridedBatched_64(hipblasHandle_t             handle,
                                                             hipblasFillMode_t           uplo,
                                                             int64_t                     n,
                                                             const hipblasDoubleComplex* alpha,
                                                             const hipblasDoubleComplex* AP,
                                                             int64_t                     lda,
                                                             hipblasStride               strideA,
                                                             const hipblasDoubleComplex* x,
                                                             int64_t                     incx,
                                                             hipblasStride               stridex,
                                                             const hipblasDoubleComplex* beta,
                                                             hipblasDoubleComplex*       y,
                                                             int64_t                     incy,
                                                             hipblasStride               stridey,
                                                             int64_t batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasChemvStridedBatched_v2_64(hipblasHandle_t   handle,
                                                                hipblasFillMode_t uplo,
                                                                int64_t           n,
                                                                const hipComplex* alpha,
                                                                const hipComplex* AP,
                                                                int64_t           lda,
                                                                hipblasStride     strideA,
                                                                const hipComplex* x,
                                                                int64_t           incx,
                                                                hipblasStride     stridex,
                                                                const hipComplex* beta,
                                                                hipComplex*       y,
                                                                int64_t           incy,
                                                                hipblasStride     stridey,
                                                                int64_t           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhemvStridedBatched_v2_64(hipblasHandle_t         handle,
                                                                hipblasFillMode_t       uplo,
                                                                int64_t                 n,
                                                                const hipDoubleComplex* alpha,
                                                                const hipDoubleComplex* AP,
                                                                int64_t                 lda,
                                                                hipblasStride           strideA,
                                                                const hipDoubleComplex* x,
                                                                int64_t                 incx,
                                                                hipblasStride           stridex,
                                                                const hipDoubleComplex* beta,
                                                                hipDoubleComplex*       y,
                                                                int64_t                 incy,
                                                                hipblasStride           stridey,
                                                                int64_t                 batchCount);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    her performs the matrix-vector operations

        A := A + alpha*x*x**H

    where alpha is a real scalar, x is a vector, and A is an
    n by n Hermitian matrix.

    - Supported precisions in rocBLAS : c,z
    - Supported precisions in cuBLAS  : c,z

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
              HIPBLAS_FILL_MODE_UPPER: The upper triangular part of A is supplied in A.
              HIPBLAS_FILL_MODE_LOWER: The lower triangular part of A is supplied in A.
    @param[in]
    n         [int]
              the number of rows and columns of matrix A, must be at least 0.
    @param[in]
    alpha
              device pointer or host pointer to scalar alpha.
    @param[in]
    x         device pointer storing vector x.
    @param[in]
    incx      [int]
              specifies the increment for the elements of x.
    @param[inout]
    AP        device pointer storing the specified triangular portion of
              the Hermitian matrix A. Of size (lda * n).
              if uplo == HIPBLAS_FILL_MODE_UPPER:
                The upper triangular portion of the Hermitian matrix A is supplied. The lower
                triangluar portion will not be touched.
            if uplo == HIPBLAS_FILL_MODE_LOWER:
                The lower triangular portion of the Hermitian matrix A is supplied. The upper
                triangular portion will not be touched.
            Note that the imaginary part of the diagonal elements are not accessed and are assumed
            to be 0.
    @param[in]
    lda       [int]
              specifies the leading dimension of A. Must be at least max(1, n).
    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasCher(hipblasHandle_t       handle,
                                           hipblasFillMode_t     uplo,
                                           int                   n,
                                           const float*          alpha,
                                           const hipblasComplex* x,
                                           int                   incx,
                                           hipblasComplex*       AP,
                                           int                   lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasZher(hipblasHandle_t             handle,
                                           hipblasFillMode_t           uplo,
                                           int                         n,
                                           const double*               alpha,
                                           const hipblasDoubleComplex* x,
                                           int                         incx,
                                           hipblasDoubleComplex*       AP,
                                           int                         lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasCher_v2(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int               n,
                                              const float*      alpha,
                                              const hipComplex* x,
                                              int               incx,
                                              hipComplex*       AP,
                                              int               lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasZher_v2(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              int                     n,
                                              const double*           alpha,
                                              const hipDoubleComplex* x,
                                              int                     incx,
                                              hipDoubleComplex*       AP,
                                              int                     lda);
//64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasCher_64(hipblasHandle_t       handle,
                                              hipblasFillMode_t     uplo,
                                              int64_t               n,
                                              const float*          alpha,
                                              const hipblasComplex* x,
                                              int64_t               incx,
                                              hipblasComplex*       AP,
                                              int64_t               lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasZher_64(hipblasHandle_t             handle,
                                              hipblasFillMode_t           uplo,
                                              int64_t                     n,
                                              const double*               alpha,
                                              const hipblasDoubleComplex* x,
                                              int64_t                     incx,
                                              hipblasDoubleComplex*       AP,
                                              int64_t                     lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasCher_v2_64(hipblasHandle_t   handle,
                                                 hipblasFillMode_t uplo,
                                                 int64_t           n,
                                                 const float*      alpha,
                                                 const hipComplex* x,
                                                 int64_t           incx,
                                                 hipComplex*       AP,
                                                 int64_t           lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasZher_v2_64(hipblasHandle_t         handle,
                                                 hipblasFillMode_t       uplo,
                                                 int64_t                 n,
                                                 const double*           alpha,
                                                 const hipDoubleComplex* x,
                                                 int64_t                 incx,
                                                 hipDoubleComplex*       AP,
                                                 int64_t                 lda);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    herBatched performs the matrix-vector operations

        A_i := A_i + alpha*x_i*x_i**H

    where alpha is a real scalar, x_i is a vector, and A_i is an
    n by n symmetric matrix, for i = 1, ..., batchCount.

    - Supported precisions in rocBLAS : c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
              HIPBLAS_FILL_MODE_UPPER: The upper triangular part of each A_i is supplied in A.
              HIPBLAS_FILL_MODE_LOWER: The lower triangular part of each A_i is supplied in A.
    @param[in]
    n         [int]
              the number of rows and columns of each matrix A_i, must be at least 0.
    @param[in]
    alpha
              device pointer or host pointer to scalar alpha.
    @param[in]
    x         device array of device pointers storing each vector x_i.
    @param[in]
    incx      [int]
              specifies the increment for the elements of each x_i.
    @param[inout]
    AP       device array of device pointers storing the specified triangular portion of
              each Hermitian matrix A_i of at least size ((n * (n + 1)) / 2). Array is of at least size batchCount.
              if uplo == HIPBLAS_FILL_MODE_UPPER:
                The upper triangular portion of each Hermitian matrix A_i is supplied. The lower triangular portion
                of each A_i will not be touched.
            if uplo == HIPBLAS_FILL_MODE_LOWER:
                The lower triangular portion of each Hermitian matrix A_i is supplied. The upper triangular portion
                of each A_i will not be touched.
            Note that the imaginary part of the diagonal elements are not accessed and are assumed
            to be 0.
    @param[in]
    lda       [int]
              specifies the leading dimension of each A_i. Must be at least max(1, n).
    @param[in]
    batchCount [int]
                number of instances in the batch.
    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasCherBatched(hipblasHandle_t             handle,
                                                  hipblasFillMode_t           uplo,
                                                  int                         n,
                                                  const float*                alpha,
                                                  const hipblasComplex* const x[],
                                                  int                         incx,
                                                  hipblasComplex* const       AP[],
                                                  int                         lda,
                                                  int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZherBatched(hipblasHandle_t                   handle,
                                                  hipblasFillMode_t                 uplo,
                                                  int                               n,
                                                  const double*                     alpha,
                                                  const hipblasDoubleComplex* const x[],
                                                  int                               incx,
                                                  hipblasDoubleComplex* const       AP[],
                                                  int                               lda,
                                                  int                               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCherBatched_v2(hipblasHandle_t         handle,
                                                     hipblasFillMode_t       uplo,
                                                     int                     n,
                                                     const float*            alpha,
                                                     const hipComplex* const x[],
                                                     int                     incx,
                                                     hipComplex* const       AP[],
                                                     int                     lda,
                                                     int                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZherBatched_v2(hipblasHandle_t               handle,
                                                     hipblasFillMode_t             uplo,
                                                     int                           n,
                                                     const double*                 alpha,
                                                     const hipDoubleComplex* const x[],
                                                     int                           incx,
                                                     hipDoubleComplex* const       AP[],
                                                     int                           lda,
                                                     int                           batchCount);

//64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasCherBatched_64(hipblasHandle_t             handle,
                                                     hipblasFillMode_t           uplo,
                                                     int64_t                     n,
                                                     const float*                alpha,
                                                     const hipblasComplex* const x[],
                                                     int64_t                     incx,
                                                     hipblasComplex* const       AP[],
                                                     int64_t                     lda,
                                                     int64_t                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZherBatched_64(hipblasHandle_t                   handle,
                                                     hipblasFillMode_t                 uplo,
                                                     int64_t                           n,
                                                     const double*                     alpha,
                                                     const hipblasDoubleComplex* const x[],
                                                     int64_t                           incx,
                                                     hipblasDoubleComplex* const       AP[],
                                                     int64_t                           lda,
                                                     int64_t                           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCherBatched_v2_64(hipblasHandle_t         handle,
                                                        hipblasFillMode_t       uplo,
                                                        int64_t                 n,
                                                        const float*            alpha,
                                                        const hipComplex* const x[],
                                                        int64_t                 incx,
                                                        hipComplex* const       AP[],
                                                        int64_t                 lda,
                                                        int64_t                 batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZherBatched_v2_64(hipblasHandle_t               handle,
                                                        hipblasFillMode_t             uplo,
                                                        int64_t                       n,
                                                        const double*                 alpha,
                                                        const hipDoubleComplex* const x[],
                                                        int64_t                       incx,
                                                        hipDoubleComplex* const       AP[],
                                                        int64_t                       lda,
                                                        int64_t                       batchCount);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    herStridedBatched performs the matrix-vector operations

        A_i := A_i + alpha*x_i*x_i**H

    where alpha is a real scalar, x_i is a vector, and A_i is an
    n by n Hermitian matrix, for i = 1, ..., batchCount.

    - Supported precisions in rocBLAS : c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
              HIPBLAS_FILL_MODE_UPPER: The upper triangular part of each A_i is supplied in A.
              HIPBLAS_FILL_MODE_LOWER: The lower triangular part of each A_i is supplied in A.
    @param[in]
    n         [int]
              the number of rows and columns of each matrix A_i, must be at least 0.
    @param[in]
    alpha
              device pointer or host pointer to scalar alpha.
    @param[in]
    x         device pointer pointing to the first vector (x_1).
    @param[in]
    incx      [int]
              specifies the increment for the elements of each x_i.
    @param[in]
    stridex  [hipblasStride]
              stride from the start of one vector (x_i) and the next one (x_i+1).
    @param[inout]
    AP        device array of device pointers storing the specified triangular portion of
              each Hermitian matrix A_i. Points to the first matrix (A_1).
              if uplo == HIPBLAS_FILL_MODE_UPPER:
                The upper triangular portion of each Hermitian matrix A_i is supplied. The lower triangular
                portion of each A_i will not be touched.
            if uplo == HIPBLAS_FILL_MODE_LOWER:
                The lower triangular portion of each Hermitian matrix A_i is supplied. The upper triangular
                portion of each A_i will not be touched.
            Note that the imaginary part of the diagonal elements are not accessed and are assumed
            to be 0.
    @param[in]
    lda       [int]
              specifies the leading dimension of each A_i.
    @param[in]
    strideA    [hipblasStride]
                stride from the start of one (A_i) and the next (A_i+1)
    @param[in]
    batchCount [int]
                number of instances in the batch.
    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasCherStridedBatched(hipblasHandle_t       handle,
                                                         hipblasFillMode_t     uplo,
                                                         int                   n,
                                                         const float*          alpha,
                                                         const hipblasComplex* x,
                                                         int                   incx,
                                                         hipblasStride         stridex,
                                                         hipblasComplex*       AP,
                                                         int                   lda,
                                                         hipblasStride         strideA,
                                                         int                   batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZherStridedBatched(hipblasHandle_t             handle,
                                                         hipblasFillMode_t           uplo,
                                                         int                         n,
                                                         const double*               alpha,
                                                         const hipblasDoubleComplex* x,
                                                         int                         incx,
                                                         hipblasStride               stridex,
                                                         hipblasDoubleComplex*       AP,
                                                         int                         lda,
                                                         hipblasStride               strideA,
                                                         int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCherStridedBatched_v2(hipblasHandle_t   handle,
                                                            hipblasFillMode_t uplo,
                                                            int               n,
                                                            const float*      alpha,
                                                            const hipComplex* x,
                                                            int               incx,
                                                            hipblasStride     stridex,
                                                            hipComplex*       AP,
                                                            int               lda,
                                                            hipblasStride     strideA,
                                                            int               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZherStridedBatched_v2(hipblasHandle_t         handle,
                                                            hipblasFillMode_t       uplo,
                                                            int                     n,
                                                            const double*           alpha,
                                                            const hipDoubleComplex* x,
                                                            int                     incx,
                                                            hipblasStride           stridex,
                                                            hipDoubleComplex*       AP,
                                                            int                     lda,
                                                            hipblasStride           strideA,
                                                            int                     batchCount);
//64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasCherStridedBatched_64(hipblasHandle_t       handle,
                                                            hipblasFillMode_t     uplo,
                                                            int64_t               n,
                                                            const float*          alpha,
                                                            const hipblasComplex* x,
                                                            int64_t               incx,
                                                            hipblasStride         stridex,
                                                            hipblasComplex*       AP,
                                                            int64_t               lda,
                                                            hipblasStride         strideA,
                                                            int64_t               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZherStridedBatched_64(hipblasHandle_t             handle,
                                                            hipblasFillMode_t           uplo,
                                                            int64_t                     n,
                                                            const double*               alpha,
                                                            const hipblasDoubleComplex* x,
                                                            int64_t                     incx,
                                                            hipblasStride               stridex,
                                                            hipblasDoubleComplex*       AP,
                                                            int64_t                     lda,
                                                            hipblasStride               strideA,
                                                            int64_t                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCherStridedBatched_v2_64(hipblasHandle_t   handle,
                                                               hipblasFillMode_t uplo,
                                                               int64_t           n,
                                                               const float*      alpha,
                                                               const hipComplex* x,
                                                               int64_t           incx,
                                                               hipblasStride     stridex,
                                                               hipComplex*       AP,
                                                               int64_t           lda,
                                                               hipblasStride     strideA,
                                                               int64_t           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZherStridedBatched_v2_64(hipblasHandle_t         handle,
                                                               hipblasFillMode_t       uplo,
                                                               int64_t                 n,
                                                               const double*           alpha,
                                                               const hipDoubleComplex* x,
                                                               int64_t                 incx,
                                                               hipblasStride           stridex,
                                                               hipDoubleComplex*       AP,
                                                               int64_t                 lda,
                                                               hipblasStride           strideA,
                                                               int64_t                 batchCount);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    her2 performs the matrix-vector operations

        A := A + alpha*x*y**H + conj(alpha)*y*x**H

    where alpha is a complex scalar, x and y are vectors, and A is an
    n by n Hermitian matrix.

    - Supported precisions in rocBLAS : c,z
    - Supported precisions in cuBLAS  : c,z

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
              HIPBLAS_FILL_MODE_UPPER: The upper triangular part of A is supplied.
              HIPBLAS_FILL_MODE_LOWER: The lower triangular part of A is supplied.
    @param[in]
    n         [int]
              the number of rows and columns of matrix A, must be at least 0.
    @param[in]
    alpha
              device pointer or host pointer to scalar alpha.
    @param[in]
    x         device pointer storing vector x.
    @param[in]
    incx      [int]
              specifies the increment for the elements of x.
    @param[in]
    y         device pointer storing vector y.
    @param[in]
    incy      [int]
              specifies the increment for the elements of y.
    @param[inout]
    AP         device pointer storing the specified triangular portion of
              the Hermitian matrix A. Of size (lda, n).
              if uplo == HIPBLAS_FILL_MODE_UPPER:
                The upper triangular portion of the Hermitian matrix A is supplied. The lower triangular
                portion of A will not be touched.
            if uplo == HIPBLAS_FILL_MODE_LOWER:
                The lower triangular portion of the Hermitian matrix A is supplied. The upper triangular
                portion of A will not be touched.
            Note that the imaginary part of the diagonal elements are not accessed and are assumed
            to be 0.
    @param[in]
    lda       [int]
              specifies the leading dimension of A. Must be at least max(lda, 1).
    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasCher2(hipblasHandle_t       handle,
                                            hipblasFillMode_t     uplo,
                                            int                   n,
                                            const hipblasComplex* alpha,
                                            const hipblasComplex* x,
                                            int                   incx,
                                            const hipblasComplex* y,
                                            int                   incy,
                                            hipblasComplex*       AP,
                                            int                   lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasZher2(hipblasHandle_t             handle,
                                            hipblasFillMode_t           uplo,
                                            int                         n,
                                            const hipblasDoubleComplex* alpha,
                                            const hipblasDoubleComplex* x,
                                            int                         incx,
                                            const hipblasDoubleComplex* y,
                                            int                         incy,
                                            hipblasDoubleComplex*       AP,
                                            int                         lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasCher2_v2(hipblasHandle_t   handle,
                                               hipblasFillMode_t uplo,
                                               int               n,
                                               const hipComplex* alpha,
                                               const hipComplex* x,
                                               int               incx,
                                               const hipComplex* y,
                                               int               incy,
                                               hipComplex*       AP,
                                               int               lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasZher2_v2(hipblasHandle_t         handle,
                                               hipblasFillMode_t       uplo,
                                               int                     n,
                                               const hipDoubleComplex* alpha,
                                               const hipDoubleComplex* x,
                                               int                     incx,
                                               const hipDoubleComplex* y,
                                               int                     incy,
                                               hipDoubleComplex*       AP,
                                               int                     lda);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasCher2_64(hipblasHandle_t       handle,
                                               hipblasFillMode_t     uplo,
                                               int64_t               n,
                                               const hipblasComplex* alpha,
                                               const hipblasComplex* x,
                                               int64_t               incx,
                                               const hipblasComplex* y,
                                               int64_t               incy,
                                               hipblasComplex*       AP,
                                               int64_t               lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasZher2_64(hipblasHandle_t             handle,
                                               hipblasFillMode_t           uplo,
                                               int64_t                     n,
                                               const hipblasDoubleComplex* alpha,
                                               const hipblasDoubleComplex* x,
                                               int64_t                     incx,
                                               const hipblasDoubleComplex* y,
                                               int64_t                     incy,
                                               hipblasDoubleComplex*       AP,
                                               int64_t                     lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasCher2_v2_64(hipblasHandle_t   handle,
                                                  hipblasFillMode_t uplo,
                                                  int64_t           n,
                                                  const hipComplex* alpha,
                                                  const hipComplex* x,
                                                  int64_t           incx,
                                                  const hipComplex* y,
                                                  int64_t           incy,
                                                  hipComplex*       AP,
                                                  int64_t           lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasZher2_v2_64(hipblasHandle_t         handle,
                                                  hipblasFillMode_t       uplo,
                                                  int64_t                 n,
                                                  const hipDoubleComplex* alpha,
                                                  const hipDoubleComplex* x,
                                                  int64_t                 incx,
                                                  const hipDoubleComplex* y,
                                                  int64_t                 incy,
                                                  hipDoubleComplex*       AP,
                                                  int64_t                 lda);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    her2Batched performs the matrix-vector operations

        A_i := A_i + alpha*x_i*y_i**H + conj(alpha)*y_i*x_i**H

    where alpha is a complex scalar, x_i and y_i are vectors, and A_i is an
    n by n Hermitian matrix for each batch in i = [1, batchCount].

    - Supported precisions in rocBLAS : c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
              HIPBLAS_FILL_MODE_UPPER: The upper triangular part of each A_i is supplied.
              HIPBLAS_FILL_MODE_LOWER: The lower triangular part of each A_i is supplied.
    @param[in]
    n         [int]
              the number of rows and columns of each matrix A_i, must be at least 0.
    @param[in]
    alpha
              device pointer or host pointer to scalar alpha.
    @param[in]
    x         device array of device pointers storing each vector x_i.
    @param[in]
    incx      [int]
              specifies the increment for the elements of x.
    @param[in]
    y         device array of device pointers storing each vector y_i.
    @param[in]
    incy      [int]
              specifies the increment for the elements of each y_i.
    @param[inout]
    AP         device array of device pointers storing the specified triangular portion of
              each Hermitian matrix A_i of size (lda, n).
              if uplo == HIPBLAS_FILL_MODE_UPPER:
                The upper triangular portion of each Hermitian matrix A_i is supplied. The lower triangular
                portion of each A_i will not be touched.
            if uplo == HIPBLAS_FILL_MODE_LOWER:
                The lower triangular portion of each Hermitian matrix A_i is supplied. The upper triangular
                portion of each A_i will not be touched.
            Note that the imaginary part of the diagonal elements are not accessed and are assumed
            to be 0.
    @param[in]
    lda       [int]
              specifies the leading dimension of each A_i. Must be at least max(lda, 1).
    @param[in]
    batchCount [int]
                number of instances in the batch.
    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasCher2Batched(hipblasHandle_t             handle,
                                                   hipblasFillMode_t           uplo,
                                                   int                         n,
                                                   const hipblasComplex*       alpha,
                                                   const hipblasComplex* const x[],
                                                   int                         incx,
                                                   const hipblasComplex* const y[],
                                                   int                         incy,
                                                   hipblasComplex* const       AP[],
                                                   int                         lda,
                                                   int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZher2Batched(hipblasHandle_t                   handle,
                                                   hipblasFillMode_t                 uplo,
                                                   int                               n,
                                                   const hipblasDoubleComplex*       alpha,
                                                   const hipblasDoubleComplex* const x[],
                                                   int                               incx,
                                                   const hipblasDoubleComplex* const y[],
                                                   int                               incy,
                                                   hipblasDoubleComplex* const       AP[],
                                                   int                               lda,
                                                   int                               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCher2Batched_v2(hipblasHandle_t         handle,
                                                      hipblasFillMode_t       uplo,
                                                      int                     n,
                                                      const hipComplex*       alpha,
                                                      const hipComplex* const x[],
                                                      int                     incx,
                                                      const hipComplex* const y[],
                                                      int                     incy,
                                                      hipComplex* const       AP[],
                                                      int                     lda,
                                                      int                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZher2Batched_v2(hipblasHandle_t               handle,
                                                      hipblasFillMode_t             uplo,
                                                      int                           n,
                                                      const hipDoubleComplex*       alpha,
                                                      const hipDoubleComplex* const x[],
                                                      int                           incx,
                                                      const hipDoubleComplex* const y[],
                                                      int                           incy,
                                                      hipDoubleComplex* const       AP[],
                                                      int                           lda,
                                                      int                           batchCount);
// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasCher2Batched_64(hipblasHandle_t             handle,
                                                      hipblasFillMode_t           uplo,
                                                      int64_t                     n,
                                                      const hipblasComplex*       alpha,
                                                      const hipblasComplex* const x[],
                                                      int64_t                     incx,
                                                      const hipblasComplex* const y[],
                                                      int64_t                     incy,
                                                      hipblasComplex* const       AP[],
                                                      int64_t                     lda,
                                                      int64_t                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZher2Batched_64(hipblasHandle_t                   handle,
                                                      hipblasFillMode_t                 uplo,
                                                      int64_t                           n,
                                                      const hipblasDoubleComplex*       alpha,
                                                      const hipblasDoubleComplex* const x[],
                                                      int64_t                           incx,
                                                      const hipblasDoubleComplex* const y[],
                                                      int64_t                           incy,
                                                      hipblasDoubleComplex* const       AP[],
                                                      int64_t                           lda,
                                                      int64_t                           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCher2Batched_v2_64(hipblasHandle_t         handle,
                                                         hipblasFillMode_t       uplo,
                                                         int64_t                 n,
                                                         const hipComplex*       alpha,
                                                         const hipComplex* const x[],
                                                         int64_t                 incx,
                                                         const hipComplex* const y[],
                                                         int64_t                 incy,
                                                         hipComplex* const       AP[],
                                                         int64_t                 lda,
                                                         int64_t                 batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZher2Batched_v2_64(hipblasHandle_t               handle,
                                                         hipblasFillMode_t             uplo,
                                                         int64_t                       n,
                                                         const hipDoubleComplex*       alpha,
                                                         const hipDoubleComplex* const x[],
                                                         int64_t                       incx,
                                                         const hipDoubleComplex* const y[],
                                                         int64_t                       incy,
                                                         hipDoubleComplex* const       AP[],
                                                         int64_t                       lda,
                                                         int64_t                       batchCount);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    her2StridedBatched performs the matrix-vector operations

        A_i := A_i + alpha*x_i*y_i**H + conj(alpha)*y_i*x_i**H

    where alpha is a complex scalar, x_i and y_i are vectors, and A_i is an
    n by n Hermitian matrix for each batch in i = [1, batchCount].

    - Supported precisions in rocBLAS : c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
              HIPBLAS_FILL_MODE_UPPER: The upper triangular part of each A_i is supplied.
              HIPBLAS_FILL_MODE_LOWER: The lower triangular part of each A_i is supplied.
    @param[in]
    n         [int]
              the number of rows and columns of each matrix A_i, must be at least 0.
    @param[in]
    alpha
              device pointer or host pointer to scalar alpha.
    @param[in]
    x         device pointer pointing to the first vector x_1.
    @param[in]
    incx      [int]
              specifies the increment for the elements of each x_i.
    @param[in]
    stridex  [hipblasStride]
              specifies the stride between the beginning of one vector (x_i) and the next (x_i+1).
    @param[in]
    y         device pointer pointing to the first vector y_i.
    @param[in]
    incy      [int]
              specifies the increment for the elements of each y_i.
    @param[in]
    stridey  [hipblasStride]
              specifies the stride between the beginning of one vector (y_i) and the next (y_i+1).
    @param[inout]
    AP        device pointer pointing to the first matrix (A_1). Stores the specified triangular portion of
              each Hermitian matrix A_i.
              if uplo == HIPBLAS_FILL_MODE_UPPER:
                The upper triangular portion of each Hermitian matrix A_i is supplied. The lower triangular
                portion of each A_i will not be touched.
            if uplo == HIPBLAS_FILL_MODE_LOWER:
                The lower triangular portion of each Hermitian matrix A_i is supplied. The upper triangular
                portion of each A_i will not be touched.
            Note that the imaginary part of the diagonal elements are not accessed and are assumed
            to be 0.
    @param[in]
    lda       [int]
              specifies the leading dimension of each A_i. Must be at least max(lda, 1).
    @param[in]
    strideA  [hipblasStride]
              specifies the stride between the beginning of one matrix (A_i) and the next (A_i+1).
    @param[in]
    batchCount [int]
                number of instances in the batch.
    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasCher2StridedBatched(hipblasHandle_t       handle,
                                                          hipblasFillMode_t     uplo,
                                                          int                   n,
                                                          const hipblasComplex* alpha,
                                                          const hipblasComplex* x,
                                                          int                   incx,
                                                          hipblasStride         stridex,
                                                          const hipblasComplex* y,
                                                          int                   incy,
                                                          hipblasStride         stridey,
                                                          hipblasComplex*       AP,
                                                          int                   lda,
                                                          hipblasStride         strideA,
                                                          int                   batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZher2StridedBatched(hipblasHandle_t             handle,
                                                          hipblasFillMode_t           uplo,
                                                          int                         n,
                                                          const hipblasDoubleComplex* alpha,
                                                          const hipblasDoubleComplex* x,
                                                          int                         incx,
                                                          hipblasStride               stridex,
                                                          const hipblasDoubleComplex* y,
                                                          int                         incy,
                                                          hipblasStride               stridey,
                                                          hipblasDoubleComplex*       AP,
                                                          int                         lda,
                                                          hipblasStride               strideA,
                                                          int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCher2StridedBatched_v2(hipblasHandle_t   handle,
                                                             hipblasFillMode_t uplo,
                                                             int               n,
                                                             const hipComplex* alpha,
                                                             const hipComplex* x,
                                                             int               incx,
                                                             hipblasStride     stridex,
                                                             const hipComplex* y,
                                                             int               incy,
                                                             hipblasStride     stridey,
                                                             hipComplex*       AP,
                                                             int               lda,
                                                             hipblasStride     strideA,
                                                             int               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZher2StridedBatched_v2(hipblasHandle_t         handle,
                                                             hipblasFillMode_t       uplo,
                                                             int                     n,
                                                             const hipDoubleComplex* alpha,
                                                             const hipDoubleComplex* x,
                                                             int                     incx,
                                                             hipblasStride           stridex,
                                                             const hipDoubleComplex* y,
                                                             int                     incy,
                                                             hipblasStride           stridey,
                                                             hipDoubleComplex*       AP,
                                                             int                     lda,
                                                             hipblasStride           strideA,
                                                             int                     batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasCher2StridedBatched_64(hipblasHandle_t       handle,
                                                             hipblasFillMode_t     uplo,
                                                             int64_t               n,
                                                             const hipblasComplex* alpha,
                                                             const hipblasComplex* x,
                                                             int64_t               incx,
                                                             hipblasStride         stridex,
                                                             const hipblasComplex* y,
                                                             int64_t               incy,
                                                             hipblasStride         stridey,
                                                             hipblasComplex*       AP,
                                                             int64_t               lda,
                                                             hipblasStride         strideA,
                                                             int64_t               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZher2StridedBatched_64(hipblasHandle_t             handle,
                                                             hipblasFillMode_t           uplo,
                                                             int64_t                     n,
                                                             const hipblasDoubleComplex* alpha,
                                                             const hipblasDoubleComplex* x,
                                                             int64_t                     incx,
                                                             hipblasStride               stridex,
                                                             const hipblasDoubleComplex* y,
                                                             int64_t                     incy,
                                                             hipblasStride               stridey,
                                                             hipblasDoubleComplex*       AP,
                                                             int64_t                     lda,
                                                             hipblasStride               strideA,
                                                             int64_t batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCher2StridedBatched_v2_64(hipblasHandle_t   handle,
                                                                hipblasFillMode_t uplo,
                                                                int64_t           n,
                                                                const hipComplex* alpha,
                                                                const hipComplex* x,
                                                                int64_t           incx,
                                                                hipblasStride     stridex,
                                                                const hipComplex* y,
                                                                int64_t           incy,
                                                                hipblasStride     stridey,
                                                                hipComplex*       AP,
                                                                int64_t           lda,
                                                                hipblasStride     strideA,
                                                                int64_t           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZher2StridedBatched_v2_64(hipblasHandle_t         handle,
                                                                hipblasFillMode_t       uplo,
                                                                int64_t                 n,
                                                                const hipDoubleComplex* alpha,
                                                                const hipDoubleComplex* x,
                                                                int64_t                 incx,
                                                                hipblasStride           stridex,
                                                                const hipDoubleComplex* y,
                                                                int64_t                 incy,
                                                                hipblasStride           stridey,
                                                                hipDoubleComplex*       AP,
                                                                int64_t                 lda,
                                                                hipblasStride           strideA,
                                                                int64_t                 batchCount);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    hpmv performs the matrix-vector operation

        y := alpha*A*x + beta*y

    where alpha and beta are scalars, x and y are n element vectors and A is an
    n by n Hermitian matrix, supplied in packed form (see description below).

    - Supported precisions in rocBLAS : c,z
    - Supported precisions in cuBLAS  : c,z

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              HIPBLAS_FILL_MODE_UPPER: the upper triangular part of the Hermitian matrix A is supplied in AP.
              HIPBLAS_FILL_MODE_LOWER: the lower triangular part of the Hermitian matrix A is supplied in AP.
    @param[in]
    n         [int]
              the order of the matrix A, must be >= 0.
    @param[in]
    alpha     device pointer or host pointer to scalar alpha.
    @param[in]
    AP        device pointer storing the packed version of the specified triangular portion of
              the Hermitian matrix A. Of at least size ((n * (n + 1)) / 2).
              if uplo == HIPBLAS_FILL_MODE_UPPER:
                The upper triangular portion of the Hermitian matrix A is supplied.
                The matrix is compacted so that AP contains the triangular portion column-by-column
                so that:
                AP(0) = A(0,0)
                AP(1) = A(0,1)
                AP(2) = A(1,1), etc.
                    Ex: (HIPBLAS_FILL_MODE_UPPER; n = 3)
                        (1, 0) (2, 1) (3, 2)
                        (2,-1) (4, 0) (5,-1)    -----> [(1,0), (2,1), (4,0), (3,2), (5,-1), (6,0)]
                        (3,-2) (5, 1) (6, 0)
            if uplo == HIPBLAS_FILL_MODE_LOWER:
                The lower triangular portion of the Hermitian matrix A is supplied.
                The matrix is compacted so that AP contains the triangular portion column-by-column
                so that:
                AP(0) = A(0,0)
                AP(1) = A(1,0)
                AP(2) = A(2,1), etc.
                    Ex: (HIPBLAS_FILL_MODE_LOWER; n = 3)
                        (1, 0) (2, 1) (3, 2)
                        (2,-1) (4, 0) (5,-1)    -----> [(1,0), (2,-1), (3,-2), (4,0), (5,1), (6,0)]
                        (3,-2) (5, 1) (6, 0)
            Note that the imaginary part of the diagonal elements are not accessed and are assumed
            to be 0.
    @param[in]
    x         device pointer storing vector x.
    @param[in]
    incx      [int]
              specifies the increment for the elements of x.
    @param[in]
    beta      device pointer or host pointer to scalar beta.
    @param[inout]
    y         device pointer storing vector y.
    @param[in]
    incy      [int]
              specifies the increment for the elements of y.

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasChpmv(hipblasHandle_t       handle,
                                            hipblasFillMode_t     uplo,
                                            int                   n,
                                            const hipblasComplex* alpha,
                                            const hipblasComplex* AP,
                                            const hipblasComplex* x,
                                            int                   incx,
                                            const hipblasComplex* beta,
                                            hipblasComplex*       y,
                                            int                   incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhpmv(hipblasHandle_t             handle,
                                            hipblasFillMode_t           uplo,
                                            int                         n,
                                            const hipblasDoubleComplex* alpha,
                                            const hipblasDoubleComplex* AP,
                                            const hipblasDoubleComplex* x,
                                            int                         incx,
                                            const hipblasDoubleComplex* beta,
                                            hipblasDoubleComplex*       y,
                                            int                         incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasChpmv_v2(hipblasHandle_t   handle,
                                               hipblasFillMode_t uplo,
                                               int               n,
                                               const hipComplex* alpha,
                                               const hipComplex* AP,
                                               const hipComplex* x,
                                               int               incx,
                                               const hipComplex* beta,
                                               hipComplex*       y,
                                               int               incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhpmv_v2(hipblasHandle_t         handle,
                                               hipblasFillMode_t       uplo,
                                               int                     n,
                                               const hipDoubleComplex* alpha,
                                               const hipDoubleComplex* AP,
                                               const hipDoubleComplex* x,
                                               int                     incx,
                                               const hipDoubleComplex* beta,
                                               hipDoubleComplex*       y,
                                               int                     incy);

//64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasChpmv_64(hipblasHandle_t       handle,
                                               hipblasFillMode_t     uplo,
                                               int64_t               n,
                                               const hipblasComplex* alpha,
                                               const hipblasComplex* AP,
                                               const hipblasComplex* x,
                                               int64_t               incx,
                                               const hipblasComplex* beta,
                                               hipblasComplex*       y,
                                               int64_t               incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhpmv_64(hipblasHandle_t             handle,
                                               hipblasFillMode_t           uplo,
                                               int64_t                     n,
                                               const hipblasDoubleComplex* alpha,
                                               const hipblasDoubleComplex* AP,
                                               const hipblasDoubleComplex* x,
                                               int64_t                     incx,
                                               const hipblasDoubleComplex* beta,
                                               hipblasDoubleComplex*       y,
                                               int64_t                     incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasChpmv_v2_64(hipblasHandle_t   handle,
                                                  hipblasFillMode_t uplo,
                                                  int64_t           n,
                                                  const hipComplex* alpha,
                                                  const hipComplex* AP,
                                                  const hipComplex* x,
                                                  int64_t           incx,
                                                  const hipComplex* beta,
                                                  hipComplex*       y,
                                                  int64_t           incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhpmv_v2_64(hipblasHandle_t         handle,
                                                  hipblasFillMode_t       uplo,
                                                  int64_t                 n,
                                                  const hipDoubleComplex* alpha,
                                                  const hipDoubleComplex* AP,
                                                  const hipDoubleComplex* x,
                                                  int64_t                 incx,
                                                  const hipDoubleComplex* beta,
                                                  hipDoubleComplex*       y,
                                                  int64_t                 incy);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    hpmvBatched performs the matrix-vector operation

        y_i := alpha*A_i*x_i + beta*y_i

    where alpha and beta are scalars, x_i and y_i are n element vectors and A_i is an
    n by n Hermitian matrix, supplied in packed form (see description below),
    for each batch in i = [1, batchCount].

    - Supported precisions in rocBLAS : c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              HIPBLAS_FILL_MODE_UPPER: the upper triangular part of each Hermitian matrix A_i is supplied in AP.
              HIPBLAS_FILL_MODE_LOWER: the lower triangular part of each Hermitian matrix A_i is supplied in AP.
    @param[in]
    n         [int]
              the order of each matrix A_i.
    @param[in]
    alpha     device pointer or host pointer to scalar alpha.
    @param[in]
    AP      device pointer of device pointers storing the packed version of the specified triangular
            portion of each Hermitian matrix A_i. Each A_i is of at least size ((n * (n + 1)) / 2).
            if uplo == HIPBLAS_FILL_MODE_UPPER:
            The upper triangular portion of each Hermitian matrix A_i is supplied.
            The matrix is compacted so that each AP_i contains the triangular portion column-by-column
            so that:
            AP(0) = A(0,0)
            AP(1) = A(0,1)
            AP(2) = A(1,1), etc.
                Ex: (HIPBLAS_FILL_MODE_UPPER; n = 3)
                    (1, 0) (2, 1) (3, 2)
                    (2,-1) (4, 0) (5,-1)    -----> [(1,0), (2,1), (4,0), (3,2), (5,-1), (6,0)]
                    (3,-2) (5, 1) (6, 0)
        if uplo == HIPBLAS_FILL_MODE_LOWER:
            The lower triangular portion of each Hermitian matrix A_i is supplied.
            The matrix is compacted so that each AP_i contains the triangular portion column-by-column
            so that:
            AP(0) = A(0,0)
            AP(1) = A(1,0)
            AP(2) = A(2,1), etc.
                Ex: (HIPBLAS_FILL_MODE_LOWER; n = 3)
                    (1, 0) (2, 1) (3, 2)
                    (2,-1) (4, 0) (5,-1)    -----> [(1,0), (2,-1), (3,-2), (4,0), (5,1), (6,0)]
                    (3,-2) (5, 1) (6, 0)
        Note that the imaginary part of the diagonal elements are not accessed and are assumed
        to be 0.
    @param[in]
    x         device array of device pointers storing each vector x_i.
    @param[in]
    incx      [int]
              specifies the increment for the elements of each x_i.
    @param[in]
    beta      device pointer or host pointer to scalar beta.
    @param[inout]
    y         device array of device pointers storing each vector y_i.
    @param[in]
    incy      [int]
              specifies the increment for the elements of y.
    @param[in]
    batchCount [int]
                number of instances in the batch.

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasChpmvBatched(hipblasHandle_t             handle,
                                                   hipblasFillMode_t           uplo,
                                                   int                         n,
                                                   const hipblasComplex*       alpha,
                                                   const hipblasComplex* const AP[],
                                                   const hipblasComplex* const x[],
                                                   int                         incx,
                                                   const hipblasComplex*       beta,
                                                   hipblasComplex* const       y[],
                                                   int                         incy,
                                                   int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhpmvBatched(hipblasHandle_t                   handle,
                                                   hipblasFillMode_t                 uplo,
                                                   int                               n,
                                                   const hipblasDoubleComplex*       alpha,
                                                   const hipblasDoubleComplex* const AP[],
                                                   const hipblasDoubleComplex* const x[],
                                                   int                               incx,
                                                   const hipblasDoubleComplex*       beta,
                                                   hipblasDoubleComplex* const       y[],
                                                   int                               incy,
                                                   int                               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasChpmvBatched_v2(hipblasHandle_t         handle,
                                                      hipblasFillMode_t       uplo,
                                                      int                     n,
                                                      const hipComplex*       alpha,
                                                      const hipComplex* const AP[],
                                                      const hipComplex* const x[],
                                                      int                     incx,
                                                      const hipComplex*       beta,
                                                      hipComplex* const       y[],
                                                      int                     incy,
                                                      int                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhpmvBatched_v2(hipblasHandle_t               handle,
                                                      hipblasFillMode_t             uplo,
                                                      int                           n,
                                                      const hipDoubleComplex*       alpha,
                                                      const hipDoubleComplex* const AP[],
                                                      const hipDoubleComplex* const x[],
                                                      int                           incx,
                                                      const hipDoubleComplex*       beta,
                                                      hipDoubleComplex* const       y[],
                                                      int                           incy,
                                                      int                           batchCount);

//64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasChpmvBatched_64(hipblasHandle_t             handle,
                                                      hipblasFillMode_t           uplo,
                                                      int64_t                     n,
                                                      const hipblasComplex*       alpha,
                                                      const hipblasComplex* const AP[],
                                                      const hipblasComplex* const x[],
                                                      int64_t                     incx,
                                                      const hipblasComplex*       beta,
                                                      hipblasComplex* const       y[],
                                                      int64_t                     incy,
                                                      int64_t                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhpmvBatched_64(hipblasHandle_t                   handle,
                                                      hipblasFillMode_t                 uplo,
                                                      int64_t                           n,
                                                      const hipblasDoubleComplex*       alpha,
                                                      const hipblasDoubleComplex* const AP[],
                                                      const hipblasDoubleComplex* const x[],
                                                      int64_t                           incx,
                                                      const hipblasDoubleComplex*       beta,
                                                      hipblasDoubleComplex* const       y[],
                                                      int64_t                           incy,
                                                      int64_t                           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasChpmvBatched_v2_64(hipblasHandle_t         handle,
                                                         hipblasFillMode_t       uplo,
                                                         int64_t                 n,
                                                         const hipComplex*       alpha,
                                                         const hipComplex* const AP[],
                                                         const hipComplex* const x[],
                                                         int64_t                 incx,
                                                         const hipComplex*       beta,
                                                         hipComplex* const       y[],
                                                         int64_t                 incy,
                                                         int64_t                 batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhpmvBatched_v2_64(hipblasHandle_t               handle,
                                                         hipblasFillMode_t             uplo,
                                                         int64_t                       n,
                                                         const hipDoubleComplex*       alpha,
                                                         const hipDoubleComplex* const AP[],
                                                         const hipDoubleComplex* const x[],
                                                         int64_t                       incx,
                                                         const hipDoubleComplex*       beta,
                                                         hipDoubleComplex* const       y[],
                                                         int64_t                       incy,
                                                         int64_t                       batchCount);

//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    hpmvStridedBatched performs the matrix-vector operation

        y_i := alpha*A_i*x_i + beta*y_i

    where alpha and beta are scalars, x_i and y_i are n element vectors and A_i is an
    n by n Hermitian matrix, supplied in packed form (see description below),
    for each batch in i = [1, batchCount].

    - Supported precisions in rocBLAS : c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              HIPBLAS_FILL_MODE_UPPER: the upper triangular part of each Hermitian matrix A_i is supplied in AP.
              HIPBLAS_FILL_MODE_LOWER: the lower triangular part of each Hermitian matrix A_i is supplied in AP.
    @param[in]
    n         [int]
              the order of each matrix A_i.
    @param[in]
    alpha     device pointer or host pointer to scalar alpha.
    @param[in]
    AP        device pointer pointing to the beginning of the first matrix (AP_1). Stores the packed
              version of the specified triangular portion of each Hermitian matrix AP_i of size ((n * (n + 1)) / 2).
              if uplo == HIPBLAS_FILL_MODE_UPPER:
                The upper triangular portion of each Hermitian matrix A_i is supplied.
                The matrix is compacted so that each AP_i contains the triangular portion column-by-column
                so that:
                AP(0) = A(0,0)
                AP(1) = A(0,1)
                AP(2) = A(1,1), etc.
                    Ex: (HIPBLAS_FILL_MODE_UPPER; n = 3)
                        (1, 0) (2, 1) (3, 2)
                        (2,-1) (4, 0) (5,-1)    -----> [(1,0), (2,1), (4,0), (3,2), (5,-1), (6,0)]
                        (3,-2) (5, 1) (6, 0)
            if uplo == HIPBLAS_FILL_MODE_LOWER:
                The lower triangular portion of each Hermitian matrix A_i is supplied.
                The matrix is compacted so that each AP_i contains the triangular portion column-by-column
                so that:
                AP(0) = A(0,0)
                AP(1) = A(1,0)
                AP(2) = A(2,1), etc.
                    Ex: (HIPBLAS_FILL_MODE_LOWER; n = 3)
                        (1, 0) (2, 1) (3, 2)
                        (2,-1) (4, 0) (5,-1)    -----> [(1,0), (2,-1), (3,-2), (4,0), (5,1), (6,0)]
                        (3,-2) (5, 1) (6, 0)
        Note that the imaginary part of the diagonal elements are not accessed and are assumed
        to be 0.
    @param[in]
    strideA  [hipblasStride]
              stride from the start of one matrix (AP_i) and the next one (AP_i+1).
    @param[in]
    x         device array pointing to the beginning of the first vector (x_1).
    @param[in]
    incx      [int]
              specifies the increment for the elements of each x_i.
    @param[in]
    stridex  [hipblasStride]
              stride from the start of one vector (x_i) and the next one (x_i+1).
    @param[in]
    beta      device pointer or host pointer to scalar beta.
    @param[inout]
    y         device array pointing to the beginning of the first vector (y_1).
    @param[in]
    incy      [int]
              specifies the increment for the elements of y.
    @param[in]
    stridey  [hipblasStride]
              stride from the start of one vector (y_i) and the next one (y_i+1).
    @param[in]
    batchCount [int]
                number of instances in the batch.

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasChpmvStridedBatched(hipblasHandle_t       handle,
                                                          hipblasFillMode_t     uplo,
                                                          int                   n,
                                                          const hipblasComplex* alpha,
                                                          const hipblasComplex* AP,
                                                          hipblasStride         strideA,
                                                          const hipblasComplex* x,
                                                          int                   incx,
                                                          hipblasStride         stridex,
                                                          const hipblasComplex* beta,
                                                          hipblasComplex*       y,
                                                          int                   incy,
                                                          hipblasStride         stridey,
                                                          int                   batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhpmvStridedBatched(hipblasHandle_t             handle,
                                                          hipblasFillMode_t           uplo,
                                                          int                         n,
                                                          const hipblasDoubleComplex* alpha,
                                                          const hipblasDoubleComplex* AP,
                                                          hipblasStride               strideA,
                                                          const hipblasDoubleComplex* x,
                                                          int                         incx,
                                                          hipblasStride               stridex,
                                                          const hipblasDoubleComplex* beta,
                                                          hipblasDoubleComplex*       y,
                                                          int                         incy,
                                                          hipblasStride               stridey,
                                                          int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasChpmvStridedBatched_v2(hipblasHandle_t   handle,
                                                             hipblasFillMode_t uplo,
                                                             int               n,
                                                             const hipComplex* alpha,
                                                             const hipComplex* AP,
                                                             hipblasStride     strideA,
                                                             const hipComplex* x,
                                                             int               incx,
                                                             hipblasStride     stridex,
                                                             const hipComplex* beta,
                                                             hipComplex*       y,
                                                             int               incy,
                                                             hipblasStride     stridey,
                                                             int               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhpmvStridedBatched_v2(hipblasHandle_t         handle,
                                                             hipblasFillMode_t       uplo,
                                                             int                     n,
                                                             const hipDoubleComplex* alpha,
                                                             const hipDoubleComplex* AP,
                                                             hipblasStride           strideA,
                                                             const hipDoubleComplex* x,
                                                             int                     incx,
                                                             hipblasStride           stridex,
                                                             const hipDoubleComplex* beta,
                                                             hipDoubleComplex*       y,
                                                             int                     incy,
                                                             hipblasStride           stridey,
                                                             int                     batchCount);

//64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasChpmvStridedBatched_64(hipblasHandle_t       handle,
                                                             hipblasFillMode_t     uplo,
                                                             int64_t               n,
                                                             const hipblasComplex* alpha,
                                                             const hipblasComplex* AP,
                                                             hipblasStride         strideA,
                                                             const hipblasComplex* x,
                                                             int64_t               incx,
                                                             hipblasStride         stridex,
                                                             const hipblasComplex* beta,
                                                             hipblasComplex*       y,
                                                             int64_t               incy,
                                                             hipblasStride         stridey,
                                                             int64_t               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhpmvStridedBatched_64(hipblasHandle_t             handle,
                                                             hipblasFillMode_t           uplo,
                                                             int64_t                     n,
                                                             const hipblasDoubleComplex* alpha,
                                                             const hipblasDoubleComplex* AP,
                                                             hipblasStride               strideA,
                                                             const hipblasDoubleComplex* x,
                                                             int64_t                     incx,
                                                             hipblasStride               stridex,
                                                             const hipblasDoubleComplex* beta,
                                                             hipblasDoubleComplex*       y,
                                                             int64_t                     incy,
                                                             hipblasStride               stridey,
                                                             int64_t batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasChpmvStridedBatched_v2_64(hipblasHandle_t   handle,
                                                                hipblasFillMode_t uplo,
                                                                int64_t           n,
                                                                const hipComplex* alpha,
                                                                const hipComplex* AP,
                                                                hipblasStride     strideA,
                                                                const hipComplex* x,
                                                                int64_t           incx,
                                                                hipblasStride     stridex,
                                                                const hipComplex* beta,
                                                                hipComplex*       y,
                                                                int64_t           incy,
                                                                hipblasStride     stridey,
                                                                int64_t           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhpmvStridedBatched_v2_64(hipblasHandle_t         handle,
                                                                hipblasFillMode_t       uplo,
                                                                int64_t                 n,
                                                                const hipDoubleComplex* alpha,
                                                                const hipDoubleComplex* AP,
                                                                hipblasStride           strideA,
                                                                const hipDoubleComplex* x,
                                                                int64_t                 incx,
                                                                hipblasStride           stridex,
                                                                const hipDoubleComplex* beta,
                                                                hipDoubleComplex*       y,
                                                                int64_t                 incy,
                                                                hipblasStride           stridey,
                                                                int64_t                 batchCount);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    hpr performs the matrix-vector operations

        A := A + alpha*x*x**H

    where alpha is a real scalar, x is a vector, and A is an
    n by n Hermitian matrix, supplied in packed form.

    - Supported precisions in rocBLAS : c,z
    - Supported precisions in cuBLAS  : c,z

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
              HIPBLAS_FILL_MODE_UPPER: The upper triangular part of A is supplied in AP.
              HIPBLAS_FILL_MODE_LOWER: The lower triangular part of A is supplied in AP.
    @param[in]
    n         [int]
              the number of rows and columns of matrix A, must be at least 0.
    @param[in]
    alpha
              device pointer or host pointer to scalar alpha.
    @param[in]
    x         device pointer storing vector x.
    @param[in]
    incx      [int]
              specifies the increment for the elements of x.
    @param[inout]
    AP        device pointer storing the packed version of the specified triangular portion of
              the Hermitian matrix A. Of at least size ((n * (n + 1)) / 2).
              if uplo == HIPBLAS_FILL_MODE_UPPER:
                The upper triangular portion of the Hermitian matrix A is supplied.
                The matrix is compacted so that AP contains the triangular portion column-by-column
                so that:
                AP(0) = A(0,0)
                AP(1) = A(0,1)
                AP(2) = A(1,1), etc.
                    Ex: (HIPBLAS_FILL_MODE_UPPER; n = 3)
                        (1, 0) (2, 1) (4,9)
                        (2,-1) (3, 0) (5,3)  -----> [(1,0), (2,1), (3,0), (4,9), (5,3), (6,0)]
                        (4,-9) (5,-3) (6,0)
            if uplo == HIPBLAS_FILL_MODE_LOWER:
                The lower triangular portion of the Hermitian matrix A is supplied.
                The matrix is compacted so that AP contains the triangular portion column-by-column
                so that:
                AP(0) = A(0,0)
                AP(1) = A(1,0)
                AP(2) = A(2,1), etc.
                    Ex: (HIPBLAS_FILL_MODE_LOWER; n = 3)
                        (1, 0) (2, 1) (4,9)
                        (2,-1) (3, 0) (5,3)  -----> [(1,0), (2,-1), (4,-9), (3,0), (5,-3), (6,0)]
                        (4,-9) (5,-3) (6,0)
            Note that the imaginary part of the diagonal elements are not accessed and are assumed
            to be 0.
    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasChpr(hipblasHandle_t       handle,
                                           hipblasFillMode_t     uplo,
                                           int                   n,
                                           const float*          alpha,
                                           const hipblasComplex* x,
                                           int                   incx,
                                           hipblasComplex*       AP);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhpr(hipblasHandle_t             handle,
                                           hipblasFillMode_t           uplo,
                                           int                         n,
                                           const double*               alpha,
                                           const hipblasDoubleComplex* x,
                                           int                         incx,
                                           hipblasDoubleComplex*       AP);

HIPBLAS_EXPORT hipblasStatus_t hipblasChpr_v2(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int               n,
                                              const float*      alpha,
                                              const hipComplex* x,
                                              int               incx,
                                              hipComplex*       AP);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhpr_v2(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              int                     n,
                                              const double*           alpha,
                                              const hipDoubleComplex* x,
                                              int                     incx,
                                              hipDoubleComplex*       AP);

//64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasChpr_64(hipblasHandle_t       handle,
                                              hipblasFillMode_t     uplo,
                                              int64_t               n,
                                              const float*          alpha,
                                              const hipblasComplex* x,
                                              int64_t               incx,
                                              hipblasComplex*       AP);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhpr_64(hipblasHandle_t             handle,
                                              hipblasFillMode_t           uplo,
                                              int64_t                     n,
                                              const double*               alpha,
                                              const hipblasDoubleComplex* x,
                                              int64_t                     incx,
                                              hipblasDoubleComplex*       AP);

HIPBLAS_EXPORT hipblasStatus_t hipblasChpr_v2_64(hipblasHandle_t   handle,
                                                 hipblasFillMode_t uplo,
                                                 int64_t           n,
                                                 const float*      alpha,
                                                 const hipComplex* x,
                                                 int64_t           incx,
                                                 hipComplex*       AP);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhpr_v2_64(hipblasHandle_t         handle,
                                                 hipblasFillMode_t       uplo,
                                                 int64_t                 n,
                                                 const double*           alpha,
                                                 const hipDoubleComplex* x,
                                                 int64_t                 incx,
                                                 hipDoubleComplex*       AP);

//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    hprBatched performs the matrix-vector operations

        A_i := A_i + alpha*x_i*x_i**H

    where alpha is a real scalar, x_i is a vector, and A_i is an
    n by n symmetric matrix, supplied in packed form, for i = 1, ..., batchCount.

    - Supported precisions in rocBLAS : c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
              HIPBLAS_FILL_MODE_UPPER: The upper triangular part of each A_i is supplied in AP.
              HIPBLAS_FILL_MODE_LOWER: The lower triangular part of each A_i is supplied in AP.
    @param[in]
    n         [int]
              the number of rows and columns of each matrix A_i, must be at least 0.
    @param[in]
    alpha
              device pointer or host pointer to scalar alpha.
    @param[in]
    x         device array of device pointers storing each vector x_i.
    @param[in]
    incx      [int]
              specifies the increment for the elements of each x_i.
    @param[inout]
    AP        device array of device pointers storing the packed version of the specified triangular portion of
              each Hermitian matrix A_i of at least size ((n * (n + 1)) / 2). Array is of at least size batchCount.
              if uplo == HIPBLAS_FILL_MODE_UPPER:
                The upper triangular portion of each Hermitian matrix A_i is supplied.
                The matrix is compacted so that AP contains the triangular portion column-by-column
                so that:
                AP(0) = A(0,0)
                AP(1) = A(0,1)
                AP(2) = A(1,1), etc.
                    Ex: (HIPBLAS_FILL_MODE_UPPER; n = 3)
                        (1, 0) (2, 1) (4,9)
                        (2,-1) (3, 0) (5,3)  -----> [(1,0), (2,1), (3,0), (4,9), (5,3), (6,0)]
                        (4,-9) (5,-3) (6,0)
            if uplo == HIPBLAS_FILL_MODE_LOWER:
                The lower triangular portion of each Hermitian matrix A_i is supplied.
                The matrix is compacted so that AP contains the triangular portion column-by-column
                so that:
                AP(0) = A(0,0)
                AP(1) = A(1,0)
                AP(2) = A(2,1), etc.
                    Ex: (HIPBLAS_FILL_MODE_LOWER; n = 3)
                        (1, 0) (2, 1) (4,9)
                        (2,-1) (3, 0) (5,3)  -----> [(1,0), (2,-1), (4,-9), (3,0), (5,-3), (6,0)]
                        (4,-9) (5,-3) (6,0)
            Note that the imaginary part of the diagonal elements are not accessed and are assumed
            to be 0.
    @param[in]
    batchCount [int]
                number of instances in the batch.
    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasChprBatched(hipblasHandle_t             handle,
                                                  hipblasFillMode_t           uplo,
                                                  int                         n,
                                                  const float*                alpha,
                                                  const hipblasComplex* const x[],
                                                  int                         incx,
                                                  hipblasComplex* const       AP[],
                                                  int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhprBatched(hipblasHandle_t                   handle,
                                                  hipblasFillMode_t                 uplo,
                                                  int                               n,
                                                  const double*                     alpha,
                                                  const hipblasDoubleComplex* const x[],
                                                  int                               incx,
                                                  hipblasDoubleComplex* const       AP[],
                                                  int                               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasChprBatched_v2(hipblasHandle_t         handle,
                                                     hipblasFillMode_t       uplo,
                                                     int                     n,
                                                     const float*            alpha,
                                                     const hipComplex* const x[],
                                                     int                     incx,
                                                     hipComplex* const       AP[],
                                                     int                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhprBatched_v2(hipblasHandle_t               handle,
                                                     hipblasFillMode_t             uplo,
                                                     int                           n,
                                                     const double*                 alpha,
                                                     const hipDoubleComplex* const x[],
                                                     int                           incx,
                                                     hipDoubleComplex* const       AP[],
                                                     int                           batchCount);

//64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasChprBatched_64(hipblasHandle_t             handle,
                                                     hipblasFillMode_t           uplo,
                                                     int64_t                     n,
                                                     const float*                alpha,
                                                     const hipblasComplex* const x[],
                                                     int64_t                     incx,
                                                     hipblasComplex* const       AP[],
                                                     int64_t                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhprBatched_64(hipblasHandle_t                   handle,
                                                     hipblasFillMode_t                 uplo,
                                                     int64_t                           n,
                                                     const double*                     alpha,
                                                     const hipblasDoubleComplex* const x[],
                                                     int64_t                           incx,
                                                     hipblasDoubleComplex* const       AP[],
                                                     int64_t                           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasChprBatched_v2_64(hipblasHandle_t         handle,
                                                        hipblasFillMode_t       uplo,
                                                        int64_t                 n,
                                                        const float*            alpha,
                                                        const hipComplex* const x[],
                                                        int64_t                 incx,
                                                        hipComplex* const       AP[],
                                                        int64_t                 batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhprBatched_v2_64(hipblasHandle_t               handle,
                                                        hipblasFillMode_t             uplo,
                                                        int64_t                       n,
                                                        const double*                 alpha,
                                                        const hipDoubleComplex* const x[],
                                                        int64_t                       incx,
                                                        hipDoubleComplex* const       AP[],
                                                        int64_t                       batchCount);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    hprStridedBatched performs the matrix-vector operations

        A_i := A_i + alpha*x_i*x_i**H

    where alpha is a real scalar, x_i is a vector, and A_i is an
    n by n symmetric matrix, supplied in packed form, for i = 1, ..., batchCount.

    - Supported precisions in rocBLAS : c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
              HIPBLAS_FILL_MODE_UPPER: The upper triangular part of each A_i is supplied in AP.
              HIPBLAS_FILL_MODE_LOWER: The lower triangular part of each A_i is supplied in AP.
    @param[in]
    n         [int]
              the number of rows and columns of each matrix A_i, must be at least 0.
    @param[in]
    alpha
              device pointer or host pointer to scalar alpha.
    @param[in]
    x         device pointer pointing to the first vector (x_1).
    @param[in]
    incx      [int]
              specifies the increment for the elements of each x_i.
    @param[in]
    stridex  [hipblasStride]
              stride from the start of one vector (x_i) and the next one (x_i+1).
    @param[inout]
    AP        device array of device pointers storing the packed version of the specified triangular portion of
              each Hermitian matrix A_i. Points to the first matrix (A_1).
              if uplo == HIPBLAS_FILL_MODE_UPPER:
                The upper triangular portion of each Hermitian matrix A_i is supplied.
                The matrix is compacted so that AP contains the triangular portion column-by-column
                so that:
                AP(0) = A(0,0)
                AP(1) = A(0,1)
                AP(2) = A(1,1), etc.
                    Ex: (HIPBLAS_FILL_MODE_UPPER; n = 3)
                        (1, 0) (2, 1) (4,9)
                        (2,-1) (3, 0) (5,3)  -----> [(1,0), (2,1), (3,0), (4,9), (5,3), (6,0)]
                        (4,-9) (5,-3) (6,0)
            if uplo == HIPBLAS_FILL_MODE_LOWER:
                The lower triangular portion of each Hermitian matrix A_i is supplied.
                The matrix is compacted so that AP contains the triangular portion column-by-column
                so that:
                AP(0) = A(0,0)
                AP(1) = A(1,0)
                AP(2) = A(2,1), etc.
                    Ex: (HIPBLAS_FILL_MODE_LOWER; n = 3)
                        (1, 0) (2, 1) (4,9)
                        (2,-1) (3, 0) (5,3)  -----> [(1,0), (2,-1), (4,-9), (3,0), (5,-3), (6,0)]
                        (4,-9) (5,-3) (6,0)
            Note that the imaginary part of the diagonal elements are not accessed and are assumed
            to be 0.
    @param[in]
    strideA   [hipblasStride]
                stride from the start of one (A_i) and the next (A_i+1)
    @param[in]
    batchCount [int]
                number of instances in the batch.
    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasChprStridedBatched(hipblasHandle_t       handle,
                                                         hipblasFillMode_t     uplo,
                                                         int                   n,
                                                         const float*          alpha,
                                                         const hipblasComplex* x,
                                                         int                   incx,
                                                         hipblasStride         stridex,
                                                         hipblasComplex*       AP,
                                                         hipblasStride         strideA,
                                                         int                   batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhprStridedBatched(hipblasHandle_t             handle,
                                                         hipblasFillMode_t           uplo,
                                                         int                         n,
                                                         const double*               alpha,
                                                         const hipblasDoubleComplex* x,
                                                         int                         incx,
                                                         hipblasStride               stridex,
                                                         hipblasDoubleComplex*       AP,
                                                         hipblasStride               strideA,
                                                         int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasChprStridedBatched_v2(hipblasHandle_t   handle,
                                                            hipblasFillMode_t uplo,
                                                            int               n,
                                                            const float*      alpha,
                                                            const hipComplex* x,
                                                            int               incx,
                                                            hipblasStride     stridex,
                                                            hipComplex*       AP,
                                                            hipblasStride     strideA,
                                                            int               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhprStridedBatched_v2(hipblasHandle_t         handle,
                                                            hipblasFillMode_t       uplo,
                                                            int                     n,
                                                            const double*           alpha,
                                                            const hipDoubleComplex* x,
                                                            int                     incx,
                                                            hipblasStride           stridex,
                                                            hipDoubleComplex*       AP,
                                                            hipblasStride           strideA,
                                                            int                     batchCount);

//64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasChprStridedBatched_64(hipblasHandle_t       handle,
                                                            hipblasFillMode_t     uplo,
                                                            int64_t               n,
                                                            const float*          alpha,
                                                            const hipblasComplex* x,
                                                            int64_t               incx,
                                                            hipblasStride         stridex,
                                                            hipblasComplex*       AP,
                                                            hipblasStride         strideA,
                                                            int64_t               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhprStridedBatched_64(hipblasHandle_t             handle,
                                                            hipblasFillMode_t           uplo,
                                                            int64_t                     n,
                                                            const double*               alpha,
                                                            const hipblasDoubleComplex* x,
                                                            int64_t                     incx,
                                                            hipblasStride               stridex,
                                                            hipblasDoubleComplex*       AP,
                                                            hipblasStride               strideA,
                                                            int64_t                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasChprStridedBatched_v2_64(hipblasHandle_t   handle,
                                                               hipblasFillMode_t uplo,
                                                               int64_t           n,
                                                               const float*      alpha,
                                                               const hipComplex* x,
                                                               int64_t           incx,
                                                               hipblasStride     stridex,
                                                               hipComplex*       AP,
                                                               hipblasStride     strideA,
                                                               int64_t           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhprStridedBatched_v2_64(hipblasHandle_t         handle,
                                                               hipblasFillMode_t       uplo,
                                                               int64_t                 n,
                                                               const double*           alpha,
                                                               const hipDoubleComplex* x,
                                                               int64_t                 incx,
                                                               hipblasStride           stridex,
                                                               hipDoubleComplex*       AP,
                                                               hipblasStride           strideA,
                                                               int64_t                 batchCount);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    hpr2 performs the matrix-vector operations

        A := A + alpha*x*y**H + conj(alpha)*y*x**H

    where alpha is a complex scalar, x and y are vectors, and A is an
    n by n Hermitian matrix, supplied in packed form.

    - Supported precisions in rocBLAS : c,z
    - Supported precisions in cuBLAS  : c,z

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
              HIPBLAS_FILL_MODE_UPPER: The upper triangular part of A is supplied in AP.
              HIPBLAS_FILL_MODE_LOWER: The lower triangular part of A is supplied in AP.
    @param[in]
    n         [int]
              the number of rows and columns of matrix A, must be at least 0.
    @param[in]
    alpha
              device pointer or host pointer to scalar alpha.
    @param[in]
    x         device pointer storing vector x.
    @param[in]
    incx      [int]
              specifies the increment for the elements of x.
    @param[in]
    y         device pointer storing vector y.
    @param[in]
    incy      [int]
              specifies the increment for the elements of y.
    @param[inout]
    AP        device pointer storing the packed version of the specified triangular portion of
              the Hermitian matrix A. Of at least size ((n * (n + 1)) / 2).
              if uplo == HIPBLAS_FILL_MODE_UPPER:
                The upper triangular portion of the Hermitian matrix A is supplied.
                The matrix is compacted so that AP contains the triangular portion column-by-column
                so that:
                AP(0) = A(0,0)
                AP(1) = A(0,1)
                AP(2) = A(1,1), etc.
                    Ex: (HIPBLAS_FILL_MODE_UPPER; n = 3)
                        (1, 0) (2, 1) (4,9)
                        (2,-1) (3, 0) (5,3)  -----> [(1,0), (2,1), (3,0), (4,9), (5,3), (6,0)]
                        (4,-9) (5,-3) (6,0)
            if uplo == HIPBLAS_FILL_MODE_LOWER:
                The lower triangular portion of the Hermitian matrix A is supplied.
                The matrix is compacted so that AP contains the triangular portion column-by-column
                so that:
                AP(0) = A(0,0)
                AP(1) = A(1,0)
                AP(2) = A(2,1), etc.
                    Ex: (HIPBLAS_FILL_MODE_LOWER; n = 3)
                        (1, 0) (2, 1) (4,9)
                        (2,-1) (3, 0) (5,3)  -----> [(1,0), (2,-1), (4,-9), (3,0), (5,-3), (6,0)]
                        (4,-9) (5,-3) (6,0)
            Note that the imaginary part of the diagonal elements are not accessed and are assumed
            to be 0.
    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasChpr2(hipblasHandle_t       handle,
                                            hipblasFillMode_t     uplo,
                                            int                   n,
                                            const hipblasComplex* alpha,
                                            const hipblasComplex* x,
                                            int                   incx,
                                            const hipblasComplex* y,
                                            int                   incy,
                                            hipblasComplex*       AP);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhpr2(hipblasHandle_t             handle,
                                            hipblasFillMode_t           uplo,
                                            int                         n,
                                            const hipblasDoubleComplex* alpha,
                                            const hipblasDoubleComplex* x,
                                            int                         incx,
                                            const hipblasDoubleComplex* y,
                                            int                         incy,
                                            hipblasDoubleComplex*       AP);

HIPBLAS_EXPORT hipblasStatus_t hipblasChpr2_v2(hipblasHandle_t   handle,
                                               hipblasFillMode_t uplo,
                                               int               n,
                                               const hipComplex* alpha,
                                               const hipComplex* x,
                                               int               incx,
                                               const hipComplex* y,
                                               int               incy,
                                               hipComplex*       AP);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhpr2_v2(hipblasHandle_t         handle,
                                               hipblasFillMode_t       uplo,
                                               int                     n,
                                               const hipDoubleComplex* alpha,
                                               const hipDoubleComplex* x,
                                               int                     incx,
                                               const hipDoubleComplex* y,
                                               int                     incy,
                                               hipDoubleComplex*       AP);

//64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasChpr2_64(hipblasHandle_t       handle,
                                               hipblasFillMode_t     uplo,
                                               int64_t               n,
                                               const hipblasComplex* alpha,
                                               const hipblasComplex* x,
                                               int64_t               incx,
                                               const hipblasComplex* y,
                                               int64_t               incy,
                                               hipblasComplex*       AP);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhpr2_64(hipblasHandle_t             handle,
                                               hipblasFillMode_t           uplo,
                                               int64_t                     n,
                                               const hipblasDoubleComplex* alpha,
                                               const hipblasDoubleComplex* x,
                                               int64_t                     incx,
                                               const hipblasDoubleComplex* y,
                                               int64_t                     incy,
                                               hipblasDoubleComplex*       AP);

HIPBLAS_EXPORT hipblasStatus_t hipblasChpr2_v2_64(hipblasHandle_t   handle,
                                                  hipblasFillMode_t uplo,
                                                  int64_t           n,
                                                  const hipComplex* alpha,
                                                  const hipComplex* x,
                                                  int64_t           incx,
                                                  const hipComplex* y,
                                                  int64_t           incy,
                                                  hipComplex*       AP);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhpr2_v2_64(hipblasHandle_t         handle,
                                                  hipblasFillMode_t       uplo,
                                                  int64_t                 n,
                                                  const hipDoubleComplex* alpha,
                                                  const hipDoubleComplex* x,
                                                  int64_t                 incx,
                                                  const hipDoubleComplex* y,
                                                  int64_t                 incy,
                                                  hipDoubleComplex*       AP);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    hpr2Batched performs the matrix-vector operations

        A_i := A_i + alpha*x_i*y_i**H + conj(alpha)*y_i*x_i**H

    where alpha is a complex scalar, x_i and y_i are vectors, and A_i is an
    n by n symmetric matrix, supplied in packed form, for i = 1, ..., batchCount.

    - Supported precisions in rocBLAS : c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
              HIPBLAS_FILL_MODE_UPPER: The upper triangular part of each A_i is supplied in AP.
              HIPBLAS_FILL_MODE_LOWER: The lower triangular part of each A_i is supplied in AP.
    @param[in]
    n         [int]
              the number of rows and columns of each matrix A_i, must be at least 0.
    @param[in]
    alpha
              device pointer or host pointer to scalar alpha.
    @param[in]
    x         device array of device pointers storing each vector x_i.
    @param[in]
    incx      [int]
              specifies the increment for the elements of each x_i.
    @param[in]
    y         device array of device pointers storing each vector y_i.
    @param[in]
    incy      [int]
              specifies the increment for the elements of each y_i.
    @param[inout]
    AP        device array of device pointers storing the packed version of the specified triangular portion of
              each Hermitian matrix A_i of at least size ((n * (n + 1)) / 2). Array is of at least size batchCount.
              if uplo == HIPBLAS_FILL_MODE_UPPER:
                The upper triangular portion of each Hermitian matrix A_i is supplied.
                The matrix is compacted so that AP contains the triangular portion column-by-column
                so that:
                AP(0) = A(0,0)
                AP(1) = A(0,1)
                AP(2) = A(1,1), etc.
                    Ex: (HIPBLAS_FILL_MODE_UPPER; n = 3)
                        (1, 0) (2, 1) (4,9)
                        (2,-1) (3, 0) (5,3)  -----> [(1,0), (2,1), (3,0), (4,9), (5,3), (6,0)]
                        (4,-9) (5,-3) (6,0)
            if uplo == HIPBLAS_FILL_MODE_LOWER:
                The lower triangular portion of each Hermitian matrix A_i is supplied.
                The matrix is compacted so that AP contains the triangular portion column-by-column
                so that:
                AP(0) = A(0,0)
                AP(1) = A(1,0)
                AP(2) = A(2,1), etc.
                    Ex: (HIPBLAS_FILL_MODE_LOWER; n = 3)
                        (1, 0) (2, 1) (4,9)
                        (2,-1) (3, 0) (5,3)  -----> [(1,0), (2,-1), (4,-9), (3,0), (5,-3), (6,0)]
                        (4,-9) (5,-3) (6,0)
            Note that the imaginary part of the diagonal elements are not accessed and are assumed
            to be 0.
    @param[in]
    batchCount [int]
                number of instances in the batch.
    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasChpr2Batched(hipblasHandle_t             handle,
                                                   hipblasFillMode_t           uplo,
                                                   int                         n,
                                                   const hipblasComplex*       alpha,
                                                   const hipblasComplex* const x[],
                                                   int                         incx,
                                                   const hipblasComplex* const y[],
                                                   int                         incy,
                                                   hipblasComplex* const       AP[],
                                                   int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhpr2Batched(hipblasHandle_t                   handle,
                                                   hipblasFillMode_t                 uplo,
                                                   int                               n,
                                                   const hipblasDoubleComplex*       alpha,
                                                   const hipblasDoubleComplex* const x[],
                                                   int                               incx,
                                                   const hipblasDoubleComplex* const y[],
                                                   int                               incy,
                                                   hipblasDoubleComplex* const       AP[],
                                                   int                               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasChpr2Batched_v2(hipblasHandle_t         handle,
                                                      hipblasFillMode_t       uplo,
                                                      int                     n,
                                                      const hipComplex*       alpha,
                                                      const hipComplex* const x[],
                                                      int                     incx,
                                                      const hipComplex* const y[],
                                                      int                     incy,
                                                      hipComplex* const       AP[],
                                                      int                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhpr2Batched_v2(hipblasHandle_t               handle,
                                                      hipblasFillMode_t             uplo,
                                                      int                           n,
                                                      const hipDoubleComplex*       alpha,
                                                      const hipDoubleComplex* const x[],
                                                      int                           incx,
                                                      const hipDoubleComplex* const y[],
                                                      int                           incy,
                                                      hipDoubleComplex* const       AP[],
                                                      int                           batchCount);

//64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasChpr2Batched_64(hipblasHandle_t             handle,
                                                      hipblasFillMode_t           uplo,
                                                      int64_t                     n,
                                                      const hipblasComplex*       alpha,
                                                      const hipblasComplex* const x[],
                                                      int64_t                     incx,
                                                      const hipblasComplex* const y[],
                                                      int64_t                     incy,
                                                      hipblasComplex* const       AP[],
                                                      int64_t                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhpr2Batched_64(hipblasHandle_t                   handle,
                                                      hipblasFillMode_t                 uplo,
                                                      int64_t                           n,
                                                      const hipblasDoubleComplex*       alpha,
                                                      const hipblasDoubleComplex* const x[],
                                                      int64_t                           incx,
                                                      const hipblasDoubleComplex* const y[],
                                                      int64_t                           incy,
                                                      hipblasDoubleComplex* const       AP[],
                                                      int64_t                           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasChpr2Batched_v2_64(hipblasHandle_t         handle,
                                                         hipblasFillMode_t       uplo,
                                                         int64_t                 n,
                                                         const hipComplex*       alpha,
                                                         const hipComplex* const x[],
                                                         int64_t                 incx,
                                                         const hipComplex* const y[],
                                                         int64_t                 incy,
                                                         hipComplex* const       AP[],
                                                         int64_t                 batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhpr2Batched_v2_64(hipblasHandle_t               handle,
                                                         hipblasFillMode_t             uplo,
                                                         int64_t                       n,
                                                         const hipDoubleComplex*       alpha,
                                                         const hipDoubleComplex* const x[],
                                                         int64_t                       incx,
                                                         const hipDoubleComplex* const y[],
                                                         int64_t                       incy,
                                                         hipDoubleComplex* const       AP[],
                                                         int64_t                       batchCount);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    hpr2StridedBatched performs the matrix-vector operations

        A_i := A_i + alpha*x_i*y_i**H + conj(alpha)*y_i*x_i**H

    where alpha is a complex scalar, x_i and y_i are vectors, and A_i is an
    n by n symmetric matrix, supplied in packed form, for i = 1, ..., batchCount.

    - Supported precisions in rocBLAS : c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
              HIPBLAS_FILL_MODE_UPPER: The upper triangular part of each A_i is supplied in AP.
              HIPBLAS_FILL_MODE_LOWER: The lower triangular part of each A_i is supplied in AP.
    @param[in]
    n         [int]
              the number of rows and columns of each matrix A_i, must be at least 0.
    @param[in]
    alpha
              device pointer or host pointer to scalar alpha.
    @param[in]
    x         device pointer pointing to the first vector (x_1).
    @param[in]
    incx      [int]
              specifies the increment for the elements of each x_i.
    @param[in]
    stridex  [hipblasStride]
              stride from the start of one vector (x_i) and the next one (x_i+1).
    @param[in]
    y         device pointer pointing to the first vector (y_1).
    @param[in]
    incy      [int]
              specifies the increment for the elements of each y_i.
    @param[in]
    stridey  [hipblasStride]
              stride from the start of one vector (y_i) and the next one (y_i+1).
    @param[inout]
    AP        device array of device pointers storing the packed version of the specified triangular portion of
              each Hermitian matrix A_i. Points to the first matrix (A_1).
              if uplo == HIPBLAS_FILL_MODE_UPPER:
                The upper triangular portion of each Hermitian matrix A_i is supplied.
                The matrix is compacted so that AP contains the triangular portion column-by-column
                so that:
                AP(0) = A(0,0)
                AP(1) = A(0,1)
                AP(2) = A(1,1), etc.
                    Ex: (HIPBLAS_FILL_MODE_UPPER; n = 3)
                        (1, 0) (2, 1) (4,9)
                        (2,-1) (3, 0) (5,3)  -----> [(1,0), (2,1), (3,0), (4,9), (5,3), (6,0)]
                        (4,-9) (5,-3) (6,0)
            if uplo == HIPBLAS_FILL_MODE_LOWER:
                The lower triangular portion of each Hermitian matrix A_i is supplied.
                The matrix is compacted so that AP contains the triangular portion column-by-column
                so that:
                AP(0) = A(0,0)
                AP(1) = A(1,0)
                AP(2) = A(2,1), etc.
                    Ex: (HIPBLAS_FILL_MODE_LOWER; n = 3)
                        (1, 0) (2, 1) (4,9)
                        (2,-1) (3, 0) (5,3)  -----> [(1,0), (2,-1), (4,-9), (3,0), (5,-3), (6,0)]
                        (4,-9) (5,-3) (6,0)
            Note that the imaginary part of the diagonal elements are not accessed and are assumed
            to be 0.
    @param[in]
    strideA    [hipblasStride]
                stride from the start of one (A_i) and the next (A_i+1)
    @param[in]
    batchCount [int]
                number of instances in the batch.
    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasChpr2StridedBatched(hipblasHandle_t       handle,
                                                          hipblasFillMode_t     uplo,
                                                          int                   n,
                                                          const hipblasComplex* alpha,
                                                          const hipblasComplex* x,
                                                          int                   incx,
                                                          hipblasStride         stridex,
                                                          const hipblasComplex* y,
                                                          int                   incy,
                                                          hipblasStride         stridey,
                                                          hipblasComplex*       AP,
                                                          hipblasStride         strideA,
                                                          int                   batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhpr2StridedBatched(hipblasHandle_t             handle,
                                                          hipblasFillMode_t           uplo,
                                                          int                         n,
                                                          const hipblasDoubleComplex* alpha,
                                                          const hipblasDoubleComplex* x,
                                                          int                         incx,
                                                          hipblasStride               stridex,
                                                          const hipblasDoubleComplex* y,
                                                          int                         incy,
                                                          hipblasStride               stridey,
                                                          hipblasDoubleComplex*       AP,
                                                          hipblasStride               strideA,
                                                          int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasChpr2StridedBatched_v2(hipblasHandle_t   handle,
                                                             hipblasFillMode_t uplo,
                                                             int               n,
                                                             const hipComplex* alpha,
                                                             const hipComplex* x,
                                                             int               incx,
                                                             hipblasStride     stridex,
                                                             const hipComplex* y,
                                                             int               incy,
                                                             hipblasStride     stridey,
                                                             hipComplex*       AP,
                                                             hipblasStride     strideA,
                                                             int               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhpr2StridedBatched_v2(hipblasHandle_t         handle,
                                                             hipblasFillMode_t       uplo,
                                                             int                     n,
                                                             const hipDoubleComplex* alpha,
                                                             const hipDoubleComplex* x,
                                                             int                     incx,
                                                             hipblasStride           stridex,
                                                             const hipDoubleComplex* y,
                                                             int                     incy,
                                                             hipblasStride           stridey,
                                                             hipDoubleComplex*       AP,
                                                             hipblasStride           strideA,
                                                             int                     batchCount);

//64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasChpr2StridedBatched_64(hipblasHandle_t       handle,
                                                             hipblasFillMode_t     uplo,
                                                             int64_t               n,
                                                             const hipblasComplex* alpha,
                                                             const hipblasComplex* x,
                                                             int64_t               incx,
                                                             hipblasStride         stridex,
                                                             const hipblasComplex* y,
                                                             int64_t               incy,
                                                             hipblasStride         stridey,
                                                             hipblasComplex*       AP,
                                                             hipblasStride         strideA,
                                                             int64_t               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhpr2StridedBatched_64(hipblasHandle_t             handle,
                                                             hipblasFillMode_t           uplo,
                                                             int64_t                     n,
                                                             const hipblasDoubleComplex* alpha,
                                                             const hipblasDoubleComplex* x,
                                                             int64_t                     incx,
                                                             hipblasStride               stridex,
                                                             const hipblasDoubleComplex* y,
                                                             int64_t                     incy,
                                                             hipblasStride               stridey,
                                                             hipblasDoubleComplex*       AP,
                                                             hipblasStride               strideA,
                                                             int64_t batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasChpr2StridedBatched_v2_64(hipblasHandle_t   handle,
                                                                hipblasFillMode_t uplo,
                                                                int64_t           n,
                                                                const hipComplex* alpha,
                                                                const hipComplex* x,
                                                                int64_t           incx,
                                                                hipblasStride     stridex,
                                                                const hipComplex* y,
                                                                int64_t           incy,
                                                                hipblasStride     stridey,
                                                                hipComplex*       AP,
                                                                hipblasStride     strideA,
                                                                int64_t           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZhpr2StridedBatched_v2_64(hipblasHandle_t         handle,
                                                                hipblasFillMode_t       uplo,
                                                                int64_t                 n,
                                                                const hipDoubleComplex* alpha,
                                                                const hipDoubleComplex* x,
                                                                int64_t                 incx,
                                                                hipblasStride           stridex,
                                                                const hipDoubleComplex* y,
                                                                int64_t                 incy,
                                                                hipblasStride           stridey,
                                                                hipDoubleComplex*       AP,
                                                                hipblasStride           strideA,
                                                                int64_t                 batchCount);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    sbmv performs the matrix-vector operation:

        y := alpha*A*x + beta*y,

    where alpha and beta are scalars, x and y are n element vectors and
    A should contain an upper or lower triangular n by n symmetric banded matrix.

    - Supported precisions in rocBLAS : s,d
    - Supported precisions in cuBLAS  : s,d

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
              if HIPBLAS_FILL_MODE_UPPER, the lower part of A is not referenced
              if HIPBLAS_FILL_MODE_LOWER, the upper part of A is not referenced
    @param[in]
    n         [int]
    @param[in]
    k         [int]
              specifies the number of sub- and super-diagonals
    @param[in]
    alpha
              specifies the scalar alpha
    @param[in]
    AP         pointer storing matrix A on the GPU
    @param[in]
    lda       [int]
              specifies the leading dimension of matrix A
    @param[in]
    x         pointer storing vector x on the GPU
    @param[in]
    incx      [int]
              specifies the increment for the elements of x
    @param[in]
    beta      specifies the scalar beta
    @param[out]
    y         pointer storing vector y on the GPU
    @param[in]
    incy      [int]
              specifies the increment for the elements of y

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSsbmv(hipblasHandle_t   handle,
                                            hipblasFillMode_t uplo,
                                            int               n,
                                            int               k,
                                            const float*      alpha,
                                            const float*      AP,
                                            int               lda,
                                            const float*      x,
                                            int               incx,
                                            const float*      beta,
                                            float*            y,
                                            int               incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasDsbmv(hipblasHandle_t   handle,
                                            hipblasFillMode_t uplo,
                                            int               n,
                                            int               k,
                                            const double*     alpha,
                                            const double*     AP,
                                            int               lda,
                                            const double*     x,
                                            int               incx,
                                            const double*     beta,
                                            double*           y,
                                            int               incy);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSsbmv_64(hipblasHandle_t   handle,
                                               hipblasFillMode_t uplo,
                                               int64_t           n,
                                               int64_t           k,
                                               const float*      alpha,
                                               const float*      AP,
                                               int64_t           lda,
                                               const float*      x,
                                               int64_t           incx,
                                               const float*      beta,
                                               float*            y,
                                               int64_t           incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasDsbmv_64(hipblasHandle_t   handle,
                                               hipblasFillMode_t uplo,
                                               int64_t           n,
                                               int64_t           k,
                                               const double*     alpha,
                                               const double*     AP,
                                               int64_t           lda,
                                               const double*     x,
                                               int64_t           incx,
                                               const double*     beta,
                                               double*           y,
                                               int64_t           incy);

//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    sbmvBatched performs the matrix-vector operation:

        y_i := alpha*A_i*x_i + beta*y_i,

    where (A_i, x_i, y_i) is the i-th instance of the batch.
    alpha and beta are scalars, x_i and y_i are vectors and A_i is an
    n by n symmetric banded matrix, for i = 1, ..., batchCount.
    A should contain an upper or lower triangular n by n symmetric banded matrix.

    - Supported precisions in rocBLAS : s,d
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue
    @param[in]
    uplo      [hipblasFillMode_t]
              specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
              if HIPBLAS_FILL_MODE_UPPER, the lower part of A is not referenced
              if HIPBLAS_FILL_MODE_LOWER, the upper part of A is not referenced
    @param[in]
    n         [int]
              number of rows and columns of each matrix A_i
    @param[in]
    k         [int]
              specifies the number of sub- and super-diagonals
    @param[in]
    alpha
              device pointer or host pointer to scalar alpha
    @param[in]
    AP         device array of device pointers storing each matrix A_i
    @param[in]
    lda       [int]
              specifies the leading dimension of each matrix A_i
    @param[in]
    x         device array of device pointers storing each vector x_i
    @param[in]
    incx      [int]
              specifies the increment for the elements of each vector x_i
    @param[in]
    beta      device pointer or host pointer to scalar beta
    @param[out]
    y         device array of device pointers storing each vector y_i
    @param[in]
    incy      [int]
              specifies the increment for the elements of each vector y_i
    @param[in]
    batchCount [int]
                number of instances in the batch

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSsbmvBatched(hipblasHandle_t    handle,
                                                   hipblasFillMode_t  uplo,
                                                   int                n,
                                                   int                k,
                                                   const float*       alpha,
                                                   const float* const AP[],
                                                   int                lda,
                                                   const float* const x[],
                                                   int                incx,
                                                   const float*       beta,
                                                   float* const       y[],
                                                   int                incy,
                                                   int                batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDsbmvBatched(hipblasHandle_t     handle,
                                                   hipblasFillMode_t   uplo,
                                                   int                 n,
                                                   int                 k,
                                                   const double*       alpha,
                                                   const double* const AP[],
                                                   int                 lda,
                                                   const double* const x[],
                                                   int                 incx,
                                                   const double*       beta,
                                                   double* const       y[],
                                                   int                 incy,
                                                   int                 batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSsbmvBatched_64(hipblasHandle_t    handle,
                                                      hipblasFillMode_t  uplo,
                                                      int64_t            n,
                                                      int64_t            k,
                                                      const float*       alpha,
                                                      const float* const AP[],
                                                      int64_t            lda,
                                                      const float* const x[],
                                                      int64_t            incx,
                                                      const float*       beta,
                                                      float* const       y[],
                                                      int64_t            incy,
                                                      int64_t            batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDsbmvBatched_64(hipblasHandle_t     handle,
                                                      hipblasFillMode_t   uplo,
                                                      int64_t             n,
                                                      int64_t             k,
                                                      const double*       alpha,
                                                      const double* const AP[],
                                                      int64_t             lda,
                                                      const double* const x[],
                                                      int64_t             incx,
                                                      const double*       beta,
                                                      double* const       y[],
                                                      int64_t             incy,
                                                      int64_t             batchCount);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    sbmvStridedBatched performs the matrix-vector operation:

        y_i := alpha*A_i*x_i + beta*y_i,

    where (A_i, x_i, y_i) is the i-th instance of the batch.
    alpha and beta are scalars, x_i and y_i are vectors and A_i is an
    n by n symmetric banded matrix, for i = 1, ..., batchCount.
    A should contain an upper or lower triangular n by n symmetric banded matrix.

    - Supported precisions in rocBLAS : s,d
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue
    @param[in]
    uplo      [hipblasFillMode_t]
              specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
              if HIPBLAS_FILL_MODE_UPPER, the lower part of A is not referenced
              if HIPBLAS_FILL_MODE_LOWER, the upper part of A is not referenced
    @param[in]
    n         [int]
              number of rows and columns of each matrix A_i
    @param[in]
    k         [int]
              specifies the number of sub- and super-diagonals
    @param[in]
    alpha
              device pointer or host pointer to scalar alpha
    @param[in]
    AP        Device pointer to the first matrix A_1 on the GPU
    @param[in]
    lda       [int]
              specifies the leading dimension of each matrix A_i
    @param[in]
    strideA     [hipblasStride]
                stride from the start of one matrix (A_i) and the next one (A_i+1)
    @param[in]
    x         Device pointer to the first vector x_1 on the GPU
    @param[in]
    incx      [int]
              specifies the increment for the elements of each vector x_i
    @param[in]
    stridex     [hipblasStride]
                stride from the start of one vector (x_i) and the next one (x_i+1).
                There are no restrictions placed on stridex, however the user should
                take care to ensure that stridex is of appropriate size.
                This typically means stridex >= n * incx. stridex should be non zero.
    @param[in]
    beta      device pointer or host pointer to scalar beta
    @param[out]
    y         Device pointer to the first vector y_1 on the GPU
    @param[in]
    incy      [int]
              specifies the increment for the elements of each vector y_i
    @param[in]
    stridey     [hipblasStride]
                stride from the start of one vector (y_i) and the next one (y_i+1).
                There are no restrictions placed on stridey, however the user should
                take care to ensure that stridey is of appropriate size.
                This typically means stridey >= n * incy. stridey should be non zero.
    @param[in]
    batchCount [int]
                number of instances in the batch

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSsbmvStridedBatched(hipblasHandle_t   handle,
                                                          hipblasFillMode_t uplo,
                                                          int               n,
                                                          int               k,
                                                          const float*      alpha,
                                                          const float*      AP,
                                                          int               lda,
                                                          hipblasStride     strideA,
                                                          const float*      x,
                                                          int               incx,
                                                          hipblasStride     stridex,
                                                          const float*      beta,
                                                          float*            y,
                                                          int               incy,
                                                          hipblasStride     stridey,
                                                          int               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDsbmvStridedBatched(hipblasHandle_t   handle,
                                                          hipblasFillMode_t uplo,
                                                          int               n,
                                                          int               k,
                                                          const double*     alpha,
                                                          const double*     AP,
                                                          int               lda,
                                                          hipblasStride     strideA,
                                                          const double*     x,
                                                          int               incx,
                                                          hipblasStride     stridex,
                                                          const double*     beta,
                                                          double*           y,
                                                          int               incy,
                                                          hipblasStride     stridey,
                                                          int               batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSsbmvStridedBatched_64(hipblasHandle_t   handle,
                                                             hipblasFillMode_t uplo,
                                                             int64_t           n,
                                                             int64_t           k,
                                                             const float*      alpha,
                                                             const float*      AP,
                                                             int64_t           lda,
                                                             hipblasStride     strideA,
                                                             const float*      x,
                                                             int64_t           incx,
                                                             hipblasStride     stridex,
                                                             const float*      beta,
                                                             float*            y,
                                                             int64_t           incy,
                                                             hipblasStride     stridey,
                                                             int64_t           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDsbmvStridedBatched_64(hipblasHandle_t   handle,
                                                             hipblasFillMode_t uplo,
                                                             int64_t           n,
                                                             int64_t           k,
                                                             const double*     alpha,
                                                             const double*     AP,
                                                             int64_t           lda,
                                                             hipblasStride     strideA,
                                                             const double*     x,
                                                             int64_t           incx,
                                                             hipblasStride     stridex,
                                                             const double*     beta,
                                                             double*           y,
                                                             int64_t           incy,
                                                             hipblasStride     stridey,
                                                             int64_t           batchCount);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    spmv performs the matrix-vector operation:

        y := alpha*A*x + beta*y,

    where alpha and beta are scalars, x and y are n element vectors and
    A should contain an upper or lower triangular n by n packed symmetric matrix.

    - Supported precisions in rocBLAS : s,d
    - Supported precisions in cuBLAS  : s,d

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
              if HIPBLAS_FILL_MODE_UPPER, the lower part of A is not referenced
              if HIPBLAS_FILL_MODE_LOWER, the upper part of A is not referenced
    @param[in]
    n         [int]
    @param[in]
    alpha
              specifies the scalar alpha
    @param[in]
    AP         pointer storing matrix A on the GPU
    @param[in]
    x         pointer storing vector x on the GPU
    @param[in]
    incx      [int]
              specifies the increment for the elements of x
    @param[in]
    beta      specifies the scalar beta
    @param[out]
    y         pointer storing vector y on the GPU
    @param[in]
    incy      [int]
              specifies the increment for the elements of y

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSspmv(hipblasHandle_t   handle,
                                            hipblasFillMode_t uplo,
                                            int               n,
                                            const float*      alpha,
                                            const float*      AP,
                                            const float*      x,
                                            int               incx,
                                            const float*      beta,
                                            float*            y,
                                            int               incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasDspmv(hipblasHandle_t   handle,
                                            hipblasFillMode_t uplo,
                                            int               n,
                                            const double*     alpha,
                                            const double*     AP,
                                            const double*     x,
                                            int               incx,
                                            const double*     beta,
                                            double*           y,
                                            int               incy);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSspmv_64(hipblasHandle_t   handle,
                                               hipblasFillMode_t uplo,
                                               int64_t           n,
                                               const float*      alpha,
                                               const float*      AP,
                                               const float*      x,
                                               int64_t           incx,
                                               const float*      beta,
                                               float*            y,
                                               int64_t           incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasDspmv_64(hipblasHandle_t   handle,
                                               hipblasFillMode_t uplo,
                                               int64_t           n,
                                               const double*     alpha,
                                               const double*     AP,
                                               const double*     x,
                                               int64_t           incx,
                                               const double*     beta,
                                               double*           y,
                                               int64_t           incy);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    spmvBatched performs the matrix-vector operation:

        y_i := alpha*AP_i*x_i + beta*y_i,

    where (A_i, x_i, y_i) is the i-th instance of the batch.
    alpha and beta are scalars, x_i and y_i are vectors and A_i is an
    n by n symmetric matrix, for i = 1, ..., batchCount.
    A should contain an upper or lower triangular n by n packed symmetric matrix.

    - Supported precisions in rocBLAS : s,d
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue
    @param[in]
    uplo      [hipblasFillMode_t]
              specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
              if HIPBLAS_FILL_MODE_UPPER, the lower part of A is not referenced
              if HIPBLAS_FILL_MODE_LOWER, the upper part of A is not referenced
    @param[in]
    n         [int]
              number of rows and columns of each matrix A_i
    @param[in]
    alpha
              device pointer or host pointer to scalar alpha
    @param[in]
    AP         device array of device pointers storing each matrix A_i
    @param[in]
    x         device array of device pointers storing each vector x_i
    @param[in]
    incx      [int]
              specifies the increment for the elements of each vector x_i
    @param[in]
    beta      device pointer or host pointer to scalar beta
    @param[out]
    y         device array of device pointers storing each vector y_i
    @param[in]
    incy      [int]
              specifies the increment for the elements of each vector y_i
    @param[in]
    batchCount [int]
                number of instances in the batch

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSspmvBatched(hipblasHandle_t    handle,
                                                   hipblasFillMode_t  uplo,
                                                   int                n,
                                                   const float*       alpha,
                                                   const float* const AP[],
                                                   const float* const x[],
                                                   int                incx,
                                                   const float*       beta,
                                                   float* const       y[],
                                                   int                incy,
                                                   int                batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDspmvBatched(hipblasHandle_t     handle,
                                                   hipblasFillMode_t   uplo,
                                                   int                 n,
                                                   const double*       alpha,
                                                   const double* const AP[],
                                                   const double* const x[],
                                                   int                 incx,
                                                   const double*       beta,
                                                   double* const       y[],
                                                   int                 incy,
                                                   int                 batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSspmvBatched_64(hipblasHandle_t    handle,
                                                      hipblasFillMode_t  uplo,
                                                      int64_t            n,
                                                      const float*       alpha,
                                                      const float* const AP[],
                                                      const float* const x[],
                                                      int64_t            incx,
                                                      const float*       beta,
                                                      float* const       y[],
                                                      int64_t            incy,
                                                      int64_t            batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDspmvBatched_64(hipblasHandle_t     handle,
                                                      hipblasFillMode_t   uplo,
                                                      int64_t             n,
                                                      const double*       alpha,
                                                      const double* const AP[],
                                                      const double* const x[],
                                                      int64_t             incx,
                                                      const double*       beta,
                                                      double* const       y[],
                                                      int64_t             incy,
                                                      int64_t             batchCount);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    spmvStridedBatched performs the matrix-vector operation:

        y_i := alpha*A_i*x_i + beta*y_i,

    where (A_i, x_i, y_i) is the i-th instance of the batch.
    alpha and beta are scalars, x_i and y_i are vectors and A_i is an
    n by n symmetric matrix, for i = 1, ..., batchCount.
    A should contain an upper or lower triangular n by n packed symmetric matrix.

    - Supported precisions in rocBLAS : s,d
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue
    @param[in]
    uplo      [hipblasFillMode_t]
              specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
              if HIPBLAS_FILL_MODE_UPPER, the lower part of A is not referenced
              if HIPBLAS_FILL_MODE_LOWER, the upper part of A is not referenced
    @param[in]
    n         [int]
              number of rows and columns of each matrix A_i
    @param[in]
    alpha
              device pointer or host pointer to scalar alpha
    @param[in]
    AP        Device pointer to the first matrix A_1 on the GPU
    @param[in]
    strideA    [hipblasStride]
                stride from the start of one matrix (A_i) and the next one (A_i+1)
    @param[in]
    x         Device pointer to the first vector x_1 on the GPU
    @param[in]
    incx      [int]
              specifies the increment for the elements of each vector x_i
    @param[in]
    stridex     [hipblasStride]
                stride from the start of one vector (x_i) and the next one (x_i+1).
                There are no restrictions placed on stridex, however the user should
                take care to ensure that stridex is of appropriate size.
                This typically means stridex >= n * incx. stridex should be non zero.
    @param[in]
    beta      device pointer or host pointer to scalar beta
    @param[out]
    y         Device pointer to the first vector y_1 on the GPU
    @param[in]
    incy      [int]
              specifies the increment for the elements of each vector y_i
    @param[in]
    stridey     [hipblasStride]
                stride from the start of one vector (y_i) and the next one (y_i+1).
                There are no restrictions placed on stridey, however the user should
                take care to ensure that stridey is of appropriate size.
                This typically means stridey >= n * incy. stridey should be non zero.
    @param[in]
    batchCount [int]
                number of instances in the batch

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSspmvStridedBatched(hipblasHandle_t   handle,
                                                          hipblasFillMode_t uplo,
                                                          int               n,
                                                          const float*      alpha,
                                                          const float*      AP,
                                                          hipblasStride     strideA,
                                                          const float*      x,
                                                          int               incx,
                                                          hipblasStride     stridex,
                                                          const float*      beta,
                                                          float*            y,
                                                          int               incy,
                                                          hipblasStride     stridey,
                                                          int               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDspmvStridedBatched(hipblasHandle_t   handle,
                                                          hipblasFillMode_t uplo,
                                                          int               n,
                                                          const double*     alpha,
                                                          const double*     AP,
                                                          hipblasStride     strideA,
                                                          const double*     x,
                                                          int               incx,
                                                          hipblasStride     stridex,
                                                          const double*     beta,
                                                          double*           y,
                                                          int               incy,
                                                          hipblasStride     stridey,
                                                          int               batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSspmvStridedBatched_64(hipblasHandle_t   handle,
                                                             hipblasFillMode_t uplo,
                                                             int64_t           n,
                                                             const float*      alpha,
                                                             const float*      AP,
                                                             hipblasStride     strideA,
                                                             const float*      x,
                                                             int64_t           incx,
                                                             hipblasStride     stridex,
                                                             const float*      beta,
                                                             float*            y,
                                                             int64_t           incy,
                                                             hipblasStride     stridey,
                                                             int64_t           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDspmvStridedBatched_64(hipblasHandle_t   handle,
                                                             hipblasFillMode_t uplo,
                                                             int64_t           n,
                                                             const double*     alpha,
                                                             const double*     AP,
                                                             hipblasStride     strideA,
                                                             const double*     x,
                                                             int64_t           incx,
                                                             hipblasStride     stridex,
                                                             const double*     beta,
                                                             double*           y,
                                                             int64_t           incy,
                                                             hipblasStride     stridey,
                                                             int64_t           batchCount);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    spr performs the matrix-vector operations

        A := A + alpha*x*x**T

    where alpha is a scalar, x is a vector, and A is an
    n by n symmetric matrix, supplied in packed form.

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : s,d,c,z

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
              HIPBLAS_FILL_MODE_UPPER: The upper triangular part of A is supplied in AP.
              HIPBLAS_FILL_MODE_LOWER: The lower triangular part of A is supplied in AP.
    @param[in]
    n         [int]
              the number of rows and columns of matrix A, must be at least 0.
    @param[in]
    alpha
              device pointer or host pointer to scalar alpha.
    @param[in]
    x         device pointer storing vector x.
    @param[in]
    incx      [int]
              specifies the increment for the elements of x.
    @param[inout]
    AP        device pointer storing the packed version of the specified triangular portion of
              the symmetric matrix A. Of at least size ((n * (n + 1)) / 2).
              if uplo == HIPBLAS_FILL_MODE_UPPER:
                The upper triangular portion of the symmetric matrix A is supplied.
                The matrix is compacted so that AP contains the triangular portion column-by-column
                so that:
                AP(0) = A(0,0)
                AP(1) = A(0,1)
                AP(2) = A(1,1), etc.
                    Ex: (HIPBLAS_FILL_MODE_UPPER; n = 4)
                        1 2 4 7
                        2 3 5 8   -----> [1, 2, 3, 4, 5, 6, 7, 8, 9, 0]
                        4 5 6 9
                        7 8 9 0
            if uplo == HIPBLAS_FILL_MODE_LOWER:
                The lower triangular portion of the symmetric matrix A is supplied.
                The matrix is compacted so that AP contains the triangular portion column-by-column
                so that:
                AP(0) = A(0,0)
                AP(1) = A(1,0)
                AP(2) = A(2,1), etc.
                    Ex: (HIPBLAS_FILL_MODE_LOWER; n = 4)
                        1 2 3 4
                        2 5 6 7    -----> [1, 2, 3, 4, 5, 6, 7, 8, 9, 0]
                        3 6 8 9
                        4 7 9 0
    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSspr(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           const float*      alpha,
                                           const float*      x,
                                           int               incx,
                                           float*            AP);

HIPBLAS_EXPORT hipblasStatus_t hipblasDspr(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           const double*     alpha,
                                           const double*     x,
                                           int               incx,
                                           double*           AP);

HIPBLAS_EXPORT hipblasStatus_t hipblasCspr(hipblasHandle_t       handle,
                                           hipblasFillMode_t     uplo,
                                           int                   n,
                                           const hipblasComplex* alpha,
                                           const hipblasComplex* x,
                                           int                   incx,
                                           hipblasComplex*       AP);

HIPBLAS_EXPORT hipblasStatus_t hipblasZspr(hipblasHandle_t             handle,
                                           hipblasFillMode_t           uplo,
                                           int                         n,
                                           const hipblasDoubleComplex* alpha,
                                           const hipblasDoubleComplex* x,
                                           int                         incx,
                                           hipblasDoubleComplex*       AP);

HIPBLAS_EXPORT hipblasStatus_t hipblasCspr_v2(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int               n,
                                              const hipComplex* alpha,
                                              const hipComplex* x,
                                              int               incx,
                                              hipComplex*       AP);

HIPBLAS_EXPORT hipblasStatus_t hipblasZspr_v2(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              int                     n,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* x,
                                              int                     incx,
                                              hipDoubleComplex*       AP);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSspr_64(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int64_t           n,
                                              const float*      alpha,
                                              const float*      x,
                                              int64_t           incx,
                                              float*            AP);

HIPBLAS_EXPORT hipblasStatus_t hipblasDspr_64(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int64_t           n,
                                              const double*     alpha,
                                              const double*     x,
                                              int64_t           incx,
                                              double*           AP);

HIPBLAS_EXPORT hipblasStatus_t hipblasCspr_64(hipblasHandle_t       handle,
                                              hipblasFillMode_t     uplo,
                                              int64_t               n,
                                              const hipblasComplex* alpha,
                                              const hipblasComplex* x,
                                              int64_t               incx,
                                              hipblasComplex*       AP);

HIPBLAS_EXPORT hipblasStatus_t hipblasZspr_64(hipblasHandle_t             handle,
                                              hipblasFillMode_t           uplo,
                                              int64_t                     n,
                                              const hipblasDoubleComplex* alpha,
                                              const hipblasDoubleComplex* x,
                                              int64_t                     incx,
                                              hipblasDoubleComplex*       AP);

HIPBLAS_EXPORT hipblasStatus_t hipblasCspr_v2_64(hipblasHandle_t   handle,
                                                 hipblasFillMode_t uplo,
                                                 int64_t           n,
                                                 const hipComplex* alpha,
                                                 const hipComplex* x,
                                                 int64_t           incx,
                                                 hipComplex*       AP);

HIPBLAS_EXPORT hipblasStatus_t hipblasZspr_v2_64(hipblasHandle_t         handle,
                                                 hipblasFillMode_t       uplo,
                                                 int64_t                 n,
                                                 const hipDoubleComplex* alpha,
                                                 const hipDoubleComplex* x,
                                                 int64_t                 incx,
                                                 hipDoubleComplex*       AP);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    sprBatched performs the matrix-vector operations

        A_i := A_i + alpha*x_i*x_i**T

    where alpha is a scalar, x_i is a vector, and A_i is an
    n by n symmetric matrix, supplied in packed form, for i = 1, ..., batchCount.

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
              HIPBLAS_FILL_MODE_UPPER: The upper triangular part of each A_i is supplied in AP.
              HIPBLAS_FILL_MODE_LOWER: The lower triangular part of each A_i is supplied in AP.
    @param[in]
    n         [int]
              the number of rows and columns of each matrix A_i, must be at least 0.
    @param[in]
    alpha
              device pointer or host pointer to scalar alpha.
    @param[in]
    x         device array of device pointers storing each vector x_i.
    @param[in]
    incx      [int]
              specifies the increment for the elements of each x_i.
    @param[inout]
    AP        device array of device pointers storing the packed version of the specified triangular portion of
              each symmetric matrix A_i of at least size ((n * (n + 1)) / 2). Array is of at least size batchCount.
              if uplo == HIPBLAS_FILL_MODE_UPPER:
                The upper triangular portion of each symmetric matrix A_i is supplied.
                The matrix is compacted so that AP contains the triangular portion column-by-column
                so that:
                AP(0) = A(0,0)
                AP(1) = A(0,1)
                AP(2) = A(1,1), etc.
                    Ex: (HIPBLAS_FILL_MODE_UPPER; n = 4)
                        1 2 4 7
                        2 3 5 8   -----> [1, 2, 3, 4, 5, 6, 7, 8, 9, 0]
                        4 5 6 9
                        7 8 9 0
            if uplo == HIPBLAS_FILL_MODE_LOWER:
                The lower triangular portion of each symmetric matrix A_i is supplied.
                The matrix is compacted so that AP contains the triangular portion column-by-column
                so that:
                AP(0) = A(0,0)
                AP(1) = A(1,0)
                AP(2) = A(2,1), etc.
                    Ex: (HIPBLAS_FILL_MODE_LOWER; n = 4)
                        1 2 3 4
                        2 5 6 7    -----> [1, 2, 3, 4, 5, 6, 7, 8, 9, 0]
                        3 6 8 9
                        4 7 9 0
    @param[in]
    batchCount [int]
                number of instances in the batch.
    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSsprBatched(hipblasHandle_t    handle,
                                                  hipblasFillMode_t  uplo,
                                                  int                n,
                                                  const float*       alpha,
                                                  const float* const x[],
                                                  int                incx,
                                                  float* const       AP[],
                                                  int                batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDsprBatched(hipblasHandle_t     handle,
                                                  hipblasFillMode_t   uplo,
                                                  int                 n,
                                                  const double*       alpha,
                                                  const double* const x[],
                                                  int                 incx,
                                                  double* const       AP[],
                                                  int                 batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsprBatched(hipblasHandle_t             handle,
                                                  hipblasFillMode_t           uplo,
                                                  int                         n,
                                                  const hipblasComplex*       alpha,
                                                  const hipblasComplex* const x[],
                                                  int                         incx,
                                                  hipblasComplex* const       AP[],
                                                  int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsprBatched(hipblasHandle_t                   handle,
                                                  hipblasFillMode_t                 uplo,
                                                  int                               n,
                                                  const hipblasDoubleComplex*       alpha,
                                                  const hipblasDoubleComplex* const x[],
                                                  int                               incx,
                                                  hipblasDoubleComplex* const       AP[],
                                                  int                               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsprBatched_v2(hipblasHandle_t         handle,
                                                     hipblasFillMode_t       uplo,
                                                     int                     n,
                                                     const hipComplex*       alpha,
                                                     const hipComplex* const x[],
                                                     int                     incx,
                                                     hipComplex* const       AP[],
                                                     int                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsprBatched_v2(hipblasHandle_t               handle,
                                                     hipblasFillMode_t             uplo,
                                                     int                           n,
                                                     const hipDoubleComplex*       alpha,
                                                     const hipDoubleComplex* const x[],
                                                     int                           incx,
                                                     hipDoubleComplex* const       AP[],
                                                     int                           batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSsprBatched_64(hipblasHandle_t    handle,
                                                     hipblasFillMode_t  uplo,
                                                     int64_t            n,
                                                     const float*       alpha,
                                                     const float* const x[],
                                                     int64_t            incx,
                                                     float* const       AP[],
                                                     int64_t            batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDsprBatched_64(hipblasHandle_t     handle,
                                                     hipblasFillMode_t   uplo,
                                                     int64_t             n,
                                                     const double*       alpha,
                                                     const double* const x[],
                                                     int64_t             incx,
                                                     double* const       AP[],
                                                     int64_t             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsprBatched_64(hipblasHandle_t             handle,
                                                     hipblasFillMode_t           uplo,
                                                     int64_t                     n,
                                                     const hipblasComplex*       alpha,
                                                     const hipblasComplex* const x[],
                                                     int64_t                     incx,
                                                     hipblasComplex* const       AP[],
                                                     int64_t                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsprBatched_64(hipblasHandle_t                   handle,
                                                     hipblasFillMode_t                 uplo,
                                                     int64_t                           n,
                                                     const hipblasDoubleComplex*       alpha,
                                                     const hipblasDoubleComplex* const x[],
                                                     int64_t                           incx,
                                                     hipblasDoubleComplex* const       AP[],
                                                     int64_t                           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsprBatched_v2_64(hipblasHandle_t         handle,
                                                        hipblasFillMode_t       uplo,
                                                        int64_t                 n,
                                                        const hipComplex*       alpha,
                                                        const hipComplex* const x[],
                                                        int64_t                 incx,
                                                        hipComplex* const       AP[],
                                                        int64_t                 batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsprBatched_v2_64(hipblasHandle_t               handle,
                                                        hipblasFillMode_t             uplo,
                                                        int64_t                       n,
                                                        const hipDoubleComplex*       alpha,
                                                        const hipDoubleComplex* const x[],
                                                        int64_t                       incx,
                                                        hipDoubleComplex* const       AP[],
                                                        int64_t                       batchCount);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    sprStridedBatched performs the matrix-vector operations

        A_i := A_i + alpha*x_i*x_i**T

    where alpha is a scalar, x_i is a vector, and A_i is an
    n by n symmetric matrix, supplied in packed form, for i = 1, ..., batchCount.

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
              HIPBLAS_FILL_MODE_UPPER: The upper triangular part of each A_i is supplied in AP.
              HIPBLAS_FILL_MODE_LOWER: The lower triangular part of each A_i is supplied in AP.
    @param[in]
    n         [int]
              the number of rows and columns of each matrix A_i, must be at least 0.
    @param[in]
    alpha
              device pointer or host pointer to scalar alpha.
    @param[in]
    x         device pointer pointing to the first vector (x_1).
    @param[in]
    incx      [int]
              specifies the increment for the elements of each x_i.
    @param[in]
    stridex  [hipblasStride]
              stride from the start of one vector (x_i) and the next one (x_i+1).
    @param[inout]
    AP        device pointer storing the packed version of the specified triangular portion of
              each symmetric matrix A_i. Points to the first A_1.
              if uplo == HIPBLAS_FILL_MODE_UPPER:
                The upper triangular portion of each symmetric matrix A_i is supplied.
                The matrix is compacted so that AP contains the triangular portion column-by-column
                so that:
                AP(0) = A(0,0)
                AP(1) = A(0,1)
                AP(2) = A(1,1), etc.
                    Ex: (HIPBLAS_FILL_MODE_UPPER; n = 4)
                        1 2 4 7
                        2 3 5 8   -----> [1, 2, 3, 4, 5, 6, 7, 8, 9, 0]
                        4 5 6 9
                        7 8 9 0
            if uplo == HIPBLAS_FILL_MODE_LOWER:
                The lower triangular portion of each symmetric matrix A_i is supplied.
                The matrix is compacted so that AP contains the triangular portion column-by-column
                so that:
                AP(0) = A(0,0)
                AP(1) = A(1,0)
                AP(2) = A(2,1), etc.
                    Ex: (HIPBLAS_FILL_MODE_LOWER; n = 4)
                        1 2 3 4
                        2 5 6 7    -----> [1, 2, 3, 4, 5, 6, 7, 8, 9, 0]
                        3 6 8 9
                        4 7 9 0
    @param[in]
    strideA    [hipblasStride]
                stride from the start of one (A_i) and the next (A_i+1)
    @param[in]
    batchCount [int]
                number of instances in the batch.
    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSsprStridedBatched(hipblasHandle_t   handle,
                                                         hipblasFillMode_t uplo,
                                                         int               n,
                                                         const float*      alpha,
                                                         const float*      x,
                                                         int               incx,
                                                         hipblasStride     stridex,
                                                         float*            AP,
                                                         hipblasStride     strideA,
                                                         int               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDsprStridedBatched(hipblasHandle_t   handle,
                                                         hipblasFillMode_t uplo,
                                                         int               n,
                                                         const double*     alpha,
                                                         const double*     x,
                                                         int               incx,
                                                         hipblasStride     stridex,
                                                         double*           AP,
                                                         hipblasStride     strideA,
                                                         int               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsprStridedBatched(hipblasHandle_t       handle,
                                                         hipblasFillMode_t     uplo,
                                                         int                   n,
                                                         const hipblasComplex* alpha,
                                                         const hipblasComplex* x,
                                                         int                   incx,
                                                         hipblasStride         stridex,
                                                         hipblasComplex*       AP,
                                                         hipblasStride         strideA,
                                                         int                   batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsprStridedBatched(hipblasHandle_t             handle,
                                                         hipblasFillMode_t           uplo,
                                                         int                         n,
                                                         const hipblasDoubleComplex* alpha,
                                                         const hipblasDoubleComplex* x,
                                                         int                         incx,
                                                         hipblasStride               stridex,
                                                         hipblasDoubleComplex*       AP,
                                                         hipblasStride               strideA,
                                                         int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsprStridedBatched_v2(hipblasHandle_t   handle,
                                                            hipblasFillMode_t uplo,
                                                            int               n,
                                                            const hipComplex* alpha,
                                                            const hipComplex* x,
                                                            int               incx,
                                                            hipblasStride     stridex,
                                                            hipComplex*       AP,
                                                            hipblasStride     strideA,
                                                            int               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsprStridedBatched_v2(hipblasHandle_t         handle,
                                                            hipblasFillMode_t       uplo,
                                                            int                     n,
                                                            const hipDoubleComplex* alpha,
                                                            const hipDoubleComplex* x,
                                                            int                     incx,
                                                            hipblasStride           stridex,
                                                            hipDoubleComplex*       AP,
                                                            hipblasStride           strideA,
                                                            int                     batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSsprStridedBatched_64(hipblasHandle_t   handle,
                                                            hipblasFillMode_t uplo,
                                                            int64_t           n,
                                                            const float*      alpha,
                                                            const float*      x,
                                                            int64_t           incx,
                                                            hipblasStride     stridex,
                                                            float*            AP,
                                                            hipblasStride     strideA,
                                                            int64_t           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDsprStridedBatched_64(hipblasHandle_t   handle,
                                                            hipblasFillMode_t uplo,
                                                            int64_t           n,
                                                            const double*     alpha,
                                                            const double*     x,
                                                            int64_t           incx,
                                                            hipblasStride     stridex,
                                                            double*           AP,
                                                            hipblasStride     strideA,
                                                            int64_t           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsprStridedBatched_64(hipblasHandle_t       handle,
                                                            hipblasFillMode_t     uplo,
                                                            int64_t               n,
                                                            const hipblasComplex* alpha,
                                                            const hipblasComplex* x,
                                                            int64_t               incx,
                                                            hipblasStride         stridex,
                                                            hipblasComplex*       AP,
                                                            hipblasStride         strideA,
                                                            int64_t               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsprStridedBatched_64(hipblasHandle_t             handle,
                                                            hipblasFillMode_t           uplo,
                                                            int64_t                     n,
                                                            const hipblasDoubleComplex* alpha,
                                                            const hipblasDoubleComplex* x,
                                                            int64_t                     incx,
                                                            hipblasStride               stridex,
                                                            hipblasDoubleComplex*       AP,
                                                            hipblasStride               strideA,
                                                            int64_t                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsprStridedBatched_v2_64(hipblasHandle_t   handle,
                                                               hipblasFillMode_t uplo,
                                                               int64_t           n,
                                                               const hipComplex* alpha,
                                                               const hipComplex* x,
                                                               int64_t           incx,
                                                               hipblasStride     stridex,
                                                               hipComplex*       AP,
                                                               hipblasStride     strideA,
                                                               int64_t           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsprStridedBatched_v2_64(hipblasHandle_t         handle,
                                                               hipblasFillMode_t       uplo,
                                                               int64_t                 n,
                                                               const hipDoubleComplex* alpha,
                                                               const hipDoubleComplex* x,
                                                               int64_t                 incx,
                                                               hipblasStride           stridex,
                                                               hipDoubleComplex*       AP,
                                                               hipblasStride           strideA,
                                                               int64_t                 batchCount);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    spr2 performs the matrix-vector operation

        A := A + alpha*x*y**T + alpha*y*x**T

    where alpha is a scalar, x and y are vectors, and A is an
    n by n symmetric matrix, supplied in packed form.

    - Supported precisions in rocBLAS : s,d
    - Supported precisions in cuBLAS  : s,d

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
              HIPBLAS_FILL_MODE_UPPER: The upper triangular part of A is supplied in AP.
              HIPBLAS_FILL_MODE_LOWER: The lower triangular part of A is supplied in AP.
    @param[in]
    n         [int]
              the number of rows and columns of matrix A, must be at least 0.
    @param[in]
    alpha
              device pointer or host pointer to scalar alpha.
    @param[in]
    x         device pointer storing vector x.
    @param[in]
    incx      [int]
              specifies the increment for the elements of x.
    @param[in]
    y         device pointer storing vector y.
    @param[in]
    incy      [int]
              specifies the increment for the elements of y.
    @param[inout]
    AP        device pointer storing the packed version of the specified triangular portion of
              the symmetric matrix A. Of at least size ((n * (n + 1)) / 2).
              if uplo == HIPBLAS_FILL_MODE_UPPER:
                The upper triangular portion of the symmetric matrix A is supplied.
                The matrix is compacted so that AP contains the triangular portion column-by-column
                so that:
                AP(0) = A(0,0)
                AP(1) = A(0,1)
                AP(2) = A(1,1), etc.
                    Ex: (HIPBLAS_FILL_MODE_UPPER; n = 4)
                        1 2 4 7
                        2 3 5 8   -----> [1, 2, 3, 4, 5, 6, 7, 8, 9, 0]
                        4 5 6 9
                        7 8 9 0
            if uplo == HIPBLAS_FILL_MODE_LOWER:
                The lower triangular portion of the symmetric matrix A is supplied.
                The matrix is compacted so that AP contains the triangular portion column-by-column
                so that:
                AP(0) = A(0,0)
                AP(1) = A(1,0)
                AP(n) = A(2,1), etc.
                    Ex: (HIPBLAS_FILL_MODE_LOWER; n = 4)
                        1 2 3 4
                        2 5 6 7    -----> [1, 2, 3, 4, 5, 6, 7, 8, 9, 0]
                        3 6 8 9
                        4 7 9 0
    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSspr2(hipblasHandle_t   handle,
                                            hipblasFillMode_t uplo,
                                            int               n,
                                            const float*      alpha,
                                            const float*      x,
                                            int               incx,
                                            const float*      y,
                                            int               incy,
                                            float*            AP);

HIPBLAS_EXPORT hipblasStatus_t hipblasDspr2(hipblasHandle_t   handle,
                                            hipblasFillMode_t uplo,
                                            int               n,
                                            const double*     alpha,
                                            const double*     x,
                                            int               incx,
                                            const double*     y,
                                            int               incy,
                                            double*           AP);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSspr2_64(hipblasHandle_t   handle,
                                               hipblasFillMode_t uplo,
                                               int64_t           n,
                                               const float*      alpha,
                                               const float*      x,
                                               int64_t           incx,
                                               const float*      y,
                                               int64_t           incy,
                                               float*            AP);

HIPBLAS_EXPORT hipblasStatus_t hipblasDspr2_64(hipblasHandle_t   handle,
                                               hipblasFillMode_t uplo,
                                               int64_t           n,
                                               const double*     alpha,
                                               const double*     x,
                                               int64_t           incx,
                                               const double*     y,
                                               int64_t           incy,
                                               double*           AP);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    spr2Batched performs the matrix-vector operation

        A_i := A_i + alpha*x_i*y_i**T + alpha*y_i*x_i**T

    where alpha is a scalar, x_i and y_i are vectors, and A_i is an
    n by n symmetric matrix, supplied in packed form, for i = 1, ..., batchCount.

    - Supported precisions in rocBLAS : s,d
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
              HIPBLAS_FILL_MODE_UPPER: The upper triangular part of each A_i is supplied in AP.
              HIPBLAS_FILL_MODE_LOWER: The lower triangular part of each A_i is supplied in AP.
    @param[in]
    n         [int]
              the number of rows and columns of each matrix A_i, must be at least 0.
    @param[in]
    alpha
              device pointer or host pointer to scalar alpha.
    @param[in]
    x         device array of device pointers storing each vector x_i.
    @param[in]
    incx      [int]
              specifies the increment for the elements of each x_i.
    @param[in]
    y         device array of device pointers storing each vector y_i.
    @param[in]
    incy      [int]
              specifies the increment for the elements of each y_i.
    @param[inout]
    AP        device array of device pointers storing the packed version of the specified triangular portion of
              each symmetric matrix A_i of at least size ((n * (n + 1)) / 2). Array is of at least size batchCount.
              if uplo == HIPBLAS_FILL_MODE_UPPER:
                The upper triangular portion of each symmetric matrix A_i is supplied.
                The matrix is compacted so that AP contains the triangular portion column-by-column
                so that:
                AP(0) = A(0,0)
                AP(1) = A(0,1)
                AP(2) = A(1,1), etc.
                    Ex: (HIPBLAS_FILL_MODE_UPPER; n = 4)
                        1 2 4 7
                        2 3 5 8   -----> [1, 2, 3, 4, 5, 6, 7, 8, 9, 0]
                        4 5 6 9
                        7 8 9 0
            if uplo == HIPBLAS_FILL_MODE_LOWER:
                The lower triangular portion of each symmetric matrix A_i is supplied.
                The matrix is compacted so that AP contains the triangular portion column-by-column
                so that:
                AP(0) = A(0,0)
                AP(1) = A(1,0)
                AP(n) = A(2,1), etc.
                    Ex: (HIPBLAS_FILL_MODE_LOWER; n = 4)
                        1 2 3 4
                        2 5 6 7    -----> [1, 2, 3, 4, 5, 6, 7, 8, 9, 0]
                        3 6 8 9
                        4 7 9 0
    @param[in]
    batchCount [int]
                number of instances in the batch.
    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSspr2Batched(hipblasHandle_t    handle,
                                                   hipblasFillMode_t  uplo,
                                                   int                n,
                                                   const float*       alpha,
                                                   const float* const x[],
                                                   int                incx,
                                                   const float* const y[],
                                                   int                incy,
                                                   float* const       AP[],
                                                   int                batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDspr2Batched(hipblasHandle_t     handle,
                                                   hipblasFillMode_t   uplo,
                                                   int                 n,
                                                   const double*       alpha,
                                                   const double* const x[],
                                                   int                 incx,
                                                   const double* const y[],
                                                   int                 incy,
                                                   double* const       AP[],
                                                   int                 batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSspr2Batched_64(hipblasHandle_t    handle,
                                                      hipblasFillMode_t  uplo,
                                                      int64_t            n,
                                                      const float*       alpha,
                                                      const float* const x[],
                                                      int64_t            incx,
                                                      const float* const y[],
                                                      int64_t            incy,
                                                      float* const       AP[],
                                                      int64_t            batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDspr2Batched_64(hipblasHandle_t     handle,
                                                      hipblasFillMode_t   uplo,
                                                      int64_t             n,
                                                      const double*       alpha,
                                                      const double* const x[],
                                                      int64_t             incx,
                                                      const double* const y[],
                                                      int64_t             incy,
                                                      double* const       AP[],
                                                      int64_t             batchCount);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    spr2StridedBatched performs the matrix-vector operation

        A_i := A_i + alpha*x_i*y_i**T + alpha*y_i*x_i**T

    where alpha is a scalar, x_i amd y_i are vectors, and A_i is an
    n by n symmetric matrix, supplied in packed form, for i = 1, ..., batchCount.

    - Supported precisions in rocBLAS : s,d
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
              HIPBLAS_FILL_MODE_UPPER: The upper triangular part of each A_i is supplied in AP.
              HIPBLAS_FILL_MODE_LOWER: The lower triangular part of each A_i is supplied in AP.
    @param[in]
    n         [int]
              the number of rows and columns of each matrix A_i, must be at least 0.
    @param[in]
    alpha
              device pointer or host pointer to scalar alpha.
    @param[in]
    x         device pointer pointing to the first vector (x_1).
    @param[in]
    incx      [int]
              specifies the increment for the elements of each x_i.
    @param[in]
    stridex  [hipblasStride]
              stride from the start of one vector (x_i) and the next one (x_i+1).
    @param[in]
    y         device pointer pointing to the first vector (y_1).
    @param[in]
    incy      [int]
              specifies the increment for the elements of each y_i.
    @param[in]
    stridey  [hipblasStride]
              stride from the start of one vector (y_i) and the next one (y_i+1).
    @param[inout]
    AP        device pointer storing the packed version of the specified triangular portion of
              each symmetric matrix A_i. Points to the first A_1.
              if uplo == HIPBLAS_FILL_MODE_UPPER:
                The upper triangular portion of each symmetric matrix A_i is supplied.
                The matrix is compacted so that AP contains the triangular portion column-by-column
                so that:
                AP(0) = A(0,0)
                AP(1) = A(0,1)
                AP(2) = A(1,1), etc.
                    Ex: (HIPBLAS_FILL_MODE_UPPER; n = 4)
                        1 2 4 7
                        2 3 5 8   -----> [1, 2, 3, 4, 5, 6, 7, 8, 9, 0]
                        4 5 6 9
                        7 8 9 0
            if uplo == HIPBLAS_FILL_MODE_LOWER:
                The lower triangular portion of each symmetric matrix A_i is supplied.
                The matrix is compacted so that AP contains the triangular portion column-by-column
                so that:
                AP(0) = A(0,0)
                AP(1) = A(1,0)
                AP(n) = A(2,1), etc.
                    Ex: (HIPBLAS_FILL_MODE_LOWER; n = 4)
                        1 2 3 4
                        2 5 6 7    -----> [1, 2, 3, 4, 5, 6, 7, 8, 9, 0]
                        3 6 8 9
                        4 7 9 0
    @param[in]
    strideA   [hipblasStride]
                stride from the start of one (A_i) and the next (A_i+1)
    @param[in]
    batchCount [int]
                number of instances in the batch.
    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSspr2StridedBatched(hipblasHandle_t   handle,
                                                          hipblasFillMode_t uplo,
                                                          int               n,
                                                          const float*      alpha,
                                                          const float*      x,
                                                          int               incx,
                                                          hipblasStride     stridex,
                                                          const float*      y,
                                                          int               incy,
                                                          hipblasStride     stridey,
                                                          float*            AP,
                                                          hipblasStride     strideA,
                                                          int               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDspr2StridedBatched(hipblasHandle_t   handle,
                                                          hipblasFillMode_t uplo,
                                                          int               n,
                                                          const double*     alpha,
                                                          const double*     x,
                                                          int               incx,
                                                          hipblasStride     stridex,
                                                          const double*     y,
                                                          int               incy,
                                                          hipblasStride     stridey,
                                                          double*           AP,
                                                          hipblasStride     strideA,
                                                          int               batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSspr2StridedBatched_64(hipblasHandle_t   handle,
                                                             hipblasFillMode_t uplo,
                                                             int64_t           n,
                                                             const float*      alpha,
                                                             const float*      x,
                                                             int64_t           incx,
                                                             hipblasStride     stridex,
                                                             const float*      y,
                                                             int64_t           incy,
                                                             hipblasStride     stridey,
                                                             float*            AP,
                                                             hipblasStride     strideA,
                                                             int64_t           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDspr2StridedBatched_64(hipblasHandle_t   handle,
                                                             hipblasFillMode_t uplo,
                                                             int64_t           n,
                                                             const double*     alpha,
                                                             const double*     x,
                                                             int64_t           incx,
                                                             hipblasStride     stridex,
                                                             const double*     y,
                                                             int64_t           incy,
                                                             hipblasStride     stridey,
                                                             double*           AP,
                                                             hipblasStride     strideA,
                                                             int64_t           batchCount);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    symv performs the matrix-vector operation:

        y := alpha*A*x + beta*y,

    where alpha and beta are scalars, x and y are n element vectors and
    A should contain an upper or lower triangular n by n symmetric matrix.

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : s,d,c,z

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
              if HIPBLAS_FILL_MODE_UPPER, the lower part of A is not referenced
              if HIPBLAS_FILL_MODE_LOWER, the upper part of A is not referenced
    @param[in]
    n         [int]
    @param[in]
    alpha
              specifies the scalar alpha
    @param[in]
    AP         pointer storing matrix A on the GPU
    @param[in]
    lda       [int]
              specifies the leading dimension of A
    @param[in]
    x         pointer storing vector x on the GPU
    @param[in]
    incx      [int]
              specifies the increment for the elements of x
    @param[in]
    beta      specifies the scalar beta
    @param[out]
    y         pointer storing vector y on the GPU
    @param[in]
    incy      [int]
              specifies the increment for the elements of y

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSsymv(hipblasHandle_t   handle,
                                            hipblasFillMode_t uplo,
                                            int               n,
                                            const float*      alpha,
                                            const float*      AP,
                                            int               lda,
                                            const float*      x,
                                            int               incx,
                                            const float*      beta,
                                            float*            y,
                                            int               incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasDsymv(hipblasHandle_t   handle,
                                            hipblasFillMode_t uplo,
                                            int               n,
                                            const double*     alpha,
                                            const double*     AP,
                                            int               lda,
                                            const double*     x,
                                            int               incx,
                                            const double*     beta,
                                            double*           y,
                                            int               incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsymv(hipblasHandle_t       handle,
                                            hipblasFillMode_t     uplo,
                                            int                   n,
                                            const hipblasComplex* alpha,
                                            const hipblasComplex* AP,
                                            int                   lda,
                                            const hipblasComplex* x,
                                            int                   incx,
                                            const hipblasComplex* beta,
                                            hipblasComplex*       y,
                                            int                   incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsymv(hipblasHandle_t             handle,
                                            hipblasFillMode_t           uplo,
                                            int                         n,
                                            const hipblasDoubleComplex* alpha,
                                            const hipblasDoubleComplex* AP,
                                            int                         lda,
                                            const hipblasDoubleComplex* x,
                                            int                         incx,
                                            const hipblasDoubleComplex* beta,
                                            hipblasDoubleComplex*       y,
                                            int                         incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsymv_v2(hipblasHandle_t   handle,
                                               hipblasFillMode_t uplo,
                                               int               n,
                                               const hipComplex* alpha,
                                               const hipComplex* AP,
                                               int               lda,
                                               const hipComplex* x,
                                               int               incx,
                                               const hipComplex* beta,
                                               hipComplex*       y,
                                               int               incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsymv_v2(hipblasHandle_t         handle,
                                               hipblasFillMode_t       uplo,
                                               int                     n,
                                               const hipDoubleComplex* alpha,
                                               const hipDoubleComplex* AP,
                                               int                     lda,
                                               const hipDoubleComplex* x,
                                               int                     incx,
                                               const hipDoubleComplex* beta,
                                               hipDoubleComplex*       y,
                                               int                     incy);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSsymv_64(hipblasHandle_t   handle,
                                               hipblasFillMode_t uplo,
                                               int64_t           n,
                                               const float*      alpha,
                                               const float*      AP,
                                               int64_t           lda,
                                               const float*      x,
                                               int64_t           incx,
                                               const float*      beta,
                                               float*            y,
                                               int64_t           incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasDsymv_64(hipblasHandle_t   handle,
                                               hipblasFillMode_t uplo,
                                               int64_t           n,
                                               const double*     alpha,
                                               const double*     AP,
                                               int64_t           lda,
                                               const double*     x,
                                               int64_t           incx,
                                               const double*     beta,
                                               double*           y,
                                               int64_t           incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsymv_64(hipblasHandle_t       handle,
                                               hipblasFillMode_t     uplo,
                                               int64_t               n,
                                               const hipblasComplex* alpha,
                                               const hipblasComplex* AP,
                                               int64_t               lda,
                                               const hipblasComplex* x,
                                               int64_t               incx,
                                               const hipblasComplex* beta,
                                               hipblasComplex*       y,
                                               int64_t               incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsymv_64(hipblasHandle_t             handle,
                                               hipblasFillMode_t           uplo,
                                               int64_t                     n,
                                               const hipblasDoubleComplex* alpha,
                                               const hipblasDoubleComplex* AP,
                                               int64_t                     lda,
                                               const hipblasDoubleComplex* x,
                                               int64_t                     incx,
                                               const hipblasDoubleComplex* beta,
                                               hipblasDoubleComplex*       y,
                                               int64_t                     incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsymv_v2_64(hipblasHandle_t   handle,
                                                  hipblasFillMode_t uplo,
                                                  int64_t           n,
                                                  const hipComplex* alpha,
                                                  const hipComplex* AP,
                                                  int64_t           lda,
                                                  const hipComplex* x,
                                                  int64_t           incx,
                                                  const hipComplex* beta,
                                                  hipComplex*       y,
                                                  int64_t           incy);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsymv_v2_64(hipblasHandle_t         handle,
                                                  hipblasFillMode_t       uplo,
                                                  int64_t                 n,
                                                  const hipDoubleComplex* alpha,
                                                  const hipDoubleComplex* AP,
                                                  int64_t                 lda,
                                                  const hipDoubleComplex* x,
                                                  int64_t                 incx,
                                                  const hipDoubleComplex* beta,
                                                  hipDoubleComplex*       y,
                                                  int64_t                 incy);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    symvBatched performs the matrix-vector operation:

        y_i := alpha*A_i*x_i + beta*y_i,

    where (A_i, x_i, y_i) is the i-th instance of the batch.
    alpha and beta are scalars, x_i and y_i are vectors and A_i is an
    n by n symmetric matrix, for i = 1, ..., batchCount.
    A a should contain an upper or lower triangular symmetric matrix
    and the opposing triangular part of A is not referenced

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue
    @param[in]
    uplo      [hipblasFillMode_t]
              specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
              if HIPBLAS_FILL_MODE_UPPER, the lower part of A is not referenced
              if HIPBLAS_FILL_MODE_LOWER, the upper part of A is not referenced
    @param[in]
    n         [int]
              number of rows and columns of each matrix A_i
    @param[in]
    alpha
              device pointer or host pointer to scalar alpha
    @param[in]
    AP        device array of device pointers storing each matrix A_i
    @param[in]
    lda       [int]
              specifies the leading dimension of each matrix A_i
    @param[in]
    x         device array of device pointers storing each vector x_i
    @param[in]
    incx      [int]
              specifies the increment for the elements of each vector x_i
    @param[in]
    beta      device pointer or host pointer to scalar beta
    @param[out]
    y         device array of device pointers storing each vector y_i
    @param[in]
    incy      [int]
              specifies the increment for the elements of each vector y_i
    @param[in]
    batchCount [int]
                number of instances in the batch

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSsymvBatched(hipblasHandle_t    handle,
                                                   hipblasFillMode_t  uplo,
                                                   int                n,
                                                   const float*       alpha,
                                                   const float* const AP[],
                                                   int                lda,
                                                   const float* const x[],
                                                   int                incx,
                                                   const float*       beta,
                                                   float* const       y[],
                                                   int                incy,
                                                   int                batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDsymvBatched(hipblasHandle_t     handle,
                                                   hipblasFillMode_t   uplo,
                                                   int                 n,
                                                   const double*       alpha,
                                                   const double* const AP[],
                                                   int                 lda,
                                                   const double* const x[],
                                                   int                 incx,
                                                   const double*       beta,
                                                   double* const       y[],
                                                   int                 incy,
                                                   int                 batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsymvBatched(hipblasHandle_t             handle,
                                                   hipblasFillMode_t           uplo,
                                                   int                         n,
                                                   const hipblasComplex*       alpha,
                                                   const hipblasComplex* const AP[],
                                                   int                         lda,
                                                   const hipblasComplex* const x[],
                                                   int                         incx,
                                                   const hipblasComplex*       beta,
                                                   hipblasComplex* const       y[],
                                                   int                         incy,
                                                   int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsymvBatched(hipblasHandle_t                   handle,
                                                   hipblasFillMode_t                 uplo,
                                                   int                               n,
                                                   const hipblasDoubleComplex*       alpha,
                                                   const hipblasDoubleComplex* const AP[],
                                                   int                               lda,
                                                   const hipblasDoubleComplex* const x[],
                                                   int                               incx,
                                                   const hipblasDoubleComplex*       beta,
                                                   hipblasDoubleComplex* const       y[],
                                                   int                               incy,
                                                   int                               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsymvBatched_v2(hipblasHandle_t         handle,
                                                      hipblasFillMode_t       uplo,
                                                      int                     n,
                                                      const hipComplex*       alpha,
                                                      const hipComplex* const AP[],
                                                      int                     lda,
                                                      const hipComplex* const x[],
                                                      int                     incx,
                                                      const hipComplex*       beta,
                                                      hipComplex* const       y[],
                                                      int                     incy,
                                                      int                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsymvBatched_v2(hipblasHandle_t               handle,
                                                      hipblasFillMode_t             uplo,
                                                      int                           n,
                                                      const hipDoubleComplex*       alpha,
                                                      const hipDoubleComplex* const AP[],
                                                      int                           lda,
                                                      const hipDoubleComplex* const x[],
                                                      int                           incx,
                                                      const hipDoubleComplex*       beta,
                                                      hipDoubleComplex* const       y[],
                                                      int                           incy,
                                                      int                           batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSsymvBatched_64(hipblasHandle_t    handle,
                                                      hipblasFillMode_t  uplo,
                                                      int64_t            n,
                                                      const float*       alpha,
                                                      const float* const AP[],
                                                      int64_t            lda,
                                                      const float* const x[],
                                                      int64_t            incx,
                                                      const float*       beta,
                                                      float* const       y[],
                                                      int64_t            incy,
                                                      int64_t            batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDsymvBatched_64(hipblasHandle_t     handle,
                                                      hipblasFillMode_t   uplo,
                                                      int64_t             n,
                                                      const double*       alpha,
                                                      const double* const AP[],
                                                      int64_t             lda,
                                                      const double* const x[],
                                                      int64_t             incx,
                                                      const double*       beta,
                                                      double* const       y[],
                                                      int64_t             incy,
                                                      int64_t             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsymvBatched_64(hipblasHandle_t             handle,
                                                      hipblasFillMode_t           uplo,
                                                      int64_t                     n,
                                                      const hipblasComplex*       alpha,
                                                      const hipblasComplex* const AP[],
                                                      int64_t                     lda,
                                                      const hipblasComplex* const x[],
                                                      int64_t                     incx,
                                                      const hipblasComplex*       beta,
                                                      hipblasComplex* const       y[],
                                                      int64_t                     incy,
                                                      int64_t                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsymvBatched_64(hipblasHandle_t                   handle,
                                                      hipblasFillMode_t                 uplo,
                                                      int64_t                           n,
                                                      const hipblasDoubleComplex*       alpha,
                                                      const hipblasDoubleComplex* const AP[],
                                                      int64_t                           lda,
                                                      const hipblasDoubleComplex* const x[],
                                                      int64_t                           incx,
                                                      const hipblasDoubleComplex*       beta,
                                                      hipblasDoubleComplex* const       y[],
                                                      int64_t                           incy,
                                                      int64_t                           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsymvBatched_v2_64(hipblasHandle_t         handle,
                                                         hipblasFillMode_t       uplo,
                                                         int64_t                 n,
                                                         const hipComplex*       alpha,
                                                         const hipComplex* const AP[],
                                                         int64_t                 lda,
                                                         const hipComplex* const x[],
                                                         int64_t                 incx,
                                                         const hipComplex*       beta,
                                                         hipComplex* const       y[],
                                                         int64_t                 incy,
                                                         int64_t                 batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsymvBatched_v2_64(hipblasHandle_t               handle,
                                                         hipblasFillMode_t             uplo,
                                                         int64_t                       n,
                                                         const hipDoubleComplex*       alpha,
                                                         const hipDoubleComplex* const AP[],
                                                         int64_t                       lda,
                                                         const hipDoubleComplex* const x[],
                                                         int64_t                       incx,
                                                         const hipDoubleComplex*       beta,
                                                         hipDoubleComplex* const       y[],
                                                         int64_t                       incy,
                                                         int64_t                       batchCount);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    symvStridedBatched performs the matrix-vector operation:

        y_i := alpha*A_i*x_i + beta*y_i,

    where (A_i, x_i, y_i) is the i-th instance of the batch.
    alpha and beta are scalars, x_i and y_i are vectors and A_i is an
    n by n symmetric matrix, for i = 1, ..., batchCount.
    A a should contain an upper or lower triangular symmetric matrix
    and the opposing triangular part of A is not referenced

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue
    @param[in]
    uplo      [hipblasFillMode_t]
              specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
              if HIPBLAS_FILL_MODE_UPPER, the lower part of A is not referenced
              if HIPBLAS_FILL_MODE_LOWER, the upper part of A is not referenced
    @param[in]
    n         [int]
              number of rows and columns of each matrix A_i
    @param[in]
    alpha
              device pointer or host pointer to scalar alpha
    @param[in]
    AP         Device pointer to the first matrix A_1 on the GPU
    @param[in]
    lda       [int]
              specifies the leading dimension of each matrix A_i
    @param[in]
    strideA     [hipblasStride]
                stride from the start of one matrix (A_i) and the next one (A_i+1)
    @param[in]
    x         Device pointer to the first vector x_1 on the GPU
    @param[in]
    incx      [int]
              specifies the increment for the elements of each vector x_i
    @param[in]
    stridex     [hipblasStride]
                stride from the start of one vector (x_i) and the next one (x_i+1).
                There are no restrictions placed on stridex, however the user should
                take care to ensure that stridex is of appropriate size.
                This typically means stridex >= n * incx. stridex should be non zero.
    @param[in]
    beta      device pointer or host pointer to scalar beta
    @param[out]
    y         Device pointer to the first vector y_1 on the GPU
    @param[in]
    incy      [int]
              specifies the increment for the elements of each vector y_i
    @param[in]
    stridey     [hipblasStride]
                stride from the start of one vector (y_i) and the next one (y_i+1).
                There are no restrictions placed on stridey, however the user should
                take care to ensure that stridey is of appropriate size.
                This typically means stridey >= n * incy. stridey should be non zero.
    @param[in]
    batchCount [int]
                number of instances in the batch

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSsymvStridedBatched(hipblasHandle_t   handle,
                                                          hipblasFillMode_t uplo,
                                                          int               n,
                                                          const float*      alpha,
                                                          const float*      AP,
                                                          int               lda,
                                                          hipblasStride     strideA,
                                                          const float*      x,
                                                          int               incx,
                                                          hipblasStride     stridex,
                                                          const float*      beta,
                                                          float*            y,
                                                          int               incy,
                                                          hipblasStride     stridey,
                                                          int               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDsymvStridedBatched(hipblasHandle_t   handle,
                                                          hipblasFillMode_t uplo,
                                                          int               n,
                                                          const double*     alpha,
                                                          const double*     AP,
                                                          int               lda,
                                                          hipblasStride     strideA,
                                                          const double*     x,
                                                          int               incx,
                                                          hipblasStride     stridex,
                                                          const double*     beta,
                                                          double*           y,
                                                          int               incy,
                                                          hipblasStride     stridey,
                                                          int               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsymvStridedBatched(hipblasHandle_t       handle,
                                                          hipblasFillMode_t     uplo,
                                                          int                   n,
                                                          const hipblasComplex* alpha,
                                                          const hipblasComplex* AP,
                                                          int                   lda,
                                                          hipblasStride         strideA,
                                                          const hipblasComplex* x,
                                                          int                   incx,
                                                          hipblasStride         stridex,
                                                          const hipblasComplex* beta,
                                                          hipblasComplex*       y,
                                                          int                   incy,
                                                          hipblasStride         stridey,
                                                          int                   batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsymvStridedBatched(hipblasHandle_t             handle,
                                                          hipblasFillMode_t           uplo,
                                                          int                         n,
                                                          const hipblasDoubleComplex* alpha,
                                                          const hipblasDoubleComplex* AP,
                                                          int                         lda,
                                                          hipblasStride               strideA,
                                                          const hipblasDoubleComplex* x,
                                                          int                         incx,
                                                          hipblasStride               stridex,
                                                          const hipblasDoubleComplex* beta,
                                                          hipblasDoubleComplex*       y,
                                                          int                         incy,
                                                          hipblasStride               stridey,
                                                          int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsymvStridedBatched_v2(hipblasHandle_t   handle,
                                                             hipblasFillMode_t uplo,
                                                             int               n,
                                                             const hipComplex* alpha,
                                                             const hipComplex* AP,
                                                             int               lda,
                                                             hipblasStride     strideA,
                                                             const hipComplex* x,
                                                             int               incx,
                                                             hipblasStride     stridex,
                                                             const hipComplex* beta,
                                                             hipComplex*       y,
                                                             int               incy,
                                                             hipblasStride     stridey,
                                                             int               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsymvStridedBatched_v2(hipblasHandle_t         handle,
                                                             hipblasFillMode_t       uplo,
                                                             int                     n,
                                                             const hipDoubleComplex* alpha,
                                                             const hipDoubleComplex* AP,
                                                             int                     lda,
                                                             hipblasStride           strideA,
                                                             const hipDoubleComplex* x,
                                                             int                     incx,
                                                             hipblasStride           stridex,
                                                             const hipDoubleComplex* beta,
                                                             hipDoubleComplex*       y,
                                                             int                     incy,
                                                             hipblasStride           stridey,
                                                             int                     batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSsymvStridedBatched_64(hipblasHandle_t   handle,
                                                             hipblasFillMode_t uplo,
                                                             int64_t           n,
                                                             const float*      alpha,
                                                             const float*      AP,
                                                             int64_t           lda,
                                                             hipblasStride     strideA,
                                                             const float*      x,
                                                             int64_t           incx,
                                                             hipblasStride     stridex,
                                                             const float*      beta,
                                                             float*            y,
                                                             int64_t           incy,
                                                             hipblasStride     stridey,
                                                             int64_t           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDsymvStridedBatched_64(hipblasHandle_t   handle,
                                                             hipblasFillMode_t uplo,
                                                             int64_t           n,
                                                             const double*     alpha,
                                                             const double*     AP,
                                                             int64_t           lda,
                                                             hipblasStride     strideA,
                                                             const double*     x,
                                                             int64_t           incx,
                                                             hipblasStride     stridex,
                                                             const double*     beta,
                                                             double*           y,
                                                             int64_t           incy,
                                                             hipblasStride     stridey,
                                                             int64_t           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsymvStridedBatched_64(hipblasHandle_t       handle,
                                                             hipblasFillMode_t     uplo,
                                                             int64_t               n,
                                                             const hipblasComplex* alpha,
                                                             const hipblasComplex* AP,
                                                             int64_t               lda,
                                                             hipblasStride         strideA,
                                                             const hipblasComplex* x,
                                                             int64_t               incx,
                                                             hipblasStride         stridex,
                                                             const hipblasComplex* beta,
                                                             hipblasComplex*       y,
                                                             int64_t               incy,
                                                             hipblasStride         stridey,
                                                             int64_t               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsymvStridedBatched_64(hipblasHandle_t             handle,
                                                             hipblasFillMode_t           uplo,
                                                             int64_t                     n,
                                                             const hipblasDoubleComplex* alpha,
                                                             const hipblasDoubleComplex* AP,
                                                             int64_t                     lda,
                                                             hipblasStride               strideA,
                                                             const hipblasDoubleComplex* x,
                                                             int64_t                     incx,
                                                             hipblasStride               stridex,
                                                             const hipblasDoubleComplex* beta,
                                                             hipblasDoubleComplex*       y,
                                                             int64_t                     incy,
                                                             hipblasStride               stridey,
                                                             int64_t batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsymvStridedBatched_v2_64(hipblasHandle_t   handle,
                                                                hipblasFillMode_t uplo,
                                                                int64_t           n,
                                                                const hipComplex* alpha,
                                                                const hipComplex* AP,
                                                                int64_t           lda,
                                                                hipblasStride     strideA,
                                                                const hipComplex* x,
                                                                int64_t           incx,
                                                                hipblasStride     stridex,
                                                                const hipComplex* beta,
                                                                hipComplex*       y,
                                                                int64_t           incy,
                                                                hipblasStride     stridey,
                                                                int64_t           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsymvStridedBatched_v2_64(hipblasHandle_t         handle,
                                                                hipblasFillMode_t       uplo,
                                                                int64_t                 n,
                                                                const hipDoubleComplex* alpha,
                                                                const hipDoubleComplex* AP,
                                                                int64_t                 lda,
                                                                hipblasStride           strideA,
                                                                const hipDoubleComplex* x,
                                                                int64_t                 incx,
                                                                hipblasStride           stridex,
                                                                const hipDoubleComplex* beta,
                                                                hipDoubleComplex*       y,
                                                                int64_t                 incy,
                                                                hipblasStride           stridey,
                                                                int64_t                 batchCount);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    syr performs the matrix-vector operations

        A := A + alpha*x*x**T

    where alpha is a scalar, x is a vector, and A is an
    n by n symmetric matrix.

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : s,d,c,z

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
              if HIPBLAS_FILL_MODE_UPPER, the lower part of A is not referenced
              if HIPBLAS_FILL_MODE_LOWER, the upper part of A is not referenced

    @param[in]
    n         [int]
              the number of rows and columns of matrix A.
    @param[in]
    alpha
              device pointer or host pointer to scalar alpha.
    @param[in]
    x         device pointer storing vector x.
    @param[in]
    incx      [int]
              specifies the increment for the elements of x.
    @param[inout]
    AP         device pointer storing matrix A.
    @param[in]
    lda       [int]
              specifies the leading dimension of A.

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSsyr(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           const float*      alpha,
                                           const float*      x,
                                           int               incx,
                                           float*            AP,
                                           int               lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasDsyr(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           const double*     alpha,
                                           const double*     x,
                                           int               incx,
                                           double*           AP,
                                           int               lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsyr(hipblasHandle_t       handle,
                                           hipblasFillMode_t     uplo,
                                           int                   n,
                                           const hipblasComplex* alpha,
                                           const hipblasComplex* x,
                                           int                   incx,
                                           hipblasComplex*       AP,
                                           int                   lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsyr(hipblasHandle_t             handle,
                                           hipblasFillMode_t           uplo,
                                           int                         n,
                                           const hipblasDoubleComplex* alpha,
                                           const hipblasDoubleComplex* x,
                                           int                         incx,
                                           hipblasDoubleComplex*       AP,
                                           int                         lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsyr_v2(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int               n,
                                              const hipComplex* alpha,
                                              const hipComplex* x,
                                              int               incx,
                                              hipComplex*       AP,
                                              int               lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsyr_v2(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              int                     n,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* x,
                                              int                     incx,
                                              hipDoubleComplex*       AP,
                                              int                     lda);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSsyr_64(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int64_t           n,
                                              const float*      alpha,
                                              const float*      x,
                                              int64_t           incx,
                                              float*            AP,
                                              int64_t           lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasDsyr_64(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int64_t           n,
                                              const double*     alpha,
                                              const double*     x,
                                              int64_t           incx,
                                              double*           AP,
                                              int64_t           lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsyr_64(hipblasHandle_t       handle,
                                              hipblasFillMode_t     uplo,
                                              int64_t               n,
                                              const hipblasComplex* alpha,
                                              const hipblasComplex* x,
                                              int64_t               incx,
                                              hipblasComplex*       AP,
                                              int64_t               lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsyr_64(hipblasHandle_t             handle,
                                              hipblasFillMode_t           uplo,
                                              int64_t                     n,
                                              const hipblasDoubleComplex* alpha,
                                              const hipblasDoubleComplex* x,
                                              int64_t                     incx,
                                              hipblasDoubleComplex*       AP,
                                              int64_t                     lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsyr_v2_64(hipblasHandle_t   handle,
                                                 hipblasFillMode_t uplo,
                                                 int64_t           n,
                                                 const hipComplex* alpha,
                                                 const hipComplex* x,
                                                 int64_t           incx,
                                                 hipComplex*       AP,
                                                 int64_t           lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsyr_v2_64(hipblasHandle_t         handle,
                                                 hipblasFillMode_t       uplo,
                                                 int64_t                 n,
                                                 const hipDoubleComplex* alpha,
                                                 const hipDoubleComplex* x,
                                                 int64_t                 incx,
                                                 hipDoubleComplex*       AP,
                                                 int64_t                 lda);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    syrBatched performs a batch of matrix-vector operations

        A[i] := A[i] + alpha*x[i]*x[i]**T

    where alpha is a scalar, x is an array of vectors, and A is an array of
    n by n symmetric matrices, for i = 1 , ... , batchCount.

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
              if HIPBLAS_FILL_MODE_UPPER, the lower part of A is not referenced
              if HIPBLAS_FILL_MODE_LOWER, the upper part of A is not referenced
    @param[in]
    n         [int]
              the number of rows and columns of matrix A.
    @param[in]
    alpha
              device pointer or host pointer to scalar alpha.
    @param[in]
    x         device array of device pointers storing each vector x_i.
    @param[in]
    incx      [int]
              specifies the increment for the elements of each x_i.
    @param[inout]
    AP         device array of device pointers storing each matrix A_i.
    @param[in]
    lda       [int]
              specifies the leading dimension of each A_i.
    @param[in]
    batchCount [int]
                number of instances in the batch

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSsyrBatched(hipblasHandle_t    handle,
                                                  hipblasFillMode_t  uplo,
                                                  int                n,
                                                  const float*       alpha,
                                                  const float* const x[],
                                                  int                incx,
                                                  float* const       AP[],
                                                  int                lda,
                                                  int                batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDsyrBatched(hipblasHandle_t     handle,
                                                  hipblasFillMode_t   uplo,
                                                  int                 n,
                                                  const double*       alpha,
                                                  const double* const x[],
                                                  int                 incx,
                                                  double* const       AP[],
                                                  int                 lda,
                                                  int                 batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsyrBatched(hipblasHandle_t             handle,
                                                  hipblasFillMode_t           uplo,
                                                  int                         n,
                                                  const hipblasComplex*       alpha,
                                                  const hipblasComplex* const x[],
                                                  int                         incx,
                                                  hipblasComplex* const       AP[],
                                                  int                         lda,
                                                  int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsyrBatched(hipblasHandle_t                   handle,
                                                  hipblasFillMode_t                 uplo,
                                                  int                               n,
                                                  const hipblasDoubleComplex*       alpha,
                                                  const hipblasDoubleComplex* const x[],
                                                  int                               incx,
                                                  hipblasDoubleComplex* const       AP[],
                                                  int                               lda,
                                                  int                               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsyrBatched_v2(hipblasHandle_t         handle,
                                                     hipblasFillMode_t       uplo,
                                                     int                     n,
                                                     const hipComplex*       alpha,
                                                     const hipComplex* const x[],
                                                     int                     incx,
                                                     hipComplex* const       AP[],
                                                     int                     lda,
                                                     int                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsyrBatched_v2(hipblasHandle_t               handle,
                                                     hipblasFillMode_t             uplo,
                                                     int                           n,
                                                     const hipDoubleComplex*       alpha,
                                                     const hipDoubleComplex* const x[],
                                                     int                           incx,
                                                     hipDoubleComplex* const       AP[],
                                                     int                           lda,
                                                     int                           batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSsyrBatched_64(hipblasHandle_t    handle,
                                                     hipblasFillMode_t  uplo,
                                                     int64_t            n,
                                                     const float*       alpha,
                                                     const float* const x[],
                                                     int64_t            incx,
                                                     float* const       AP[],
                                                     int64_t            lda,
                                                     int64_t            batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDsyrBatched_64(hipblasHandle_t     handle,
                                                     hipblasFillMode_t   uplo,
                                                     int64_t             n,
                                                     const double*       alpha,
                                                     const double* const x[],
                                                     int64_t             incx,
                                                     double* const       AP[],
                                                     int64_t             lda,
                                                     int64_t             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsyrBatched_64(hipblasHandle_t             handle,
                                                     hipblasFillMode_t           uplo,
                                                     int64_t                     n,
                                                     const hipblasComplex*       alpha,
                                                     const hipblasComplex* const x[],
                                                     int64_t                     incx,
                                                     hipblasComplex* const       AP[],
                                                     int64_t                     lda,
                                                     int64_t                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsyrBatched_64(hipblasHandle_t                   handle,
                                                     hipblasFillMode_t                 uplo,
                                                     int64_t                           n,
                                                     const hipblasDoubleComplex*       alpha,
                                                     const hipblasDoubleComplex* const x[],
                                                     int64_t                           incx,
                                                     hipblasDoubleComplex* const       AP[],
                                                     int64_t                           lda,
                                                     int64_t                           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsyrBatched_v2_64(hipblasHandle_t         handle,
                                                        hipblasFillMode_t       uplo,
                                                        int64_t                 n,
                                                        const hipComplex*       alpha,
                                                        const hipComplex* const x[],
                                                        int64_t                 incx,
                                                        hipComplex* const       AP[],
                                                        int64_t                 lda,
                                                        int64_t                 batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsyrBatched_v2_64(hipblasHandle_t               handle,
                                                        hipblasFillMode_t             uplo,
                                                        int64_t                       n,
                                                        const hipDoubleComplex*       alpha,
                                                        const hipDoubleComplex* const x[],
                                                        int64_t                       incx,
                                                        hipDoubleComplex* const       AP[],
                                                        int64_t                       lda,
                                                        int64_t                       batchCount);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    syrStridedBatched performs the matrix-vector operations

        A[i] := A[i] + alpha*x[i]*x[i]**T

    where alpha is a scalar, vectors, and A is an array of
    n by n symmetric matrices, for i = 1 , ... , batchCount.

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
              if HIPBLAS_FILL_MODE_UPPER, the lower part of A is not referenced
              if HIPBLAS_FILL_MODE_LOWER, the upper part of A is not referenced
    @param[in]
    n         [int]
              the number of rows and columns of each matrix A.
    @param[in]
    alpha
              device pointer or host pointer to scalar alpha.
    @param[in]
    x         device pointer to the first vector x_1.
    @param[in]
    incx      [int]
              specifies the increment for the elements of each x_i.
    @param[in]
    stridex   [hipblasStride]
              specifies the pointer increment between vectors (x_i) and (x_i+1).
    @param[inout]
    AP         device pointer to the first matrix A_1.
    @param[in]
    lda       [int]
              specifies the leading dimension of each A_i.
    @param[in]
    strideA   [hipblasStride]
              stride from the start of one matrix (A_i) and the next one (A_i+1)
    @param[in]
    batchCount [int]
              number of instances in the batch

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSsyrStridedBatched(hipblasHandle_t   handle,
                                                         hipblasFillMode_t uplo,
                                                         int               n,
                                                         const float*      alpha,
                                                         const float*      x,
                                                         int               incx,
                                                         hipblasStride     stridex,
                                                         float*            AP,
                                                         int               lda,
                                                         hipblasStride     strideA,
                                                         int               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDsyrStridedBatched(hipblasHandle_t   handle,
                                                         hipblasFillMode_t uplo,
                                                         int               n,
                                                         const double*     alpha,
                                                         const double*     x,
                                                         int               incx,
                                                         hipblasStride     stridex,
                                                         double*           AP,
                                                         int               lda,
                                                         hipblasStride     strideA,
                                                         int               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsyrStridedBatched(hipblasHandle_t       handle,
                                                         hipblasFillMode_t     uplo,
                                                         int                   n,
                                                         const hipblasComplex* alpha,
                                                         const hipblasComplex* x,
                                                         int                   incx,
                                                         hipblasStride         stridex,
                                                         hipblasComplex*       AP,
                                                         int                   lda,
                                                         hipblasStride         strideA,
                                                         int                   batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsyrStridedBatched(hipblasHandle_t             handle,
                                                         hipblasFillMode_t           uplo,
                                                         int                         n,
                                                         const hipblasDoubleComplex* alpha,
                                                         const hipblasDoubleComplex* x,
                                                         int                         incx,
                                                         hipblasStride               stridex,
                                                         hipblasDoubleComplex*       AP,
                                                         int                         lda,
                                                         hipblasStride               strideA,
                                                         int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsyrStridedBatched_v2(hipblasHandle_t   handle,
                                                            hipblasFillMode_t uplo,
                                                            int               n,
                                                            const hipComplex* alpha,
                                                            const hipComplex* x,
                                                            int               incx,
                                                            hipblasStride     stridex,
                                                            hipComplex*       AP,
                                                            int               lda,
                                                            hipblasStride     strideA,
                                                            int               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsyrStridedBatched_v2(hipblasHandle_t         handle,
                                                            hipblasFillMode_t       uplo,
                                                            int                     n,
                                                            const hipDoubleComplex* alpha,
                                                            const hipDoubleComplex* x,
                                                            int                     incx,
                                                            hipblasStride           stridex,
                                                            hipDoubleComplex*       AP,
                                                            int                     lda,
                                                            hipblasStride           strideA,
                                                            int                     batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSsyrStridedBatched_64(hipblasHandle_t   handle,
                                                            hipblasFillMode_t uplo,
                                                            int64_t           n,
                                                            const float*      alpha,
                                                            const float*      x,
                                                            int64_t           incx,
                                                            hipblasStride     stridex,
                                                            float*            AP,
                                                            int64_t           lda,
                                                            hipblasStride     strideA,
                                                            int64_t           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDsyrStridedBatched_64(hipblasHandle_t   handle,
                                                            hipblasFillMode_t uplo,
                                                            int64_t           n,
                                                            const double*     alpha,
                                                            const double*     x,
                                                            int64_t           incx,
                                                            hipblasStride     stridex,
                                                            double*           AP,
                                                            int64_t           lda,
                                                            hipblasStride     strideA,
                                                            int64_t           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsyrStridedBatched_64(hipblasHandle_t       handle,
                                                            hipblasFillMode_t     uplo,
                                                            int64_t               n,
                                                            const hipblasComplex* alpha,
                                                            const hipblasComplex* x,
                                                            int64_t               incx,
                                                            hipblasStride         stridex,
                                                            hipblasComplex*       AP,
                                                            int64_t               lda,
                                                            hipblasStride         strideA,
                                                            int64_t               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsyrStridedBatched_64(hipblasHandle_t             handle,
                                                            hipblasFillMode_t           uplo,
                                                            int64_t                     n,
                                                            const hipblasDoubleComplex* alpha,
                                                            const hipblasDoubleComplex* x,
                                                            int64_t                     incx,
                                                            hipblasStride               stridex,
                                                            hipblasDoubleComplex*       AP,
                                                            int64_t                     lda,
                                                            hipblasStride               strideA,
                                                            int64_t                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsyrStridedBatched_v2_64(hipblasHandle_t   handle,
                                                               hipblasFillMode_t uplo,
                                                               int64_t           n,
                                                               const hipComplex* alpha,
                                                               const hipComplex* x,
                                                               int64_t           incx,
                                                               hipblasStride     stridex,
                                                               hipComplex*       AP,
                                                               int64_t           lda,
                                                               hipblasStride     strideA,
                                                               int64_t           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsyrStridedBatched_v2_64(hipblasHandle_t         handle,
                                                               hipblasFillMode_t       uplo,
                                                               int64_t                 n,
                                                               const hipDoubleComplex* alpha,
                                                               const hipDoubleComplex* x,
                                                               int64_t                 incx,
                                                               hipblasStride           stridex,
                                                               hipDoubleComplex*       AP,
                                                               int64_t                 lda,
                                                               hipblasStride           strideA,
                                                               int64_t                 batchCount);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    syr2 performs the matrix-vector operations

        A := A + alpha*x*y**T + alpha*y*x**T

    where alpha is a scalar, x and y are vectors, and A is an
    n by n symmetric matrix.

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
              if HIPBLAS_FILL_MODE_UPPER, the lower part of A is not referenced
              if HIPBLAS_FILL_MODE_LOWER, the upper part of A is not referenced

    @param[in]
    n         [int]
              the number of rows and columns of matrix A.
    @param[in]
    alpha
              device pointer or host pointer to scalar alpha.
    @param[in]
    x         device pointer storing vector x.
    @param[in]
    incx      [int]
              specifies the increment for the elements of x.
    @param[in]
    y         device pointer storing vector y.
    @param[in]
    incy      [int]
              specifies the increment for the elements of y.
    @param[inout]
    AP         device pointer storing matrix A.
    @param[in]
    lda       [int]
              specifies the leading dimension of A.

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSsyr2(hipblasHandle_t   handle,
                                            hipblasFillMode_t uplo,
                                            int               n,
                                            const float*      alpha,
                                            const float*      x,
                                            int               incx,
                                            const float*      y,
                                            int               incy,
                                            float*            AP,
                                            int               lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasDsyr2(hipblasHandle_t   handle,
                                            hipblasFillMode_t uplo,
                                            int               n,
                                            const double*     alpha,
                                            const double*     x,
                                            int               incx,
                                            const double*     y,
                                            int               incy,
                                            double*           AP,
                                            int               lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsyr2(hipblasHandle_t       handle,
                                            hipblasFillMode_t     uplo,
                                            int                   n,
                                            const hipblasComplex* alpha,
                                            const hipblasComplex* x,
                                            int                   incx,
                                            const hipblasComplex* y,
                                            int                   incy,
                                            hipblasComplex*       AP,
                                            int                   lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsyr2(hipblasHandle_t             handle,
                                            hipblasFillMode_t           uplo,
                                            int                         n,
                                            const hipblasDoubleComplex* alpha,
                                            const hipblasDoubleComplex* x,
                                            int                         incx,
                                            const hipblasDoubleComplex* y,
                                            int                         incy,
                                            hipblasDoubleComplex*       AP,
                                            int                         lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsyr2_v2(hipblasHandle_t   handle,
                                               hipblasFillMode_t uplo,
                                               int               n,
                                               const hipComplex* alpha,
                                               const hipComplex* x,
                                               int               incx,
                                               const hipComplex* y,
                                               int               incy,
                                               hipComplex*       AP,
                                               int               lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsyr2_v2(hipblasHandle_t         handle,
                                               hipblasFillMode_t       uplo,
                                               int                     n,
                                               const hipDoubleComplex* alpha,
                                               const hipDoubleComplex* x,
                                               int                     incx,
                                               const hipDoubleComplex* y,
                                               int                     incy,
                                               hipDoubleComplex*       AP,
                                               int                     lda);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSsyr2_64(hipblasHandle_t   handle,
                                               hipblasFillMode_t uplo,
                                               int64_t           n,
                                               const float*      alpha,
                                               const float*      x,
                                               int64_t           incx,
                                               const float*      y,
                                               int64_t           incy,
                                               float*            AP,
                                               int64_t           lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasDsyr2_64(hipblasHandle_t   handle,
                                               hipblasFillMode_t uplo,
                                               int64_t           n,
                                               const double*     alpha,
                                               const double*     x,
                                               int64_t           incx,
                                               const double*     y,
                                               int64_t           incy,
                                               double*           AP,
                                               int64_t           lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsyr2_64(hipblasHandle_t       handle,
                                               hipblasFillMode_t     uplo,
                                               int64_t               n,
                                               const hipblasComplex* alpha,
                                               const hipblasComplex* x,
                                               int64_t               incx,
                                               const hipblasComplex* y,
                                               int64_t               incy,
                                               hipblasComplex*       AP,
                                               int64_t               lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsyr2_64(hipblasHandle_t             handle,
                                               hipblasFillMode_t           uplo,
                                               int64_t                     n,
                                               const hipblasDoubleComplex* alpha,
                                               const hipblasDoubleComplex* x,
                                               int64_t                     incx,
                                               const hipblasDoubleComplex* y,
                                               int64_t                     incy,
                                               hipblasDoubleComplex*       AP,
                                               int64_t                     lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsyr2_v2_64(hipblasHandle_t   handle,
                                                  hipblasFillMode_t uplo,
                                                  int64_t           n,
                                                  const hipComplex* alpha,
                                                  const hipComplex* x,
                                                  int64_t           incx,
                                                  const hipComplex* y,
                                                  int64_t           incy,
                                                  hipComplex*       AP,
                                                  int64_t           lda);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsyr2_v2_64(hipblasHandle_t         handle,
                                                  hipblasFillMode_t       uplo,
                                                  int64_t                 n,
                                                  const hipDoubleComplex* alpha,
                                                  const hipDoubleComplex* x,
                                                  int64_t                 incx,
                                                  const hipDoubleComplex* y,
                                                  int64_t                 incy,
                                                  hipDoubleComplex*       AP,
                                                  int64_t                 lda);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    syr2Batched performs a batch of matrix-vector operations

        A[i] := A[i] + alpha*x[i]*y[i]**T + alpha*y[i]*x[i]**T

    where alpha is a scalar, x[i] and y[i] are vectors, and A[i] is a
    n by n symmetric matrix, for i = 1 , ... , batchCount.

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
              if HIPBLAS_FILL_MODE_UPPER, the lower part of A is not referenced
              if HIPBLAS_FILL_MODE_LOWER, the upper part of A is not referenced
    @param[in]
    n         [int]
              the number of rows and columns of matrix A.
    @param[in]
    alpha
              device pointer or host pointer to scalar alpha.
    @param[in]
    x         device array of device pointers storing each vector x_i.
    @param[in]
    incx      [int]
              specifies the increment for the elements of each x_i.
    @param[in]
    y         device array of device pointers storing each vector y_i.
    @param[in]
    incy      [int]
              specifies the increment for the elements of each y_i.
    @param[inout]
    AP         device array of device pointers storing each matrix A_i.
    @param[in]
    lda       [int]
              specifies the leading dimension of each A_i.
    @param[in]
    batchCount [int]
                number of instances in the batch

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSsyr2Batched(hipblasHandle_t    handle,
                                                   hipblasFillMode_t  uplo,
                                                   int                n,
                                                   const float*       alpha,
                                                   const float* const x[],
                                                   int                incx,
                                                   const float* const y[],
                                                   int                incy,
                                                   float* const       AP[],
                                                   int                lda,
                                                   int                batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDsyr2Batched(hipblasHandle_t     handle,
                                                   hipblasFillMode_t   uplo,
                                                   int                 n,
                                                   const double*       alpha,
                                                   const double* const x[],
                                                   int                 incx,
                                                   const double* const y[],
                                                   int                 incy,
                                                   double* const       AP[],
                                                   int                 lda,
                                                   int                 batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsyr2Batched(hipblasHandle_t             handle,
                                                   hipblasFillMode_t           uplo,
                                                   int                         n,
                                                   const hipblasComplex*       alpha,
                                                   const hipblasComplex* const x[],
                                                   int                         incx,
                                                   const hipblasComplex* const y[],
                                                   int                         incy,
                                                   hipblasComplex* const       AP[],
                                                   int                         lda,
                                                   int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsyr2Batched(hipblasHandle_t                   handle,
                                                   hipblasFillMode_t                 uplo,
                                                   int                               n,
                                                   const hipblasDoubleComplex*       alpha,
                                                   const hipblasDoubleComplex* const x[],
                                                   int                               incx,
                                                   const hipblasDoubleComplex* const y[],
                                                   int                               incy,
                                                   hipblasDoubleComplex* const       AP[],
                                                   int                               lda,
                                                   int                               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsyr2Batched_v2(hipblasHandle_t         handle,
                                                      hipblasFillMode_t       uplo,
                                                      int                     n,
                                                      const hipComplex*       alpha,
                                                      const hipComplex* const x[],
                                                      int                     incx,
                                                      const hipComplex* const y[],
                                                      int                     incy,
                                                      hipComplex* const       AP[],
                                                      int                     lda,
                                                      int                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsyr2Batched_v2(hipblasHandle_t               handle,
                                                      hipblasFillMode_t             uplo,
                                                      int                           n,
                                                      const hipDoubleComplex*       alpha,
                                                      const hipDoubleComplex* const x[],
                                                      int                           incx,
                                                      const hipDoubleComplex* const y[],
                                                      int                           incy,
                                                      hipDoubleComplex* const       AP[],
                                                      int                           lda,
                                                      int                           batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSsyr2Batched_64(hipblasHandle_t    handle,
                                                      hipblasFillMode_t  uplo,
                                                      int64_t            n,
                                                      const float*       alpha,
                                                      const float* const x[],
                                                      int64_t            incx,
                                                      const float* const y[],
                                                      int64_t            incy,
                                                      float* const       AP[],
                                                      int64_t            lda,
                                                      int64_t            batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDsyr2Batched_64(hipblasHandle_t     handle,
                                                      hipblasFillMode_t   uplo,
                                                      int64_t             n,
                                                      const double*       alpha,
                                                      const double* const x[],
                                                      int64_t             incx,
                                                      const double* const y[],
                                                      int64_t             incy,
                                                      double* const       AP[],
                                                      int64_t             lda,
                                                      int64_t             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsyr2Batched_64(hipblasHandle_t             handle,
                                                      hipblasFillMode_t           uplo,
                                                      int64_t                     n,
                                                      const hipblasComplex*       alpha,
                                                      const hipblasComplex* const x[],
                                                      int64_t                     incx,
                                                      const hipblasComplex* const y[],
                                                      int64_t                     incy,
                                                      hipblasComplex* const       AP[],
                                                      int64_t                     lda,
                                                      int64_t                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsyr2Batched_64(hipblasHandle_t                   handle,
                                                      hipblasFillMode_t                 uplo,
                                                      int64_t                           n,
                                                      const hipblasDoubleComplex*       alpha,
                                                      const hipblasDoubleComplex* const x[],
                                                      int64_t                           incx,
                                                      const hipblasDoubleComplex* const y[],
                                                      int64_t                           incy,
                                                      hipblasDoubleComplex* const       AP[],
                                                      int64_t                           lda,
                                                      int64_t                           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsyr2Batched_v2_64(hipblasHandle_t         handle,
                                                         hipblasFillMode_t       uplo,
                                                         int64_t                 n,
                                                         const hipComplex*       alpha,
                                                         const hipComplex* const x[],
                                                         int64_t                 incx,
                                                         const hipComplex* const y[],
                                                         int64_t                 incy,
                                                         hipComplex* const       AP[],
                                                         int64_t                 lda,
                                                         int64_t                 batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsyr2Batched_v2_64(hipblasHandle_t               handle,
                                                         hipblasFillMode_t             uplo,
                                                         int64_t                       n,
                                                         const hipDoubleComplex*       alpha,
                                                         const hipDoubleComplex* const x[],
                                                         int64_t                       incx,
                                                         const hipDoubleComplex* const y[],
                                                         int64_t                       incy,
                                                         hipDoubleComplex* const       AP[],
                                                         int64_t                       lda,
                                                         int64_t                       batchCount);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    syr2StridedBatched the matrix-vector operations

        A[i] := A[i] + alpha*x[i]*y[i]**T + alpha*y[i]*x[i]**T

    where alpha is a scalar, x[i] and y[i] are vectors, and A[i] is a
    n by n symmetric matrices, for i = 1 , ... , batchCount.

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              specifies whether the upper 'HIPBLAS_FILL_MODE_UPPER' or lower 'HIPBLAS_FILL_MODE_LOWER'
              if HIPBLAS_FILL_MODE_UPPER, the lower part of A is not referenced
              if HIPBLAS_FILL_MODE_LOWER, the upper part of A is not referenced
    @param[in]
    n         [int]
              the number of rows and columns of each matrix A.
    @param[in]
    alpha
              device pointer or host pointer to scalar alpha.
    @param[in]
    x         device pointer to the first vector x_1.
    @param[in]
    incx      [int]
              specifies the increment for the elements of each x_i.
    @param[in]
    stridex   [hipblasStride]
              specifies the pointer increment between vectors (x_i) and (x_i+1).
    @param[in]
    y         device pointer to the first vector y_1.
    @param[in]
    incy      [int]
              specifies the increment for the elements of each y_i.
    @param[in]
    stridey   [hipblasStride]
              specifies the pointer increment between vectors (y_i) and (y_i+1).
    @param[inout]
    AP         device pointer to the first matrix A_1.
    @param[in]
    lda       [int]
              specifies the leading dimension of each A_i.
    @param[in]
    strideA   [hipblasStride]
              stride from the start of one matrix (A_i) and the next one (A_i+1)
    @param[in]
    batchCount [int]
              number of instances in the batch

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasSsyr2StridedBatched(hipblasHandle_t   handle,
                                                          hipblasFillMode_t uplo,
                                                          int               n,
                                                          const float*      alpha,
                                                          const float*      x,
                                                          int               incx,
                                                          hipblasStride     stridex,
                                                          const float*      y,
                                                          int               incy,
                                                          hipblasStride     stridey,
                                                          float*            AP,
                                                          int               lda,
                                                          hipblasStride     strideA,
                                                          int               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDsyr2StridedBatched(hipblasHandle_t   handle,
                                                          hipblasFillMode_t uplo,
                                                          int               n,
                                                          const double*     alpha,
                                                          const double*     x,
                                                          int               incx,
                                                          hipblasStride     stridex,
                                                          const double*     y,
                                                          int               incy,
                                                          hipblasStride     stridey,
                                                          double*           AP,
                                                          int               lda,
                                                          hipblasStride     strideA,
                                                          int               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsyr2StridedBatched(hipblasHandle_t       handle,
                                                          hipblasFillMode_t     uplo,
                                                          int                   n,
                                                          const hipblasComplex* alpha,
                                                          const hipblasComplex* x,
                                                          int                   incx,
                                                          hipblasStride         stridex,
                                                          const hipblasComplex* y,
                                                          int                   incy,
                                                          hipblasStride         stridey,
                                                          hipblasComplex*       AP,
                                                          int                   lda,
                                                          hipblasStride         strideA,
                                                          int                   batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsyr2StridedBatched(hipblasHandle_t             handle,
                                                          hipblasFillMode_t           uplo,
                                                          int                         n,
                                                          const hipblasDoubleComplex* alpha,
                                                          const hipblasDoubleComplex* x,
                                                          int                         incx,
                                                          hipblasStride               stridex,
                                                          const hipblasDoubleComplex* y,
                                                          int                         incy,
                                                          hipblasStride               stridey,
                                                          hipblasDoubleComplex*       AP,
                                                          int                         lda,
                                                          hipblasStride               strideA,
                                                          int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsyr2StridedBatched_v2(hipblasHandle_t   handle,
                                                             hipblasFillMode_t uplo,
                                                             int               n,
                                                             const hipComplex* alpha,
                                                             const hipComplex* x,
                                                             int               incx,
                                                             hipblasStride     stridex,
                                                             const hipComplex* y,
                                                             int               incy,
                                                             hipblasStride     stridey,
                                                             hipComplex*       AP,
                                                             int               lda,
                                                             hipblasStride     strideA,
                                                             int               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsyr2StridedBatched_v2(hipblasHandle_t         handle,
                                                             hipblasFillMode_t       uplo,
                                                             int                     n,
                                                             const hipDoubleComplex* alpha,
                                                             const hipDoubleComplex* x,
                                                             int                     incx,
                                                             hipblasStride           stridex,
                                                             const hipDoubleComplex* y,
                                                             int                     incy,
                                                             hipblasStride           stridey,
                                                             hipDoubleComplex*       AP,
                                                             int                     lda,
                                                             hipblasStride           strideA,
                                                             int                     batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasSsyr2StridedBatched_64(hipblasHandle_t   handle,
                                                             hipblasFillMode_t uplo,
                                                             int64_t           n,
                                                             const float*      alpha,
                                                             const float*      x,
                                                             int64_t           incx,
                                                             hipblasStride     stridex,
                                                             const float*      y,
                                                             int64_t           incy,
                                                             hipblasStride     stridey,
                                                             float*            AP,
                                                             int64_t           lda,
                                                             hipblasStride     strideA,
                                                             int64_t           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDsyr2StridedBatched_64(hipblasHandle_t   handle,
                                                             hipblasFillMode_t uplo,
                                                             int64_t           n,
                                                             const double*     alpha,
                                                             const double*     x,
                                                             int64_t           incx,
                                                             hipblasStride     stridex,
                                                             const double*     y,
                                                             int64_t           incy,
                                                             hipblasStride     stridey,
                                                             double*           AP,
                                                             int64_t           lda,
                                                             hipblasStride     strideA,
                                                             int64_t           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsyr2StridedBatched_64(hipblasHandle_t       handle,
                                                             hipblasFillMode_t     uplo,
                                                             int64_t               n,
                                                             const hipblasComplex* alpha,
                                                             const hipblasComplex* x,
                                                             int64_t               incx,
                                                             hipblasStride         stridex,
                                                             const hipblasComplex* y,
                                                             int64_t               incy,
                                                             hipblasStride         stridey,
                                                             hipblasComplex*       AP,
                                                             int64_t               lda,
                                                             hipblasStride         strideA,
                                                             int64_t               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsyr2StridedBatched_64(hipblasHandle_t             handle,
                                                             hipblasFillMode_t           uplo,
                                                             int64_t                     n,
                                                             const hipblasDoubleComplex* alpha,
                                                             const hipblasDoubleComplex* x,
                                                             int64_t                     incx,
                                                             hipblasStride               stridex,
                                                             const hipblasDoubleComplex* y,
                                                             int64_t                     incy,
                                                             hipblasStride               stridey,
                                                             hipblasDoubleComplex*       AP,
                                                             int64_t                     lda,
                                                             hipblasStride               strideA,
                                                             int64_t batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCsyr2StridedBatched_v2_64(hipblasHandle_t   handle,
                                                                hipblasFillMode_t uplo,
                                                                int64_t           n,
                                                                const hipComplex* alpha,
                                                                const hipComplex* x,
                                                                int64_t           incx,
                                                                hipblasStride     stridex,
                                                                const hipComplex* y,
                                                                int64_t           incy,
                                                                hipblasStride     stridey,
                                                                hipComplex*       AP,
                                                                int64_t           lda,
                                                                hipblasStride     strideA,
                                                                int64_t           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZsyr2StridedBatched_v2_64(hipblasHandle_t         handle,
                                                                hipblasFillMode_t       uplo,
                                                                int64_t                 n,
                                                                const hipDoubleComplex* alpha,
                                                                const hipDoubleComplex* x,
                                                                int64_t                 incx,
                                                                hipblasStride           stridex,
                                                                const hipDoubleComplex* y,
                                                                int64_t                 incy,
                                                                hipblasStride           stridey,
                                                                hipDoubleComplex*       AP,
                                                                int64_t                 lda,
                                                                hipblasStride           strideA,
                                                                int64_t                 batchCount);
//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    tbmv performs one of the matrix-vector operations

        x := A*x      or
        x := A**T*x   or
        x := A**H*x,

    x is a vectors and A is a banded n by n matrix (see description below).

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : s,d,c,z

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              HIPBLAS_FILL_MODE_UPPER: A is an upper banded triangular matrix.
              HIPBLAS_FILL_MODE_LOWER: A is a  lower banded triangular matrix.
    @param[in]
    transA     [hipblasOperation_t]
              indicates whether matrix A is tranposed (conjugated) or not.
    @param[in]
    diag      [hipblasDiagType_t]
              HIPBLAS_DIAG_UNIT: The main diagonal of A is assumed to consist of only
                                     1's and is not referenced.
              HIPBLAS_DIAG_NON_UNIT: No assumptions are made of A's main diagonal.
    @param[in]
    n         [int]
              the number of rows and columns of the matrix represented by A.
    @param[in]
    k         [int]
              if uplo == HIPBLAS_FILL_MODE_UPPER, k specifies the number of super-diagonals
              of the matrix A.
              if uplo == HIPBLAS_FILL_MODE_LOWER, k specifies the number of sub-diagonals
              of the matrix A.
              k must satisfy k > 0 && k < lda.
    @param[in]
    AP         device pointer storing banded triangular matrix A.
              if uplo == HIPBLAS_FILL_MODE_UPPER:
                The matrix represented is an upper banded triangular matrix
                with the main diagonal and k super-diagonals, everything
                else can be assumed to be 0.
                The matrix is compacted so that the main diagonal resides on the k'th
                row, the first super diagonal resides on the RHS of the k-1'th row, etc,
                with the k'th diagonal on the RHS of the 0'th row.
                   Ex: (HIPBLAS_FILL_MODE_UPPER; n = 5; k = 2)
                      1 6 9 0 0              0 0 9 8 7
                      0 2 7 8 0              0 6 7 8 9
                      0 0 3 8 7     ---->    1 2 3 4 5
                      0 0 0 4 9              0 0 0 0 0
                      0 0 0 0 5              0 0 0 0 0
              if uplo == HIPBLAS_FILL_MODE_LOWER:
                The matrix represnted is a lower banded triangular matrix
                with the main diagonal and k sub-diagonals, everything else can be
                assumed to be 0.
                The matrix is compacted so that the main diagonal resides on the 0'th row,
                working up to the k'th diagonal residing on the LHS of the k'th row.
                   Ex: (HIPBLAS_FILL_MODE_LOWER; n = 5; k = 2)
                      1 0 0 0 0              1 2 3 4 5
                      6 2 0 0 0              6 7 8 9 0
                      9 7 3 0 0     ---->    9 8 7 0 0
                      0 8 8 4 0              0 0 0 0 0
                      0 0 7 9 5              0 0 0 0 0
    @param[in]
    lda       [int]
              specifies the leading dimension of A. lda must satisfy lda > k.
    @param[inout]
    x         device pointer storing vector x.
    @param[in]
    incx      [int]
              specifies the increment for the elements of x.

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasStbmv(hipblasHandle_t    handle,
                                            hipblasFillMode_t  uplo,
                                            hipblasOperation_t transA,
                                            hipblasDiagType_t  diag,
                                            int                n,
                                            int                k,
                                            const float*       AP,
                                            int                lda,
                                            float*             x,
                                            int                incx);

HIPBLAS_EXPORT hipblasStatus_t hipblasDtbmv(hipblasHandle_t    handle,
                                            hipblasFillMode_t  uplo,
                                            hipblasOperation_t transA,
                                            hipblasDiagType_t  diag,
                                            int                n,
                                            int                k,
                                            const double*      AP,
                                            int                lda,
                                            double*            x,
                                            int                incx);

HIPBLAS_EXPORT hipblasStatus_t hipblasCtbmv(hipblasHandle_t       handle,
                                            hipblasFillMode_t     uplo,
                                            hipblasOperation_t    transA,
                                            hipblasDiagType_t     diag,
                                            int                   n,
                                            int                   k,
                                            const hipblasComplex* AP,
                                            int                   lda,
                                            hipblasComplex*       x,
                                            int                   incx);

HIPBLAS_EXPORT hipblasStatus_t hipblasZtbmv(hipblasHandle_t             handle,
                                            hipblasFillMode_t           uplo,
                                            hipblasOperation_t          transA,
                                            hipblasDiagType_t           diag,
                                            int                         n,
                                            int                         k,
                                            const hipblasDoubleComplex* AP,
                                            int                         lda,
                                            hipblasDoubleComplex*       x,
                                            int                         incx);

HIPBLAS_EXPORT hipblasStatus_t hipblasCtbmv_v2(hipblasHandle_t    handle,
                                               hipblasFillMode_t  uplo,
                                               hipblasOperation_t transA,
                                               hipblasDiagType_t  diag,
                                               int                n,
                                               int                k,
                                               const hipComplex*  AP,
                                               int                lda,
                                               hipComplex*        x,
                                               int                incx);

HIPBLAS_EXPORT hipblasStatus_t hipblasZtbmv_v2(hipblasHandle_t         handle,
                                               hipblasFillMode_t       uplo,
                                               hipblasOperation_t      transA,
                                               hipblasDiagType_t       diag,
                                               int                     n,
                                               int                     k,
                                               const hipDoubleComplex* AP,
                                               int                     lda,
                                               hipDoubleComplex*       x,
                                               int                     incx);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasStbmv_64(hipblasHandle_t    handle,
                                               hipblasFillMode_t  uplo,
                                               hipblasOperation_t transA,
                                               hipblasDiagType_t  diag,
                                               int64_t            n,
                                               int64_t            k,
                                               const float*       AP,
                                               int64_t            lda,
                                               float*             x,
                                               int64_t            incx);

HIPBLAS_EXPORT hipblasStatus_t hipblasDtbmv_64(hipblasHandle_t    handle,
                                               hipblasFillMode_t  uplo,
                                               hipblasOperation_t transA,
                                               hipblasDiagType_t  diag,
                                               int64_t            n,
                                               int64_t            k,
                                               const double*      AP,
                                               int64_t            lda,
                                               double*            x,
                                               int64_t            incx);

HIPBLAS_EXPORT hipblasStatus_t hipblasCtbmv_64(hipblasHandle_t       handle,
                                               hipblasFillMode_t     uplo,
                                               hipblasOperation_t    transA,
                                               hipblasDiagType_t     diag,
                                               int64_t               n,
                                               int64_t               k,
                                               const hipblasComplex* AP,
                                               int64_t               lda,
                                               hipblasComplex*       x,
                                               int64_t               incx);

HIPBLAS_EXPORT hipblasStatus_t hipblasZtbmv_64(hipblasHandle_t             handle,
                                               hipblasFillMode_t           uplo,
                                               hipblasOperation_t          transA,
                                               hipblasDiagType_t           diag,
                                               int64_t                     n,
                                               int64_t                     k,
                                               const hipblasDoubleComplex* AP,
                                               int64_t                     lda,
                                               hipblasDoubleComplex*       x,
                                               int64_t                     incx);

HIPBLAS_EXPORT hipblasStatus_t hipblasCtbmv_v2_64(hipblasHandle_t    handle,
                                                  hipblasFillMode_t  uplo,
                                                  hipblasOperation_t transA,
                                                  hipblasDiagType_t  diag,
                                                  int64_t            n,
                                                  int64_t            k,
                                                  const hipComplex*  AP,
                                                  int64_t            lda,
                                                  hipComplex*        x,
                                                  int64_t            incx);

HIPBLAS_EXPORT hipblasStatus_t hipblasZtbmv_v2_64(hipblasHandle_t         handle,
                                                  hipblasFillMode_t       uplo,
                                                  hipblasOperation_t      transA,
                                                  hipblasDiagType_t       diag,
                                                  int64_t                 n,
                                                  int64_t                 k,
                                                  const hipDoubleComplex* AP,
                                                  int64_t                 lda,
                                                  hipDoubleComplex*       x,
                                                  int64_t                 incx);

//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    tbmvBatched performs one of the matrix-vector operations

        x_i := A_i*x_i      or
        x_i := A_i**T*x_i   or
        x_i := A_i**H*x_i,

    where (A_i, x_i) is the i-th instance of the batch.
    x_i is a vector and A_i is an n by n matrix, for i = 1, ..., batchCount.

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              HIPBLAS_FILL_MODE_UPPER: each A_i is an upper banded triangular matrix.
              HIPBLAS_FILL_MODE_LOWER: each A_i is a  lower banded triangular matrix.
    @param[in]
    transA     [hipblasOperation_t]
              indicates whether each matrix A_i is tranposed (conjugated) or not.
    @param[in]
    diag      [hipblasDiagType_t]
              HIPBLAS_DIAG_UNIT: The main diagonal of each A_i is assumed to consist of only
                                     1's and is not referenced.
              HIPBLAS_DIAG_NON_UNIT: No assumptions are made of each A_i's main diagonal.
    @param[in]
    n         [int]
              the number of rows and columns of the matrix represented by each A_i.
    @param[in]
    k         [int]
              if uplo == HIPBLAS_FILL_MODE_UPPER, k specifies the number of super-diagonals
              of each matrix A_i.
              if uplo == HIPBLAS_FILL_MODE_LOWER, k specifies the number of sub-diagonals
              of each matrix A_i.
              k must satisfy k > 0 && k < lda.
    @param[in]
    AP         device array of device pointers storing each banded triangular matrix A_i.
              if uplo == HIPBLAS_FILL_MODE_UPPER:
                The matrix represented is an upper banded triangular matrix
                with the main diagonal and k super-diagonals, everything
                else can be assumed to be 0.
                The matrix is compacted so that the main diagonal resides on the k'th
                row, the first super diagonal resides on the RHS of the k-1'th row, etc,
                with the k'th diagonal on the RHS of the 0'th row.
                   Ex: (HIPBLAS_FILL_MODE_UPPER; n = 5; k = 2)
                      1 6 9 0 0              0 0 9 8 7
                      0 2 7 8 0              0 6 7 8 9
                      0 0 3 8 7     ---->    1 2 3 4 5
                      0 0 0 4 9              0 0 0 0 0
                      0 0 0 0 5              0 0 0 0 0
              if uplo == HIPBLAS_FILL_MODE_LOWER:
                The matrix represnted is a lower banded triangular matrix
                with the main diagonal and k sub-diagonals, everything else can be
                assumed to be 0.
                The matrix is compacted so that the main diagonal resides on the 0'th row,
                working up to the k'th diagonal residing on the LHS of the k'th row.
                   Ex: (HIPBLAS_FILL_MODE_LOWER; n = 5; k = 2)
                      1 0 0 0 0              1 2 3 4 5
                      6 2 0 0 0              6 7 8 9 0
                      9 7 3 0 0     ---->    9 8 7 0 0
                      0 8 8 4 0              0 0 0 0 0
                      0 0 7 9 5              0 0 0 0 0
    @param[in]
    lda       [int]
              specifies the leading dimension of each A_i. lda must satisfy lda > k.
    @param[inout]
    x         device array of device pointer storing each vector x_i.
    @param[in]
    incx      [int]
              specifies the increment for the elements of each x_i.
    @param[in]
    batchCount [int]
                number of instances in the batch.

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasStbmvBatched(hipblasHandle_t    handle,
                                                   hipblasFillMode_t  uplo,
                                                   hipblasOperation_t transA,
                                                   hipblasDiagType_t  diag,
                                                   int                n,
                                                   int                k,
                                                   const float* const AP[],
                                                   int                lda,
                                                   float* const       x[],
                                                   int                incx,
                                                   int                batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDtbmvBatched(hipblasHandle_t     handle,
                                                   hipblasFillMode_t   uplo,
                                                   hipblasOperation_t  transA,
                                                   hipblasDiagType_t   diag,
                                                   int                 n,
                                                   int                 k,
                                                   const double* const AP[],
                                                   int                 lda,
                                                   double* const       x[],
                                                   int                 incx,
                                                   int                 batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCtbmvBatched(hipblasHandle_t             handle,
                                                   hipblasFillMode_t           uplo,
                                                   hipblasOperation_t          transA,
                                                   hipblasDiagType_t           diag,
                                                   int                         n,
                                                   int                         k,
                                                   const hipblasComplex* const AP[],
                                                   int                         lda,
                                                   hipblasComplex* const       x[],
                                                   int                         incx,
                                                   int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZtbmvBatched(hipblasHandle_t                   handle,
                                                   hipblasFillMode_t                 uplo,
                                                   hipblasOperation_t                transA,
                                                   hipblasDiagType_t                 diag,
                                                   int                               n,
                                                   int                               k,
                                                   const hipblasDoubleComplex* const AP[],
                                                   int                               lda,
                                                   hipblasDoubleComplex* const       x[],
                                                   int                               incx,
                                                   int                               batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCtbmvBatched_v2(hipblasHandle_t         handle,
                                                      hipblasFillMode_t       uplo,
                                                      hipblasOperation_t      transA,
                                                      hipblasDiagType_t       diag,
                                                      int                     n,
                                                      int                     k,
                                                      const hipComplex* const AP[],
                                                      int                     lda,
                                                      hipComplex* const       x[],
                                                      int                     incx,
                                                      int                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZtbmvBatched_v2(hipblasHandle_t               handle,
                                                      hipblasFillMode_t             uplo,
                                                      hipblasOperation_t            transA,
                                                      hipblasDiagType_t             diag,
                                                      int                           n,
                                                      int                           k,
                                                      const hipDoubleComplex* const AP[],
                                                      int                           lda,
                                                      hipDoubleComplex* const       x[],
                                                      int                           incx,
                                                      int                           batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasStbmvBatched_64(hipblasHandle_t    handle,
                                                      hipblasFillMode_t  uplo,
                                                      hipblasOperation_t transA,
                                                      hipblasDiagType_t  diag,
                                                      int64_t            n,
                                                      int64_t            k,
                                                      const float* const AP[],
                                                      int64_t            lda,
                                                      float* const       x[],
                                                      int64_t            incx,
                                                      int64_t            batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDtbmvBatched_64(hipblasHandle_t     handle,
                                                      hipblasFillMode_t   uplo,
                                                      hipblasOperation_t  transA,
                                                      hipblasDiagType_t   diag,
                                                      int64_t             n,
                                                      int64_t             k,
                                                      const double* const AP[],
                                                      int64_t             lda,
                                                      double* const       x[],
                                                      int64_t             incx,
                                                      int64_t             batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCtbmvBatched_64(hipblasHandle_t             handle,
                                                      hipblasFillMode_t           uplo,
                                                      hipblasOperation_t          transA,
                                                      hipblasDiagType_t           diag,
                                                      int64_t                     n,
                                                      int64_t                     k,
                                                      const hipblasComplex* const AP[],
                                                      int64_t                     lda,
                                                      hipblasComplex* const       x[],
                                                      int64_t                     incx,
                                                      int64_t                     batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZtbmvBatched_64(hipblasHandle_t                   handle,
                                                      hipblasFillMode_t                 uplo,
                                                      hipblasOperation_t                transA,
                                                      hipblasDiagType_t                 diag,
                                                      int64_t                           n,
                                                      int64_t                           k,
                                                      const hipblasDoubleComplex* const AP[],
                                                      int64_t                           lda,
                                                      hipblasDoubleComplex* const       x[],
                                                      int64_t                           incx,
                                                      int64_t                           batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCtbmvBatched_v2_64(hipblasHandle_t         handle,
                                                         hipblasFillMode_t       uplo,
                                                         hipblasOperation_t      transA,
                                                         hipblasDiagType_t       diag,
                                                         int64_t                 n,
                                                         int64_t                 k,
                                                         const hipComplex* const AP[],
                                                         int64_t                 lda,
                                                         hipComplex* const       x[],
                                                         int64_t                 incx,
                                                         int64_t                 batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZtbmvBatched_v2_64(hipblasHandle_t               handle,
                                                         hipblasFillMode_t             uplo,
                                                         hipblasOperation_t            transA,
                                                         hipblasDiagType_t             diag,
                                                         int64_t                       n,
                                                         int64_t                       k,
                                                         const hipDoubleComplex* const AP[],
                                                         int64_t                       lda,
                                                         hipDoubleComplex* const       x[],
                                                         int64_t                       incx,
                                                         int64_t                       batchCount);

//! @}

/*! @{
    \brief BLAS Level 2 API

    \details
    tbmvStridedBatched performs one of the matrix-vector operations

        x_i := A_i*x_i      or
        x_i := A_i**T*x_i   or
        x_i := A_i**H*x_i,

    where (A_i, x_i) is the i-th instance of the batch.
    x_i is a vector and A_i is an n by n matrix, for i = 1, ..., batchCount.

    - Supported precisions in rocBLAS : s,d,c,z
    - Supported precisions in cuBLAS  : No support

    @param[in]
    handle    [hipblasHandle_t]
              handle to the hipblas library context queue.
    @param[in]
    uplo      [hipblasFillMode_t]
              HIPBLAS_FILL_MODE_UPPER: each A_i is an upper banded triangular matrix.
              HIPBLAS_FILL_MODE_LOWER: each A_i is a  lower banded triangular matrix.
    @param[in]
    transA     [hipblasOperation_t]
              indicates whether each matrix A_i is tranposed (conjugated) or not.
    @param[in]
    diag      [hipblasDiagType_t]
              HIPBLAS_DIAG_UNIT: The main diagonal of each A_i is assumed to consist of only
                                     1's and is not referenced.
              HIPBLAS_DIAG_NON_UNIT: No assumptions are made of each A_i's main diagonal.
    @param[in]
    n         [int]
              the number of rows and columns of the matrix represented by each A_i.
    @param[in]
    k         [int]
              if uplo == HIPBLAS_FILL_MODE_UPPER, k specifies the number of super-diagonals
              of each matrix A_i.
              if uplo == HIPBLAS_FILL_MODE_LOWER, k specifies the number of sub-diagonals
              of each matrix A_i.
              k must satisfy k > 0 && k < lda.
    @param[in]
    AP         device array to the first matrix A_i of the batch. Stores each banded triangular matrix A_i.
              if uplo == HIPBLAS_FILL_MODE_UPPER:
                The matrix represented is an upper banded triangular matrix
                with the main diagonal and k super-diagonals, everything
                else can be assumed to be 0.
                The matrix is compacted so that the main diagonal resides on the k'th
                row, the first super diagonal resides on the RHS of the k-1'th row, etc,
                with the k'th diagonal on the RHS of the 0'th row.
                   Ex: (HIPBLAS_FILL_MODE_UPPER; n = 5; k = 2)
                      1 6 9 0 0              0 0 9 8 7
                      0 2 7 8 0              0 6 7 8 9
                      0 0 3 8 7     ---->    1 2 3 4 5
                      0 0 0 4 9              0 0 0 0 0
                      0 0 0 0 5              0 0 0 0 0
              if uplo == HIPBLAS_FILL_MODE_LOWER:
                The matrix represnted is a lower banded triangular matrix
                with the main diagonal and k sub-diagonals, everything else can be
                assumed to be 0.
                The matrix is compacted so that the main diagonal resides on the 0'th row,
                working up to the k'th diagonal residing on the LHS of the k'th row.
                   Ex: (HIPBLAS_FILL_MODE_LOWER; n = 5; k = 2)
                      1 0 0 0 0              1 2 3 4 5
                      6 2 0 0 0              6 7 8 9 0
                      9 7 3 0 0     ---->    9 8 7 0 0
                      0 8 8 4 0              0 0 0 0 0
                      0 0 7 9 5              0 0 0 0 0
    @param[in]
    lda       [int]
              specifies the leading dimension of each A_i. lda must satisfy lda > k.
    @param[in]
    strideA  [hipblasStride]
              stride from the start of one A_i matrix to the next A_(i + 1).
    @param[inout]
    x         device array to the first vector x_i of the batch.
    @param[in]
    incx      [int]
              specifies the increment for the elements of each x_i.
    @param[in]
    stridex  [hipblasStride]
              stride from the start of one x_i matrix to the next x_(i + 1).
    @param[in]
    batchCount [int]
                number of instances in the batch.

    ********************************************************************/

HIPBLAS_EXPORT hipblasStatus_t hipblasStbmvStridedBatched(hipblasHandle_t    handle,
                                                          hipblasFillMode_t  uplo,
                                                          hipblasOperation_t transA,
                                                          hipblasDiagType_t  diag,
                                                          int                n,
                                                          int                k,
                                                          const float*       AP,
                                                          int                lda,
                                                          hipblasStride      strideA,
                                                          float*             x,
                                                          int                incx,
                                                          hipblasStride      stridex,
                                                          int                batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDtbmvStridedBatched(hipblasHandle_t    handle,
                                                          hipblasFillMode_t  uplo,
                                                          hipblasOperation_t transA,
                                                          hipblasDiagType_t  diag,
                                                          int                n,
                                                          int                k,
                                                          const double*      AP,
                                                          int                lda,
                                                          hipblasStride      strideA,
                                                          double*            x,
                                                          int                incx,
                                                          hipblasStride      stridex,
                                                          int                batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCtbmvStridedBatched(hipblasHandle_t       handle,
                                                          hipblasFillMode_t     uplo,
                                                          hipblasOperation_t    transA,
                                                          hipblasDiagType_t     diag,
                                                          int                   n,
                                                          int                   k,
                                                          const hipblasComplex* AP,
                                                          int                   lda,
                                                          hipblasStride         strideA,
                                                          hipblasComplex*       x,
                                                          int                   incx,
                                                          hipblasStride         stridex,
                                                          int                   batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZtbmvStridedBatched(hipblasHandle_t             handle,
                                                          hipblasFillMode_t           uplo,
                                                          hipblasOperation_t          transA,
                                                          hipblasDiagType_t           diag,
                                                          int                         n,
                                                          int                         k,
                                                          const hipblasDoubleComplex* AP,
                                                          int                         lda,
                                                          hipblasStride               strideA,
                                                          hipblasDoubleComplex*       x,
                                                          int                         incx,
                                                          hipblasStride               stridex,
                                                          int                         batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasCtbmvStridedBatched_v2(hipblasHandle_t    handle,
                                                             hipblasFillMode_t  uplo,
                                                             hipblasOperation_t transA,
                                                             hipblasDiagType_t  diag,
                                                             int                n,
                                                             int                k,
                                                             const hipComplex*  AP,
                                                             int                lda,
                                                             hipblasStride      strideA,
                                                             hipComplex*        x,
                                                             int                incx,
                                                             hipblasStride      stridex,
                                                             int                batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasZtbmvStridedBatched_v2(hipblasHandle_t         handle,
                                                             hipblasFillMode_t       uplo,
                                                             hipblasOperation_t      transA,
                                                             hipblasDiagType_t       diag,
                                                             int                     n,
                                                             int                     k,
                                                             const hipDoubleComplex* AP,
                                                             int                     lda,
                                                             hipblasStride           strideA,
                                                             hipDoubleComplex*       x,
                                                             int                     incx,
                                                             hipblasStride           stridex,
                                                             int                     batchCount);

// 64-bit interface
HIPBLAS_EXPORT hipblasStatus_t hipblasStbmvStridedBatched_64(hipblasHandle_t    handle,
                                                             hipblasFillMode_t  uplo,
                                                             hipblasOperation_t transA,
                                                             hipblasDiagType_t  diag,
                                                             int64_t            n,
                                                             int64_t            k,
                                                             const float*       AP,
                                                             int64_t            lda,
                                                             hipblasStride      strideA,
                                                             float*             x,
                                                             int64_t            incx,
                                                             hipblasStride      stridex,
                                                             int64_t            batchCount);

HIPBLAS_EXPORT hipblasStatus_t hipblasDtbmvStridedBatched_64(hipblasHandle_t    handle,
                                                             hipblasFillMode_t  uplo,
                                                             hipblasOperation_t transA,
                                                             hipblasDiagType_t  diag,
