/*****************************************************************************
  Module: mpegrec.cpp - Implements the mpegrec command-line tool.
  Copyright (C) 1999  Andrew L. Sandoval

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *****************************************************************************/
/* Module: mpegrec.cpp                                                       */
/* Developer: Andrew L. Sandoval                                             */
/* Copyright: Andrew L. Sandoval (C) 1999                                    */
/* Purpose: Implements the mpegrec application.                              */
/*           mpegrec -? shows options.                                       */
/*****************************************************************************/
#include "mpegrec.h"
#include "mpgformat.h"
#include "mpgrecorder.h"
#include "auddev.h"
#include <time.h>

using namespace std;

unsigned int ctrl_c_hit = 0;
time_t endTime;

BOOL WINAPI terminateEarly(  DWORD dwCtrlType )
{
	++ctrl_c_hit;
	cerr << "(Stopping on CTRL-C)" << endl;
	return TRUE;
}

const char * parmFetch(int argc, char *argv[], 
					   const string &lookingFor, const char *defaultv = "")
{
	for(int i=0; i < argc; i++)
	{
		const string value(argv[i]);
		if(value == lookingFor && (argc > i)) return argv[i+1];
	}
	return defaultv;
}

bool flagSet(int argc, char *argv[], const string &lookingFor, bool defaultv = FALSE)
{
	for(int i = 0; i < argc; i++)
	{
		const string value(argv[i]);
		if(value == lookingFor) return TRUE;
	}
	return defaultv;
}

void MyCallback(MPEGrecorder& mpgRec)
{
	if(time(NULL) > (endTime + 1) || ctrl_c_hit)
		mpgRec.Stop();
	else cerr << '.' << flush;
}

int main(int argc, char *argv[])
{
	cerr << argv[0] << " Copyright (C) 2000 Andrew L. Sandoval" << endl
		 << "This program comes with ABSOLUTELY NO WARRANTY." << endl
		 << "See the file COPYING (http://www.gnu.org/copyleft/gpl.txt) for details." << endl
		 << endl;

	//
	// Set the CTRL-C Handler
	//
	SetConsoleCtrlHandler(terminateEarly, TRUE);

	MMDeviceList inputDevices;

	if(flagSet(argc, argv, "-display-devices"))
	{
		cout << "Devices:" << endl << inputDevices << endl;
		return 0;
	}
	cerr << "Devices:" << endl << inputDevices << endl;

	unsigned long length = atol( parmFetch(argc, argv, "-l", "3700") );
	DWORD bitrate = atol( parmFetch(argc, argv, "-bitrate", "128000") );
	DWORD samplef = atol( parmFetch(argc, argv, "-samplehz", "44100") );
	const string filename( parmFetch(argc, argv, "-o") );

	WORD mpegLayer = ACM_MPEG_LAYER3;
	if(flagSet(argc, argv, "-1")) mpegLayer = ACM_MPEG_LAYER1;
	if(flagSet(argc, argv, "-2")) mpegLayer = ACM_MPEG_LAYER2;
	if(flagSet(argc, argv, "-3")) mpegLayer = ACM_MPEG_LAYER3;

	ostringstream devstring;
	devstring << WAVE_MAPPER << ends;

	UINT deviceId = atol( parmFetch(argc, argv, "-dev", devstring.str().c_str()) );

	if(flagSet(argc, argv, "-?"))
	{
		cout << "USAGE: " << argv[0] << " -l seconds [-o outputfilename]"
			 << " [-1|2|3] [-bitrate bitrate] [-samplehz frequency] [-display-devices] [-?]"
			 << endl
			 << "\t-l # of seconds to record. Default: 3700" << endl
			 << "\t-o Output filename         Default: C:\\temp\\date-time.mp3" << endl
			 << "\t-1 Use MPEG-1 Layer-1" << endl
			 << "\t-2 Use MPEG-1 Layer-2" << endl
			 << "\t-3 Use MPEG-1 Layer-3      Default" << endl
			 << "\t-bitrate Bitrate           Default: 128000 (~1M/minute)" << endl
			 << "\t-samplehz Frequency        Default: 44100 (CD Quality)" << endl
			 << "\t-dev Device number         Default: 1st MPEG recording device." << endl
			 << "\t-display-devices Displays Recording Devices and exits." 
			 << "\t-? This menu." << endl;
		return 0;
	}
	
	MPEGrecorder *recorder;

	if(filename.length())
		recorder = new MPEGrecorder(filename.c_str(), bitrate, samplef, deviceId, 
		                            mpegLayer, MyCallback );
	else
		recorder = new MPEGrecorder(bitrate, samplef, deviceId, mpegLayer, MyCallback);

	cerr << "Starting Record...." << flush;

	time_t startTime = time(NULL);
	endTime = startTime + length;
	
	recorder->Start();
	
	recorder->WaitForRecordingToEnd();

	Sleep(200);

	delete recorder;

	ostringstream ssmsg;
	ssmsg << "Finished recording " << time(NULL) - startTime << " seconds of MPEG audio." << ends;
	string doneMsg(ssmsg.str());

	cerr << doneMsg << endl;
	cerr << endl << "Done." << endl;
	return 0;
}
